/*
**  show the url in the database
**
**  Limitations and Comments:
**  It can read berkeley db, GNU gdbm or ndbm database. It will link
**  with only one of the database.
**
**  Development History:
**      who                 when           why
**      ma_muquit@fccc.edu  Jul-26-1999    first cut
**      muquit@muquit.com   Sep-03-2000    write the md5 of the url as the
**                                         value  
*/


#include <stdio.h>
#include "ldbm.h"
#include <mutils.h>
#include <sll.h>

#include <signal.h>

/* include the counter config file at ../../src */
/* we need the database base directory and lockfile name to lock the db */

#ifdef SYS_WIN32
#include <configNT.h>
#else
#include <config.h>
#endif /* SYS_WIN32 */


#define TRUE    1
#define FALSE   0

#define VERSION_S   "1.2"

/*
#define EDEBUG                   1
*/

#define EDIT_ITEM               0x01
#define EDIT_ITEM_VISUALLY      0x02
#define DELETE_ITEM             0x03
#define INSERT_TO_DB            0x04
#define LOAD_FROM_FILE          0x05
#define SEARCH_DB               0x06
#define ADD_ITEM                0x07
#define SET_VARS                0x08
#define DUMP_DB                 0x09


/* only for SYS_UNIX */
#define DEFAULT_EDITOR          "vi"
#define DEFAULT_PAGER           "more"

#ifdef __VMS
# define DIR_SEPARATOR "" /* Already have ] at end of _DIR macros */
/* VMS DIGIT_DIR ends in . so need to add subdirectory and ] - handled later
 * */
#define SYM_PREFIX "WWW_"
#else
#define DIR_SEPARATOR "/"
#define SYM_PREFIX ""
#endif


#ifndef True
#define True    1
#endif  /* True */

#ifndef False
#define False   0
#endif /* False */

typedef struct _dbitem
{
    char
        *key,
        *value;
}Dbitem;


static int
    s_overrite;

static int
    s_lock=True;

static int
    s_be_quiet=False;

static void setOverwrite(int i)
{
    s_overrite=i;
}

static int getOverwrite(void)
{
    return(s_overrite);
}

static int setLock(int i)
{
    s_lock=i;
}

static int getLock(void)
{
    return(s_lock);
}

/* static data */
char
    s_sztmpfile[BUFSIZ];

static void printMenu(void)
{
    (void)
        fprintf(stderr,"\n-------------------------------------------------------------------------\n");
    (void) fprintf(stderr,"editdb version %s\n",VERSION_S);
    (void) fprintf(stderr,"Part of WWW Homepage Access Counter 2.6+\n");
    (void) fprintf(stderr,"http://www.muquit.com/muquit/software/Count/Count.html\n");
        fprintf(stderr,"-------------------------------------------------------------------------\n");

    (void) fprintf(stderr,"\nCommands are:\n");
    (void) fprintf(stderr," add            add data\n");
    (void) fprintf(stderr," edit string    search database and edit\n");
    (void) fprintf(stderr," md5 string     show MD5 digest of a string\n");
    (void) fprintf(stderr," vedit string   search database and edit with your editor\n");
    (void) fprintf(stderr," delete string  search database and delete\n");
    (void) fprintf(stderr," dump [file]    dump the content of database\n");
    (void) fprintf(stderr," !command       execute a shell command\n");
    (void) fprintf(stderr," search string  search database and show\n");
    (void) fprintf(stderr," load file.txt  load to database from text file, same as -f switch\n");
    /*
    (void) fprintf(stderr," set [var=foo]        set a variable\n");
    */
#ifdef SYS_UNIX
    (void) fprintf(stderr," cls            clear screen\n");
#endif /* SYS_UNIX */
    (void) fprintf(stderr," version        show the version info\n");
    (void) fprintf(stderr," quit           quit\n\n");
}


/*
** returns 0 on success
**         -1 on failure 
*/
static int dotLock(char *db_path)
{
    char
        *dir,
        szbuf[BUFSIZ],
        errbuf[BUFSIZ];

    if (getLock() == False) /* don't lock */
    {
        if (s_be_quiet == False)
            (void) fprintf(stderr,"<*> Not locking database\n");
        return(0);
    }
    if (s_be_quiet == False)
        (void) fprintf(stderr,"<*> Locking database\n");

        /* 
        ** make the path of dot lock file. The lock file will be in the
        ** same directory of the db. If fails, It has to be same as that the
        ** one counter uses . -mm Sep-29-2000
        */
        dir=mutilsGetDirname(db_path);
        if (dir != NULL)
        {
            (void)strcpy(szbuf,dir);
            if (strcmp(dir,"./") == 0)
                (void) strcat(szbuf,DB_LOCKFILE);
            else
            {
                (void) strcat(szbuf,"/");
                (void) strcat(szbuf,DB_LOCKFILE);
            }
            (void) free((char *) dir);
        }
        else
        {
            (void) strcpy(szbuf,DATABASE_DIR);
            (void) strcat(szbuf,DIR_SEPARATOR);
            (void) strcat(szbuf,DB_LOCKFILE);
        }

        *errbuf='\0';
        (void) fprintf(stderr,"Lock file: %s\n",szbuf);
        mutilsDotLock(szbuf,errbuf);


    if (*errbuf != '\0')
    {
        (void) fprintf(stderr,"%s",errbuf);
        return (-1);
    }

    return(0);
}

static void dotUnlock()
{
    if (getLock() == False)
        return;
    if (s_be_quiet == False)
        (void) fprintf(stderr,"<*> Unocking database\n");
    mutilsDotUnlock(1); /* 1 means delete the lock file */
    if (s_be_quiet == False)
        (void) fprintf(stderr,"<->Lock released.\n");
}



/*
**  splitOn()
**  split strings separated by a character
**  Parameters:
**  char c      the separating character
**  char *ptr   passed strings
**  int  *n     number of tokenized strngs, returns
**
**  Return Values:
**  pointer to pointer to strings if succeeds
**  NULL if failes
**
**  Limitations and Comments:
**  returns pointer to a malloc'd space, the caller is responsible to free
**  the memory. the tokenized strings can be a '\0', caller is responsible
**  to test it before using.
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Jul-04-1998    first cut
*/
char **splitOn(char c,char *ptr,int *n)
{
    char
        *p,
        *r,
        **s=(char **) NULL;

    int
        size,
        i;

    if ((*ptr == '\0') || (ptr == NULL))
        return (NULL);

    /*
    ** how many
    */
    r=mutilsStrdup(ptr);
    if (r == NULL)
        return (NULL);

    (*n)=1;
    for (p=r; *p != '\0'; p++)
    {
        if (*p == c)
        {
            (*n)++;
        }
    }
    free(r);
    size=(*n)+1;
    s=(char **) malloc(size*sizeof(char *));
    if (s == (char **) NULL)
    {
        (void) fprintf(stderr,"malloc failed at splitOn()]\n");
        return ((char **) NULL);
    }

    for (i=0; i < (*n)+1; i++)
    {
        s[i]=ptr;
        while (*ptr && (*ptr != c))
        {
            ptr++;
        }
        if (*ptr)
        {
            *(ptr++) = '\0';
        }
        else
        {
            break;
        }
    }

    if (++i < (*n)+1)
    {
        s[i]='\0';
    }
    else
    {
        s[(*n)-1+1]='\0';
    }

    return (s);
}

/*               
** get item. if a string is a space return NULL
** the purpose of the routine is if a command is followed by a lots of
** spaces, we need to walk over them to reach the item, example,
** show                     muquit
** it will return a pointer to muquit
*/
char *getItem(char **pp,int n,int which_one)
{           
    char
        *s; 
            
    if (which_one < 0)
        which_one=0;
         
    if (which_one > (n-1))
    {
        return (NULL);
    }       
    s=pp[which_one];
    mutilsRmallws(s);
    return(s);
                    
    /*
    ** we will escape the first one, it's the command, so will start with i=1
    */
#if 0               
    for (i=1; i < n; i++)
    {               
        /*
        stripWhite (pp[i]);
        */
        stripLeadingSpace(pp[i]);
        stripTrailingSpace(pp[i]);
        s=pp[i];
        if (*s != '\0')
            return (s);
    }
#endif

    return ((char *) NULL);
}

/* note the item can't be NULL or it wil core dump*/
static int compFunc(void *i1,void *i2)
{
    int
        rc;
    Dbitem
        *item1=(Dbitem *) i1,
        *item2=(Dbitem *) i2;

    rc=mutilsStrcasecmp(item1->key,item2->key);
    return (rc);
}

static Dbitem *allocateDbitem(void)
{
    Dbitem
        *di;
    char
        *func="editdb.c.-allocateDbitem()";

    di=(Dbitem *) malloc(sizeof(Dbitem));
    di->key=NULL;
    di->value=NULL;

    return(di);
}

/* write to database */
static void writeItemsToDb(char *db_path,char *key,char *value)
{
    Datum
        key_data,
        return_data,
        data_data;

    LDBM
        edbp=NULL;

    
    if (key && value)
    {
        /* lock before opening */
        if (dotLock(db_path) != 0)
        {
            (void) fprintf(stderr,"failed to lock .. aborting\n");
            return;
        }
        edbp=ldbm_open(db_path,LDBM_WRCREAT,00644,0);
        if (edbp == NULL)
        {
            perror(db_path);
            goto ExitProcessing;
        }


        /* we made sure earlier that the key or value changed */
        key_data.dptr=key;
        key_data.dsize=strlen(key)+1;
        data_data.dptr=value;
        data_data.dsize=strlen(value)+1;
        if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
        {
            (void) fprintf(stderr,"could not store (%s)=>(%s)\n",key,value);
            goto ExitProcessing;
        }
        else
        {
            (void) fprintf(stderr,"%s=>%s\n",key,value);
            (void) fprintf(stderr,"* saved successfully!\n\n");
        }
    }

ExitProcessing:

    if (edbp)
    {
        ldbm_close(edbp);
        dotUnlock();
    }
}




/*
** item_num is not used 
*/
static int editItems(char *db_path,Sll *list,int item_num)
{
    Dbitem
        *db_item=NULL;

    char
        *line,
        szbuf[BUFSIZ],
        *new_key=NULL,
        *new_value=NULL,
        *key,
        *value;

    int
        dirty_flag=0;

    char
        *func="editdb.c-editItems() ";

    db_item=(Dbitem *) list->data;
    key=db_item->key;
    value=db_item->value;
    (void) fprintf(stderr,"\n");
    if (key && value)
    {
        (void) fprintf(stderr,"Edit Key (%s): ",key);
        line=fgets(szbuf,sizeof(szbuf)-1,stdin);
        mutilsChopNL(line);
        mutilsRmallws(line);

       
        if (*line != '\0' && line != NULL)
        {
#ifdef EDEBUG
        (void) fprintf(stderr,"key line=\"%s\"\n",line);
#endif
 
            if (strcmp(key,line) != 0)
            {
                dirty_flag=1;
                new_key=mutilsStrdup(line);
                if (new_key == NULL)
                {
                    (void) fprintf(stderr,"malloc failed at %s\n",func);
                    return (-1);
                }
            }
        }
        else
            new_key=mutilsStrdup(key);

        (void) fprintf(stderr,"Edit Value (%s): ",value);
        line=fgets(szbuf,sizeof(szbuf)-1,stdin);
        mutilsChopNL(line);
        mutilsRmallws(line);

        if (*line != '\0' && line != NULL)
        {
#ifdef EDEBUG
        (void) fprintf(stderr,"value line=\"%s\"\n",line);
#endif
            if (strcmp(value,line) != 0)
            {
                dirty_flag=1;
                new_value=mutilsStrdup(line);
                if (new_key == NULL)
                {
                    (void) fprintf(stderr,"malloc failed at %s\n",func);
                    return (-1);
                }
            }
        }
        else
            new_value=mutilsStrdup(value);
    }


    if (dirty_flag == 0)
    {
        (void) fprintf(stderr,"\n*** Nothing changed!\n\n");
    }
    else
        writeItemsToDb(db_path,new_key,new_value);


    if (new_key)
        (void) free((char *) new_key);

    if (new_value)
        (void) free((char *) new_value);

    return(0);
}




#ifdef SYS_UNIX
/* edit items with your editor */
/* returns 0 on success, -1 on failure */
/* if item_num >= 1, edit them item from the list */
/* if item_num == -1, edit the items in all the list */

static int vEditItems(Sll *list,int item_num)
{
    Dbitem
        *di,
        *db_item;
    
    char
        *cp,
        *editor=DEFAULT_EDITOR,
        *func="editdb.c-vEditItems()";

    FILE
        *fp=(FILE *) NULL;

    int
        pid,
        status,
        dirty_flag=0,
        rc;

    Sll
        *l;

    db_item=(Dbitem *) list->data;

    di=allocateDbitem();
    if (di == NULL)
    {
        (void) fprintf(stderr,"malloc failed in %s\n",func);
        return(-1);
    }

    /* write the item to a tmp file */
    mutilsTmpFilename(s_sztmpfile);
    if (*s_sztmpfile == '\0')
    {
        perror("mktemp");
        return (-1);
    }

    fp=fopen(s_sztmpfile,"w");
    if (fp == (FILE *) NULL)
    {
        perror(func);
        return(-1);
    }
    (void) fprintf(fp,"##--------------------------------------------------------------------------\n");
    (void) fprintf(fp,"## editdb %s\n",VERSION_S);
    (void) fprintf(fp,"## Edit WWW Homepage Access Counter database editor\n");
    (void) fprintf(fp,"## Syntax is:\n");
    (void) fprintf(fp,"## key=>value\n");
    (void) fprintf(fp,"## Lines beginning with a hash mark are comments.\n");
    (void) fprintf(fp,"##\n");
    (void) fflush(fp);

    if (item_num == -1)    /* entire list */
    {
        for (l=list; l; l=l->next)
        {
            db_item=(Dbitem *) l->data;
            (void) fprintf(fp,"%s=>%s\n",db_item->key,db_item->value);
        }
    }
    else
        (void) fprintf(fp,"%s=>%s\n",db_item->key,db_item->value);


    (void) fclose(fp);


    cp=getenv("EDITOR");
    if (cp == NULL)
        cp=getenv("VISUAL");

    if (cp != NULL)
        editor=cp;
    
     pid=fork();
     if (pid == 0)   /* child */
     {
         (void) signal(SIGINT,SIG_IGN);
         (void) fprintf(stderr,"\nStarting editor: %s\n\n",editor);
         sleep(1);
         (void) execlp(editor,editor,s_sztmpfile,NULL);

         /* won't be here */
         (void) fprintf(stderr,"Failed to exec editor: %s\n",editor);
         exit(-1);
     }
     else if (pid > 0) /* parent wait until child is done editing */
     {
         (void) wait(&status);
     }
     else
     {
         (void) fprintf(stderr,"Failed to fork()\n");
         return(-1);
     }

     return(0);
}
#endif /* SYS_UNIX */


/* write to database */
static void writeVitemsToDb(char *db_path,Sll *list,int n)
{
    FILE
        *fp=(FILE *) NULL;

    char
        *p,
        *cp,
        line[BUFSIZ],
        *vkey,
        *vvalue,
        *key,
        *value,
        *func="editdb.c-writeVitemsToDb()";
    
    Dbitem
        *di=(Dbitem *) list->data,
        *db_item=NULL;

    Sll
        *l,
        *node,
        *new;

    Datum
        key_data,
        return_data,
        data_data;

    LDBM
        edbp=NULL;

    int
        dirty_flag=0;

    /* open the tmpfile for reading */
    fp=fopen(s_sztmpfile,"r");
    if (fp == (FILE *) NULL)
    {
        (void) fprintf(stderr,"could not open %s for reading\n",s_sztmpfile);
        return;
    }
    db_item=allocateDbitem();
    if (db_item == NULL)
    {
        (void) fprintf(stderr,"malloc failed at:%s\n",func);
        fclose(fp);
        return;
    }

    for(;;)
    {
        (void) fgets(line,sizeof(line),fp);
        if (feof(fp))
            break;

        cp=line;
        mutilsChopNL(cp);
        mutilsStripLeadingSpace(cp);
        mutilsStripTrailingSpace(cp);
        if (*cp == '#' || *cp == '\0')
            continue;
        
        key=mutilsStrtok(cp,"=>");
        if (key == NULL)
        {
            (void) fprintf(stderr,"Error reading key\n");
            fclose(fp);
            return;
        }
        cp=NULL;
        value=mutilsStrtok(cp,"=>");
        if (value == NULL)
        {
            (void) fprintf(stderr,"Error reading value\n");
            fclose(fp);
            return;
        }

        if (n == -1)    /* entire list */
            dirty_flag=1;
        else /* a node */
        {
            /* check if changed */
            if (mutilsStrcasecmp(key,di->key) != 0)
                dirty_flag=1;

            if (mutilsStrcasecmp(value,di->value) != 0)
                dirty_flag=1;
        }
    if (dirty_flag == 1)
    {
        if (edbp == NULL)
        {
            /* lock before opening */
            if (dotLock(db_path) != 0)
            {
                (void) fprintf(stderr,"will not write.. aborting\n");
                return;
            }
            edbp=ldbm_open(db_path,LDBM_WRCREAT,00644,0);
        }
        if (edbp == NULL)
        {
            perror(db_path);
            goto ExitProcessing;
        }

        if (n == -1) /* entire list */
        {
            if (key && value)
            {
                key_data.dptr=key;
                key_data.dsize=strlen(key)+1;
                return_data=ldbm_fetch(edbp,key_data);
                if (return_data.dptr != NULL) /* a match */
                {
                    /* check if value changed */
                    if (strcmp(return_data.dptr,value) != 0)
                    {
                        (void) fprintf(stderr,"value change\n");
                        data_data.dptr=value;
                        data_data.dsize=strlen(value)+1;
                        if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
                        {
                            (void) fprintf(stderr,"could not store (%s)=>(%s)\n",key,value);
                            goto ExitProcessing;
                        }
                        else
                        {
                            (void) fprintf(stderr,"%s=>%s\n",key,value);
                            (void) fprintf(stderr,"* saved successfully!\n\n");
                        }

                    }
                }
                else
                {
                    /* new */
                    (void) fprintf(stderr,"new:%s=>%s\n",
                                   key,value);
                    data_data.dptr=value;
                    data_data.dsize=strlen(value)+1;
                    if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
                    {
                        (void) fprintf(stderr,"could not store (%s)=>(%s)\n",key,value);
                        goto ExitProcessing;
                    }
                    else
                    {
                        (void) fprintf(stderr,"%s=>%s\n",key,value);
                        (void) fprintf(stderr,"* saved successfully!\n\n");
                    }
                }
            }
        }
        else
        {
            key_data.dptr=key;
            key_data.dsize=strlen(key)+1;
            data_data.dptr=value;
            data_data.dsize=strlen(value)+1;
            if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
            {
                (void) fprintf(stderr,"Could not store (%s)=>(%s)\n",key,value);
                goto ExitProcessing;
            }
            else
            {
                (void) fprintf(stderr,"%s=>%s\n",key,value);
                (void) fprintf(stderr,"* Saved successfully!\n\n");
            }
        }
    }
    else
    {
        (void) fprintf(stderr,"- Nothing changed\n");
    }
    }
ExitProcessing:
    if (fp)
        fclose(fp);
    if (edbp)
    {
        ldbm_close(edbp);
        dotUnlock();
    }
}

/* delete an item from Db */
static void deleteFromDb(char *db_path,Sll *list,int n)
{
    Datum
        key_data;

    Dbitem
        *data_item=(Dbitem *) list->data;

    LDBM
        edbp=NULL;

    /* lock before opening */
    if (dotLock(db_path) != 0)
    {
        (void) fprintf(stderr,"will not write.. aborting\n");
        return;
    }
    edbp=ldbm_open(db_path,LDBM_WRCREAT,00644,0);
    if (edbp == NULL)
    {
        perror(db_path);
        goto ExitProcessing;
    }

    key_data.dptr=data_item->key;
    key_data.dsize=strlen(data_item->key)+1;
    if (ldbm_delete(edbp,key_data) != 0)
        (void) fprintf(stderr,"Could not delete\n");
    else
    {
        (void) fprintf(stderr,"%s=>%s\n",data_item->key,data_item->value);
        (void) fprintf(stderr," * Deletion succeeded!\n\n");
    }

ExitProcessing:
    if (edbp)
    {
        ldbm_close(edbp);
        dotUnlock();
    }

}


static Sll *readItemsFromFile(char *filepath)
{
    FILE
        *fp=NULL;

    Sll
        *head=NULL,
        *new=NULL,
        *list=NULL;

    Dbitem
        *di=NULL;

    char
        *line,
        *p,
        *cp,
        *key,
        *val,
        szkey[BUFSIZ],
        szval[BUFSIZ],
        szline[BUFSIZ];

    unsigned char
        *md5;

    fp=fopen(filepath,"r");
    if (fp == NULL)
    {
        (void) fprintf(stderr,"Could not open text file: %s\n",filepath);
        goto ExitProcessing;
    }

    di=allocateDbitem();
    if (di == NULL)
    {
        (void) fprintf(stderr,"memory allocation problem at readItemsFromFile()\n");
        goto ExitProcessing;
    }

    for (;;)
    {
        (void) fgets(szline,sizeof(szline),fp);
        if (feof(fp))
            break;
        mutilsChopNL(szline);
        cp=szline;
        if (*cp == '#' || *cp == '\0')
            continue;

        mutilsStripLeadingSpace(cp);
        mutilsStripTrailingSpace(cp);

        key=mutilsStrtok(cp,"=>");
        if (key == NULL)
        {
            (void) fprintf(stderr,"Key Null, ignoring\n");
            continue;
        }
        cp=NULL;
        val=mutilsStrtok(cp,"=>");
        if (val == NULL)
        {
            /*
            (void) fprintf(stderr,"Value for key %s is Null, ignoring\n",key);
            */

            md5=getMD5Digest(key);
            (void) fprintf(stderr,"Enter Value (%s): ",md5);
            line=fgets(szval,sizeof(szval)-1,stdin);
            mutilsChopNL(line);
            mutilsRmallws(line);
            if (*line != '\0' && line != NULL)
            {
                val=mutilsStrdup(line);
                if (val == NULL)
                {
                    (void) fprintf(stderr,"malloc failed for val\n");
                    continue;
                }
            }
            else
            {
                if (md5 != NULL)
                    val=mutilsStrdup(md5);
            }
        }
        if (val == NULL)
        {
            (void) fprintf(stderr,"Value for key %s is Null, ignoring\n",key);
            continue;
        }

        di=allocateDbitem();
        di->key=mutilsStrdup(key);
        di->value=mutilsStrdup(val);
        if (di->key == NULL || di->value == NULL)
        {
            (void) fprintf(stderr,"malloc problem at readItemsFromFile()\n");
            goto ExitProcessing;
        }
        new=allocateNode((void *) di);
        if (new == NULL)
        {
            (void) fprintf(stderr,"malloc problem at readItemsFromFile()\n");
            goto ExitProcessing;
        }
        appendNode(&head,&new);
    }

ExitProcessing:
    if (fp)
        (void) fclose(fp);

    return(head);
}

static void writeListsToDb(char *db_path,Sll *list)
{
    Sll
        *l;

    Dbitem
        *db_item=NULL;

    Datum
        key_data,
        return_data,
        data_data;

    LDBM
        edbp=NULL;

    char
        *key=NULL,
        *value=NULL;

    char
        *func="editdb.c-writeListsToDb()";

    int
        overwrite=getOverwrite();

#ifdef DEBUG
    (void) fprintf(stderr,"%s - overwrite=%d\n",func,overwrite);
#endif /* DEBUG */
    if (list == NULL)
    {
        (void) fprintf(stderr," Nothing to write to db\n");
        return;
    }

    /* lock before opening */
    if (dotLock(db_path) != 0)
    {
        (void) fprintf(stderr,"will not write.. aborting\n");
        return;
    }

    /* open the database to write */
    edbp=ldbm_open(db_path,LDBM_WRCREAT,00644,0);
    if (edbp == NULL)
    {
        perror(db_path);
        goto ExitProcessing;
    }
    
    for (l=list; l; l=l->next)
    {
        db_item=(Dbitem *) l->data;
        key=db_item->key;
        value=db_item->value;

        /* see if the item is in the database or not */
        if (key && value)
        {
            key_data.dptr=key;
            key_data.dsize=strlen(key)+1;
            return_data=ldbm_fetch(edbp,key_data);
            if (return_data.dptr != NULL)
            {
                if (overwrite == 0)
                {
                    (void) fprintf(stderr,"key:\"%s\" has a match in db, will not overwrite\n",key);
                    (void) fprintf(stderr,"Please use flag -o at startup or type set overwrite at Command prompt to overwrite\n");
                    continue;
                }
                else
                {
                    key_data.dptr=key;
                    key_data.dsize=strlen(key)+1;

                    data_data.dptr=value;
                    data_data.dsize=strlen(value)+1;
                    if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
                    {
                        (void) fprintf(stderr,"Failed to insert item:(%s)=>(%s)\n",key,value);
                    }
                    else
                        (void) fprintf(stderr,"Replace:(%s)=>(%s)\n",key,value);

                }
            }
            else
            {
                key_data.dptr=key;
                key_data.dsize=strlen(key)+1;

                data_data.dptr=value;
                data_data.dsize=strlen(value)+1;
                if (ldbm_store(edbp,key_data,data_data,LDBM_REPLACE) != 0)
                {
                        (void) fprintf(stderr,"Failed to insert item:(%s)=>(%s)\n",key,value);
                }
                else
                    (void) fprintf(stderr,"Add:%s=>%s\n",key,value);
            }
        }
        else
            continue;
    }

ExitProcessing:
    if (edbp)
    {
        ldbm_close(edbp);
        dotUnlock();
    }

}


/* prompts for new and add the item to database */
static void promptForNew(char *db_path)
{
    char
        *line,
        *new_key,
        *new_value,
        szkey[BUFSIZ],
        szvalue[BUFSIZ];

    char
        *func="editdb.c - promptForNew()";

    unsigned char
        *md5;

    new_key=NULL;
    new_value=NULL;

    (void) fprintf(stderr,"Enter Key: ");
    line=fgets(szkey,sizeof(szkey)-1,stdin);
    mutilsChopNL(line);
    mutilsRmallws(line);
    if (*line != '\0' && line != NULL)
    {
        new_key=mutilsStrdup(line);
        if (new_key == NULL)
        {
            (void) fprintf(stderr,"malloc failed at %s\n",func);
            return;
        }
    }

    md5=getMD5Digest(new_key);

    (void) fprintf(stderr,"Enter Value (%s): ",md5);
    line=fgets(szkey,sizeof(szkey)-1,stdin);
    mutilsChopNL(line);
    mutilsRmallws(line);
    if (*line != '\0' && line != NULL)
    {
        new_value=mutilsStrdup(line);
        if (new_value == NULL)
        {
            (void) fprintf(stderr,"malloc failed at %s\n",func);
            return;
        }
    }
    else
    {
        if (md5 != NULL)
            new_value=mutilsStrdup(md5);
    }

    if (new_key && new_value)
    {
        writeItemsToDb(db_path,new_key,new_value);
    }
    else
    {
        (void) fprintf(stderr,"Error: nothing to add\n");
    }
}



/*
** dbp should be a open LDBM as readonly
*/
void processItem(char *item,char *dbpath,int do_what)
{
    char
        *pager=DEFAULT_PAGER,
        szbuf[BUFSIZ];

    LDBM
        dbp=NULL;

    Datum
        key_data,
        data_data,
        return_data;
    Sll
        *l,
        *node,
        *head=NULL,
        *new=NULL;
    
    Dbitem
        *it,
        *dbitem;

    int
       rc,
       total=0,
       n=0;
       
    FILE
        *dump_fp=stdout,
        *fp=stdout;

    if (do_what == SET_VARS)
    {
        if (item == NULL || *item == '\0')
            return;

        if (mutilsStrcasecmp(item,"lock") == 0)
            setLock(True);
        else if (mutilsStrcasecmp(item,"nolock") == 0)
            setLock(False);
        else if (mutilsStrcasecmp(item,"overwrite") == 0)
            setOverwrite(True);
        else if (mutilsStrcasecmp(item,"nooverwrite") == 0)
            setOverwrite(False);

        return;
    }
#ifdef SYS_UNIX
    if (do_what == EDIT_ITEM          || 
        do_what == EDIT_ITEM_VISUALLY ||
        do_what == SEARCH_DB          ||
        do_what == DUMP_DB            ||
        do_what == DELETE_ITEM)
    {
        pager=getenv("PAGER");
        if (pager == NULL)
            pager=DEFAULT_PAGER;

        if (mutilsWhich(pager) == 0)    /* found */
        {
            fp=popen(pager,"w");
            if (fp == NULL)
                fp=stdout;
            else
                signal(SIGPIPE,SIG_IGN);
        }
        else
        {
            (void) fprintf(stderr,"pager: \"%s\" not found\n",pager);
        }
    }
#endif /* SYS_UNIX */

    if (fp == NULL)
        fp=stdout;



    if (do_what == EDIT_ITEM          || 
        do_what == EDIT_ITEM_VISUALLY ||
        do_what == SEARCH_DB          ||
        do_what == DUMP_DB            ||
        do_what == DELETE_ITEM)
    {
        if (dotLock(dbpath) != 0)
        {
            (void) fprintf(stderr,"failed to lock .. aborting\n");
            return;
        }

        /* open the database for reading */
        dbp=ldbm_open(dbpath,LDBM_READER,00644,0);
        if (dbp == NULL)
        {
            perror(dbpath);
            exit(1);
        }
        return_data=ldbm_firstkey(dbp);
        if (do_what == DUMP_DB) /* dump */
        {
            if (item != NULL)
            {
                dump_fp=fopen(item,"w");
                if (dump_fp == (FILE *) NULL)
                {
                    (void) fprintf(stderr,"Error: could not open file for writing: %s\n",item);
                    dump_fp=stdout;
                }
            }
            
            if (return_data.dptr)
            {
                if (dump_fp == stdout)
                    (void) fprintf(stderr," ---------------------\n");
                else
                    (void) fprintf(stderr,"Dumping to: %s. ",item);
            }
            while (return_data.dptr != NULL)
            {
                key_data=return_data;
                return_data=ldbm_fetch(dbp,key_data);
                (void) fprintf(dump_fp,"%s=>%s\n",
                               key_data.dptr,return_data.dptr);
                (void) fflush(dump_fp);

                if (return_data.dptr)
                    ldbm_datum_free(dbp,return_data);
                return_data=ldbm_nextkey(dbp,key_data);
            }
            if (dump_fp == stdout)
                (void) fprintf(stderr," ---------------------\n");
            else
                (void) fprintf(stderr," -done-\n");
            if (dump_fp != stdout)
                (void) fclose(dump_fp);

            goto ExitProcessing;
        }
        else
        {
            while (return_data.dptr != NULL && item)
            {
                key_data=return_data;
                return_data=ldbm_fetch(dbp,key_data);

                if (mutilsStristr(key_data.dptr,item))
                {
                    dbitem=(Dbitem *) malloc(sizeof(Dbitem));
                    if (dbitem == NULL)
                    {
                        (void) fprintf(stderr,"malloc failed at processItem\n");
                        exit(1);
                    }

                    dbitem->key=mutilsStrdup(key_data.dptr);
                    dbitem->value=mutilsStrdup(return_data.dptr);

                    new=allocateNode((void *) dbitem);
                    appendNodeSorted(&head,&new,compFunc);
                }
                if (return_data.dptr)
                    ldbm_datum_free(dbp,return_data);
                return_data=ldbm_nextkey(dbp,key_data);
            }
        }


        /* close the database */
        if (dbp)
        {
            ldbm_close(dbp);
            dbp=NULL;
            dotUnlock();
        }


        total=numNodes(&head);
        if (total > 1)
            (void) fprintf(stderr,"\n%d items matched.\n",total);
        else if (total == 1)
            (void) fprintf(stderr,"\n%d item matched.\n",total);
        else
        {
            (void) fprintf(stderr,"\nNo item matched.\n",total);
            goto ExitProcessing;
        }
    }

    if (do_what == EDIT_ITEM          || 
        do_what == EDIT_ITEM_VISUALLY ||
        do_what == SEARCH_DB          ||
        do_what == DUMP_DB            ||
        do_what == DELETE_ITEM)
    {
        for (l=head; l; l=l->next)
        {
            n++;
            it=(Dbitem *) l->data;
            if (it->key == NULL)
                continue;
            (void) fprintf(fp," %3d. %s=>%s\n",n,it->key,it->value);
            if (fp == stdout)
                (void) fflush(fp);
        }
    }

    if (do_what == EDIT_ITEM          || 
        do_what == EDIT_ITEM_VISUALLY ||
        do_what == SEARCH_DB          ||
        do_what == DELETE_ITEM)
    {
#ifdef SYS_UNIX
        if (fp != stdout)
        {
            pclose(fp);
            fp=stdout;
            signal(SIGPIPE,SIG_DFL);
        }
#endif /* SYS_UNIX */
        if (do_what == SEARCH_DB || do_what == DUMP_DB)
            goto ExitProcessing;
    }
        
    if (total >= 1)
    {
        if (do_what == EDIT_ITEM_VISUALLY)
            (void) fprintf(stderr,"Enter the number to edit, * to edit all, Q to quit: ");
        else if (do_what == EDIT_ITEM)
            (void) fprintf(stderr,"Enter the number to edit, Q to quit: ");
        else if (do_what == DELETE_ITEM)
            (void) fprintf(stderr,"Enter the number to delete, Q to quit: ");

        (void) fgets(szbuf,sizeof(szbuf)-1,stdin);
        mutilsChopNL(szbuf);
    }

    if (*szbuf != '\0')
    {
        if (*szbuf == 'q' || *szbuf == 'Q')
            goto ExitProcessing;
        if (*szbuf == '*')
        {
            if (do_what == EDIT_ITEM_VISUALLY)
            {

#ifdef SYS_UNIX
                rc=vEditItems(head,-1);
                if (rc == 0)
                    writeVitemsToDb(dbpath,head,-1);
                (void) unlink(s_sztmpfile);
                goto ExitProcessing;
#else
                (void) fprintf(stderr,"Sorry this option is not available in your oS\n");
                goto ExitProcessing;
#endif /* SYS_UNIX */
            }
            else if (do_what == EDIT_ITEM)
            {
                (void) fprintf(stderr,"wild card available with vedit opton!\n");
                goto ExitProcessing;
            }
        }

        if (do_what == EDIT_ITEM          || 
            do_what == EDIT_ITEM_VISUALLY ||
            do_what == DELETE_ITEM)
        {
            n=atoi(szbuf);
            if (n > 0 && n <= total)
            {
                node=getNthNode(head,n);
                if (node == NULL)
                {
                    (void) fprintf(stderr,"Failed to get node: %d\n",n);
                    goto ExitProcessing;
                }
            }
        }
    }

    switch(do_what)
    {
        case EDIT_ITEM_VISUALLY:
        {
            if (*szbuf == '\0')
                goto ExitProcessing;

#ifdef SYS_UNIX
            rc=vEditItems(node,n);
            if (rc == 0)
                writeVitemsToDb(dbpath,node,n);
            (void) unlink(s_sztmpfile);
#else
 (void) fprintf(stderr,"Sorry you can't use vedit option on your OS\n");
#endif /* SYS_UNIX */

            break;
        }

        case EDIT_ITEM:
        {
            if (*szbuf == '\0')
                 goto ExitProcessing;

            rc=editItems(dbpath,node,n);
            break;
        }


        case DELETE_ITEM:
        {
            if (*szbuf == '\0')
                goto ExitProcessing;
            deleteFromDb(dbpath,node,n);
            break;
        }

        case ADD_ITEM:
        {
            promptForNew(dbpath);
            break;
        }

       
        default:
        {
            (void) fprintf(stderr,"unknown option\n");
            break;
        }
    }/* switch */

ExitProcessing:
#ifdef SYS_UNIX
    if (fp != stdout)
        pclose(fp);
#endif /* SYS_UNIX */

    if(dbp)
    {
        ldbm_close(dbp);
        dotUnlock();
    }
}
static void showVersionInfo(void)
{
        (void) fprintf(stderr,"\n============================================================================\n");
        (void) fprintf(stderr,"editdb version %s\n",VERSION_S);
        (void) fprintf(stderr,"Edit WWW Homepage Counter database. ");
        (void) fprintf(stderr,"It supprts Berkeley DB 1.85 or 2.x with 1.85\n");
        (void) fprintf(stderr,"API support, GNU gdbm, ndbm and sdbm.\n");
        (void) fprintf(stderr,"\n*This version is compiled with ");
#ifdef LDBM_USE_DBBTREE
        (void) fprintf(stderr,"Berkeyley DBTREE support\n");
#endif /* LDBM_USE_DBBTREE */
#ifdef LDBM_USE_DBHASH
        (void) fprintf(stderr,"Berkeyley DBHASH support\n");
#endif /* LDBM_USE_DBBTREE */

#ifdef LDBM_USE_GDBM
        (void) fprintf(stderr,"GNU gdbm support\n");
#endif /* LDBM_USE_GDBM */

#ifdef LDBM_USE_SDBM
        (void) fprintf(stderr,"sdbm support\n");
#endif /* LDBM_USE_SDBM */

#ifdef LDBM_USE_NDBM
        (void) fprintf(stderr,"ndbm support\n");
#endif /* LDBM_USE_NDBM */
        (void) fprintf(stderr,"\n");
        (void) fprintf(stderr,"Author: Muhammad A Muquit\n");
        (void) fprintf(stderr,"http://www.muquit.com/muquit/software/Count/Count.html\n");
        (void) fprintf(stderr,"============================================================================\n");
}



/*
** process command */
int processCommand(char *command,char *dbpath)
{
    char
        szbuf[BUFSIZ],
        *item,
        **comm;

    int
        i,
        n;

    if (mutilsStrncasecmp(command,"quit",1) == 0)
    {
        return (1);
    }
    else if ((mutilsStrncasecmp(command,"?",1) == 0) ||
             (mutilsStrncasecmp(command,"help",4) == 0))
    {
        printMenu();
    }
    else if(mutilsStrncasecmp(command,"cls",3) == 0)
    {
#ifdef SYS_UNIX
        system("clear");
#endif /* SYS_UNIX */
    }
    else if (mutilsStrncasecmp(command,"dump",3) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        /* item is the file to dump to */
        processItem(item,dbpath,DUMP_DB);
    }
    else if (mutilsStrncasecmp(command,"!",1) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        if (n >= 1)
        {
            item=getItem(comm,n,0);
            if (item != NULL)
            {
                (void) strcpy(szbuf,item);
                for (i=1; i < n; i++)
                {
                    (void) sprintf(szbuf,"%s %s",szbuf,getItem(comm,n,i));
                }
#ifdef SYS_UNIX
                system(szbuf+1);
#endif /* SYS_UNIX */
            }
        }
    }
    else if (mutilsStrncasecmp(command,"vedit",3) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        processItem(item,dbpath,EDIT_ITEM_VISUALLY);
    }
    else if (mutilsStrncasecmp(command,"set",3) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        if (item == NULL)
        {
        (void) fprintf(stderr,"\n");
            if (getLock())
                (void) fprintf(stderr," lock\n");
            else
                (void) fprintf(stderr," nolock\n");

            if (getOverwrite())
                (void) fprintf(stderr," overwrite\n");
            else
                (void) fprintf(stderr," nooverwrite\n");
        (void) fprintf(stderr,"\n");

            return(0);
        }
        processItem(item,dbpath,SET_VARS);
    }
    else if (mutilsStrncasecmp(command,"ver",3) == 0)
    {
        showVersionInfo();
    }
    else if (mutilsStrncasecmp(command,"edit",2) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        processItem(item,dbpath,EDIT_ITEM);
    }
    else if (mutilsStrncasecmp(command,"md5",3) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        if (item != NULL)
            (void) fprintf(stdout,"%s=>%s\n",item,getMD5Digest(item));
    }
    else if (mutilsStrncasecmp(command,"search",2) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        processItem(item,dbpath,SEARCH_DB);
    }
    else if (mutilsStrncasecmp(command,"add",3) == 0)
    {
        processItem(NULL,dbpath,ADD_ITEM);
    }
    else if (mutilsStrncasecmp(command,"load",2) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        {
            Sll
                *list;

            char
                *text_file;

            text_file=item;
            list=readItemsFromFile(text_file);
            if (list)
                writeListsToDb(dbpath,list);
            else
                (void) fprintf(stderr,"Failed to read from file\n");
        }
    }
    else if (mutilsStrncasecmp(command,"delete",3) == 0)
    {
        comm=splitOn(' ',command,&n);
        item=getItem(comm,n,1);
        processItem(item,dbpath,DELETE_ITEM);
    }
    else if (mutilsStrncasecmp(command,"version",3) == 0)
    {
        showVersionInfo();
        return(0);
    }
    else
    {
        printMenu();
    }


    return (0);
}

static int Usage(char *progname,char *dbtype)
{
    (void) fprintf(stderr,"\n");
    (void) fprintf(stderr," Usage: %s [options] <%s file>\n",
                   progname,dbtype);

    (void) fprintf(stderr," Where the optins include:\n");
    (void) fprintf(stderr,"  -c              show the command set\n");
    (void) fprintf(stderr,"  -f textfile     read items from text file and insert to the database\n");
    (void) fprintf(stderr,"  -n              don't lock database (use for testing only)\n");
    (void) fprintf(stderr,"  -o              overwrite items while loading data\n");
    (void) fprintf(stderr,"  -v              show the version info\n");
    (void) fprintf(stderr,"  -q              be quiet\n\n"); 
    (void) fprintf(stderr,"The syntax of text file to load in database is:\n");
    (void) fprintf(stderr,"key=>value\n");

    (void) fprintf(stderr,"\nNote: If the database does not exist, it will be created\n");


    return(1);
}

static void printItems(Sll *list)
{
    Dbitem
        *di;
    Sll
        *l;

    for (l=list; l; l=l->next)
    {
        di=(Dbitem *) l->data;
        (void) fprintf(stderr,"(%s)=>(%s)\n",di->key,di->value);
    }

}

int main (int argc,char **argv) 
{
    int
        done=0,
        read_from_file,
        alt_lock_file,
        i;

    char
        *text_file=NULL,
        *option,
        *s,
        *line,
        szbuf[BUFSIZ],
        cmd_ch;

    Datum
        key_data,
        return_data;

    char
        *dbtype="<unknown>",
        *db_path=NULL;

    Sll
        *list=(Sll *) NULL;


    key_data.dptr=NULL;
    return_data.dptr=NULL;

    /* initialize */
    *s_sztmpfile='\0';
    read_from_file=0;
    alt_lock_file=0;
    text_file=NULL;

    setOverwrite(False);
    setLock(True);
    setbuf(stdout,NULL);
    setbuf(stderr,NULL);

#ifdef LDBM_USE_DBBTREE
        dbtype="Berkey DBTREE";
#endif /* LDBM_USE_DBBTREE */

#ifdef LDBM_USE_DBHASH
        dbtype="Berkey DBHASH";
#endif /* LDBM_USE_DBBTREE */

#ifdef LDBM_USE_GDBM
        dbtype="GNU gdbm";
#endif /* LDBM_USE_GDBM */

#ifdef LDBM_USE_SDBM
        dbtype="sdbm";
#endif /* LDBM_USE_SDBM */

#ifdef LDBM_USE_NDBM
        dbtype="ndbm";
#endif /* LDBM_USE_NDBM */


    if (argc < 2)
    {
        (void) fprintf(stderr,"editdb %s\n",VERSION_S);
        (void) fprintf(stderr,"Edit WWW Homepage access counter database\n");
        (void) fprintf(stderr,"<url:http://www.muquit.com/muquit/software/Count/Count.html\n\n");

        (void) fprintf(stderr,"(-compiled with %s support-)\n",dbtype);
        return(Usage(argv[0],dbtype));
    }

    
    for (i=1; i < argc; i++)
    {
        option=argv[i];
        if (*option == '-')
        {
            switch(*(option+1))
            {
                case 'f':
                case 'F':
                {
                    /* next arg must be the text file to read item from */

                    i++;
                    if (i == argc)
                    {
                        (void) fprintf(stderr,"Error: missing text file\n");
                        return(Usage(argv[0],dbtype));
                    }
                    read_from_file++;
                    text_file=mutilsStrdup(argv[i]);
                    break;
                }

                case 'n':
                case 'N':
                {
                    setLock(False);
                    break;
                }

                case 'o':
                case 'O':
                {
                    setOverwrite(True);
                    break;
                }

                case 'v':
                case 'V':
                {
                    showVersionInfo();
                    return(0);
                    break; /* won't be here */
                }

                case 'h':
                case 'H':
                {
                    Usage(argv[0],dbtype);
                    return(1);
                    break; /* won't be here */
                }

                case 'c':
                case 'C':
                {
                    (void) fprintf(stderr,"After opening the database, the following commands will be available:\n");
                    printMenu();
                    return(0);
                    break; /* won't be here */
                }
                case 'q':
                {
                    s_be_quiet=True;
                    break;
                }

                default:
                {
                    return (Usage(argv[0],dbtype));
                    break; /* won't be here */
                }
            } /* end switch */
        }
        else
            db_path=mutilsStrdup(argv[i]);
    }

    if (db_path == NULL)
    {
        /* make sure we still have exactly one arg left as dbname */
        (void) fprintf(stderr,"Error: no database specified\n\n");
        return (Usage(argv[0],dbtype));
    }

    if (read_from_file)
    {
        list=readItemsFromFile(text_file);
        writeListsToDb(db_path,list);
        return(0);
    }

#ifdef SYS_UNIX
    /* ignore interrupt signal */
    (void) signal(SIGINT,SIG_IGN);
#endif /* SYS_UNIX */

    while (!done)
    {
        (void) fprintf(stderr,"(Current database:%s)\n",
                       mutilsBasename(db_path));
        (void) fprintf(stderr,"Command> ");
        line=fgets(szbuf,sizeof(szbuf)-1,stdin);
        if (line == NULL)
        {
            printMenu();
            continue;
        }
        mutilsChopNL(line);
        mutilsStripLeadingSpace(line);

        if (*line == '\0')
            printMenu();
        else
            done=processCommand(line,db_path);
    }

    return(0);
}
