/* utility routines for the coutner */

#include "cdebug.h"

#include "combine.h"
#include "gdfonts.h"
#include "count.h"

#include "cdebug.h"

#ifdef USE_DATABASE
#include <ldbm.h>
#endif /* USE_DATABASE */



/*
 *  USE_DNS added by Sean Reifschneider <jafo@tummy.com> 1997-11-21.
 *
 *  When defined, USE_DNS will cause the referer host to be looked up
 *  in DNS for checking in the .cfg file.  This allows people running
 *  virtual hosts to list only their IP addresses instead of having to
 *  add each virtual domain name to the .cfg file.
 */

#ifdef USE_DNS
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#endif


#ifdef SYS_WIN32
#include "configNT.h"
#elif defined( __VMS )
#include "configVMS.h"
#else
#include "config.h"
#endif

static char *month_name[]=
{
    "Jan", "Feb", "Mar", "Apr",
    "May", "Jun", "Jul", "August",
    "Sep", "Octr","Nov", "Dec"
};


static unsigned long my_inet_addr _Declare ((char *));


/*
** print the Content-type MIME header
*/
void PrintHeader ()
{

    char
        expires[64],
        date[64];

    int
        exp;
    /* 
    ** check if caching should be disabled 
    */
    if (doHTTPcache() == 0)
    {
        exp=getExpireOffset();
        Debug2("Expire in: %d seconds",exp,0);
        getHTTPexpires(exp,expires,date,63,sizeof(expires)-1,sizeof(date)-1);
        Debug2("expires=%s",expires,0);
        Debug2("date=%s",date,0);
        if (Gdebug == False)
        {
/* Using DECthreads OSU server under VMS it allows more flexibility */
/* if we set expiry time in COUNT.COM, e.g. we can use DECnet record mode */
#ifndef __VMS
            (void) fprintf(stdout,"%s%c",expires,LF);
            (void) fprintf(stdout,"%s%c",date,LF);
#endif
        }
    }

    if (Gdebug == False)
    {
/* Using DECthreads OSU server under VMS it allows more flexibility */
/* if we send the content type ourselves, e.g. we can use DECnet record mode */
#ifndef __VMS
        (void) fprintf (stdout,"Content-type: image/gif%c%c",LF,LF);
        (void) fflush (stdout);
#endif
    }
    return;
}

/*
** check if the counter file exists
*/

int CheckFile (filename)
char
    *filename;
{
    int
        rc=0;

    rc = access (filename, F_OK);
    return rc;
}

/*
 * checkfilename:
 * - check to see if a path was specified - return 1 if so, 0 otherwise
 * it might not be foolproof, but I can't come up with
 * any other ways to specify paths.
 * by carsten@group.com (07/27/95)
 * ..\filename was not getting ignored on NT
 * reported by asolberg@pa.net>
 * fixed: 04/19/96
 */

int checkfilename(str)
char
    *str;
{
    if (strcmp(str,"..") == 0)
    {
        return 1;
    }

    while (*str)
    {
        if ((*str == '/') || 
            (*str == '\\') || 
            (*str == ':') ||
            (*str == '~'))
            return 1;
        str ++;
    }
    return 0;
}

/*
**  cMalloc()
**  a wrapper around malloc
**
**  RCS
**      $Revision$
**      $Date$
**  Return Values:
**      returns a pointer to allocated space
**
**  Parameters:
**      size    bytes to malloc
**
**  Side Effects:
**      space is allocated
**
**  Limitations and Comments:
**      will write the error message image and exit
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Oct-12-1997    first cut
*/

char *cMalloc(size)
int
    size;
{
    char
        *ptr;

    ptr=(char *) malloc(size*sizeof(char));

    if (ptr == NULL)
    {
        StringImage("Memory allocation problem!");
        exit(0);
    }

    return(ptr);
}



/*
**  authRemhost()
**  authenticare remote host
**
**  RCS
**      $Revision$
**      $Date$
**  Return Values:
**      if HTTP_REFERER env variable exists:
**        returns (1) only if host is authorized. othewise writes the error
**        message image and exits.
**      if HTTP_REFERER env variabel does not exist
**          return 0
**
**  Parameters:
**      none
**
**  Side Effects:
**      none
**
**  Limitations and Comments:
**      none
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Oct-18-1997    first cut
*/

int authRemhost()
{
    int
        rc=0;

    char
        *prem_refh=(char *) NULL,
        szbuf[BUFSIZ],
        szHttp_ref[1024],
        szRhost[1024];

#ifdef USE_DNS
    struct hostent
        *he;
    char
        *revname,
        *ipaddr;
#endif

    *szHttp_ref='\0';
    *szRhost='\0';

#ifdef __VMS
    prem_refh=(char *) getenv(SYM_PREFIX "HTTP_REFERER");
#else
    prem_refh=(char *) getenv("HTTP_REFERER");
#endif

    if (prem_refh != (char *) NULL)
    {
        mutilsSafeStrcpy(szHttp_ref,prem_refh,sizeof(szHttp_ref)-1);
        GetRemoteReferer(szHttp_ref,szRhost);

        if (*szRhost != '\0')
        {
#ifdef USE_DNS
            /*  get resolved host name and ip address as well  */
             revname = NULL;
             ipaddr = NULL;
             if ((he = gethostbyname(szRhost))) 
             {
                revname = he->h_name;
                ipaddr = inet_ntoa(*(struct in_addr *) he->h_addr);
             }
#endif

            rc=hostAuthorized(szRhost);
#ifdef USE_DNS
            if (!rc && ipaddr)
            {
                rc=hostAuthorized(ipaddr);
            }
#endif /* USE_DNS */
            if (rc == True)     /* Authorized */
                return (True);
            
            if (rc == False)
            {
                *szHttp_ref='\0';
                if (strlen(szRhost) + 27 > sizeof(szHttp_ref))
                {
                    (void) sprintf(szHttp_ref,
                                "Host: \"(name too long)\" is not authorized");
                }
                else
                {
                    (void) sprintf(szHttp_ref,"Host: \"%s\" is not authorized",
                                   szRhost);
                    Warning(szHttp_ref);
                    StringImage(szHttp_ref);

                    /*
                    ** log the offending page too
                    */
                    if (strlen(prem_refh) < sizeof(szbuf)-1)
                    {
                        (void) sprintf(szbuf,"Referer page: %s\n",
                                       prem_refh);
                        Warning(szbuf);
                    }
                }

                exit(0);
            }
        }
    }
    else
        rc=0;

    return (rc);
}


void Warning (message)
char
    *message;
{
    char
        *times;
    FILE
        *fp= (FILE *) NULL;
    char
        buf[1024];

    /*
    ** if configured not to log messages, just return
    */
    if (getcfgLogErrmsg() == False)
        return;

    *buf='\0';

#ifdef SYS_WIN32
    (void) sprintf(buf,"%s%s/%s",g_sZ_vbase,LOG_DIR,ERROR_LOG_FILE);
#else
    (void) sprintf(buf,"%s%s%s",LOG_DIR,DIR_SEPARATOR,ERROR_LOG_FILE);
#endif

    times=mutilsGetTime();


    fp = fopen(buf, "a");
    if (fp == (FILE *) NULL)
    {
        /*
        ** netscape web server merges stderr and stdout together, so if you'r
        ** using netscape server, you're busted, you not going to know why
        ** the program is not working. If you use apache, you will see error
        ** messages in the server error log
        ** --play safe, just return quietly (mm, jan-21-1999)
        */
/*
(void) fprintf (stderr,"[%s] Count %s: Could not open CountLog file %s\n",
                times,Version,buf);
*/
        return;
    }
    (void) fprintf (fp,"[%s] Count %s: %s\n", times, Version,message);

    if (fp != stderr)
        (void) fclose (fp);
}


/*
** Log visitor information
** almost all comes from env variables
** Log rotation code is taken from apache1.3.6 src/support/rotatelogs.c by
** Ben Laurie <ben@algroup.co.uk>
** -muquit, Aug-21-1999
*/

void doVisitorLog()
{
    FILE
        *fp=NULL;

    char
        *vis_time,
        *vis_host,
        *vis_ip,
        *vis_page,
        *vis_current_hits,
        *vis_browser,
        tmpbuf[BUFSIZ],
        szpath[MaxTextLength+1];

    time_t
        tls,
        trot;
    
    if (getcfgLogVisitorInfo() == True)
    {

        /* make the full path of the visitor log file */
        /* check if we need to rotate logs, if so log file name will be */
        /* Countx.xvisitor.log.time */
#ifdef SYS_WIN32
        (void) strcpy(szpath,g_sZ_vbase);
        (void) strcat(szpath,LOG_DIR);
        (void) strcat(szpath,DIR_SEPARATOR);
        (void) strcat(szpath,VISITOR_LOG_FILE);
#else
        (void) strcpy(szpath,LOG_DIR);
        (void) strcat(szpath,DIR_SEPARATOR);
        (void) strcat(szpath,VISITOR_LOG_FILE);
#endif /* SYS_WIN32 */
        
        trot=getcfgLogRotationInterval();
        if (trot > 0)
        {
            /* log starts */
            tls=(time(NULL)/trot)*trot;

#ifdef SYS_WIN32
            (void) sprintf(szpath,"%s%s/%s_%010d",
                           g_sZ_vbase,
                           LOG_DIR,
                           VISITOR_LOG_FILE,
                           (int) tls);


#else
            (void) sprintf(szpath,"%s%s%s_%010d",
                           LOG_DIR,
                           DIR_SEPARATOR,
                           VISITOR_LOG_FILE,
                           (int) tls);
#endif /* !SYS_WIN32 */
            /* log ends */
            /*
            tle=tls+trot;
            */
        }

        Debug2("visitor log file name=%s",szpath,0);
        /* open file for writing */
        fp=fopen(szpath,"a");
        if (fp == (FILE *) NULL)
        {
            (void) sprintf(tmpbuf,"Could not open visitor log file:\n%s",
                           szpath);
            multilineStringImage(tmpbuf,ALIGN_CENTER,0,0,0,0,255,0);
            exit(0);
        }
        vis_time=mutilsGetTime();
        if (vis_time == NULL)
            vis_time="<unknown>";

        vis_host=getenv("REMOTE_HOST");
        if (vis_host == NULL)
            vis_host="<unknown>";

        vis_ip=getenv("REMOTE_ADDR");
        if (vis_ip == NULL)
            vis_ip="<unknown>";

        vis_page=getenv("HTTP_REFERER");
        if (vis_page == NULL)
            vis_page="<unknown>";
        
        vis_browser=getenv("HTTP_USER_AGENT");
        if (vis_browser == NULL)
            vis_browser="<unknown>";

        vis_current_hits=getCurrentHits();
        if (vis_current_hits == NULL)
            vis_current_hits="<unknown>";

        /*
    (void) fprintf(fp, "Time: %s\nHost: %s\nIP: %s\nPage: %s\nCurrent Hits: %s\nBrowser:%s\n\n", vis_time,vis_host,vis_ip,vis_page,vis_current_hits,vis_browser);
    */
        (void) fprintf(fp,"%s|%s|%s|%s|%s|%s\n",
            vis_time,vis_host,vis_ip,vis_page,vis_current_hits,vis_browser);

        (void) fclose(fp);
    }
}


int CheckDirs()
{
    if ((strcmp(CONFIG_DIR,DATA_DIR) == 0) ||
        (strcmp(CONFIG_DIR,DATA_DIR) == 0) ||
        (strcmp(CONFIG_DIR,LOG_DIR) == 0) ||
        (strcmp(DIGIT_DIR,DATA_DIR) == 0) ||
        (strcmp(DIGIT_DIR,LOG_DIR) == 0) ||
        (strcmp(DATA_DIR,LOG_DIR) == 0))
    return (1);
return (0);
}



/*
 * the different -- and hopefully self-contained -- version of
 * CheckRemoteIP. Main difference is that ignore host block may
 * contain two IP address fields. First is then IP network address
 * and second is the netmask to be applied for comparision.  
 *
 * Davorin Bengez (Davorin.Bengez@science.hr), Jan 11. 1996.
 */

int CheckRemoteIP(remote_ip,ip_ign)
char *remote_ip, *ip_ign;
{
    unsigned long
        rem_ip,
        ignore_ip,
        ignore_mask;

    char
        addr_buf[20],
        mask_buf[20];

    int
        rc=False,
        i;

    ignore_mask = 0xffffffffL;

    rem_ip = my_inet_addr(remote_ip);
    if(rem_ip == (unsigned long)-1) return(False);

    /*
     * if ip_ign has TWO ip-look-alike fields, second field is netmask
     */

    if((i = sscanf(ip_ign, "%s %s", addr_buf, mask_buf)) < 1)
    {
        return(False);
    }
    ignore_ip = my_inet_addr(addr_buf);
    if (ignore_ip == (unsigned long)-1)
        return(False);

    /*
     * try to convert the mask to something usable and fail if it
     * is not the proper IP netmask...
     */

    if(i == 2)  /* with netmask */
    {
        Debug2("in netmask ==> ignore IP=%s",ip_ign,0);
        ignore_mask = my_inet_addr(mask_buf);
        if(ignore_mask == (unsigned long)-1)
            return(False);

        /*
         * ...and finally, compare the masked addresses...
         */
  
        if((rem_ip & ignore_mask) == (ignore_ip & ignore_mask))
            return(True);
        else
            return(False);
    }
    else if (i == 1)  /* a single IP or wild card */
    {
        Debug2("in sing/wildcard ==> ignore IP=%s",ip_ign,0);
        rc=mutilsIsinname(remote_ip,ip_ign);
        Debug2("ignore %s=%d",remote_ip,rc);
        return (rc);
    }

    return (False);
}

/*
 * and a version of inet_addr - not to link all the network services
 * just because of this one...
 */

static unsigned long my_inet_addr(s)
char *s;
{
    unsigned long n;
    int i;

    n = 0;

    for(i = 24; i >= 0; i -= 8) {
        n |= (unsigned long)atoi(s) << i;
        if((s = strchr(s,'.')) == (char *)NULL)
            break;
        ++s;
    }
    return(n);
}
/*
**  getdateGMT()
**  get the today's date in GMT
**
**  RCS
**      $Revision$
**      $Date$
**  Return Values:
**      pointer to a static string buffer of asctime. 
**
**  Parameters:
**      none
**
**  Side Effects:
**      pointer may get squashed with the next call to asctime. so
**      store it yourself if needed.
**
**  Limitations and Comments:
**      some wise guy decided to add a newline at the end of the string
**      returned by asctime. the new line is removed.
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Oct-25-1997    first cut
*/

char *getdateGMT()
{
    time_t
        t_time_t;

    struct tm
        *p_tm;

    char
        *tmp,
        *ptr;

    t_time_t=time(NULL);
    p_tm=gmtime(&t_time_t);
    ptr=asctime(p_tm);

    if (ptr && ((tmp=strchr(ptr,'\n')) != NULL))
        *tmp='\0';

    return(ptr);
}

/*
**  getHTTPexpires()
**  gets HTTP Expires and Date header
**
**  RCS
**      $Revision$
**      $Date$
**  Return Values:
**      none
**
**  Parameters:
**      offset      offset in seconds from current time 
**                  create the expire header after adding this seconds offset
**                  to current time.
**      expires     Expire header (in asctime format in GMT), returns
**      date        Date (asctiem format in GMT), returns
**      length      this many chacters to copy to expires or date buffer
**      exp_size    size of expires buffer
**      date_size   size of date buffer
**
**  Side Effects:
**      the contents of expires and date are changed. the caller is 
**      reponsible to call with corrent size.
**
**  Limitations and Comments:
**      expires and date must have enough space to hold the Expires and
**      Date strings. the caller is responsible to pass the corrent
**      size length.
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Oct-25-1997    first cut
*/

void getHTTPexpires(offset,expires,date,length,exp_size,date_size)
int
    offset;
char
    *expires,
    *date;
int
    length,
    exp_size,
    date_size;

{
    time_t
        t_time_t;

    struct tm
        *p_tm;

    char
        *tmp,
        *ptr;

    *expires='\0';
    *date='\0';

    if (offset < 0)
        offset=0;

    t_time_t=time(NULL);
    p_tm=gmtime(&t_time_t);
    ptr=mystrdup(asctime(p_tm));
    if (ptr && ((tmp=strchr(ptr,'\n')) != NULL))
        *tmp='\0';
    
    /*
    ** Date: header
    */
    mutilsSafeStrcpy(date,"Date: ",length);
    mutilsSafeStrcat(date,ptr,length,date_size,strlen(date));
    (void) free(ptr);

    t_time_t += offset;
    p_tm=gmtime(&t_time_t);
    ptr=mystrdup(asctime(p_tm));
    if (ptr && ((tmp=strchr(ptr,'\n')) != NULL))
        *tmp='\0';

    /*
    ** Expires: header
    */
    mutilsSafeStrcpy(expires,"Expires: ",length);
    mutilsSafeStrcat(expires,ptr,length,exp_size,strlen(expires));
    (void) free(ptr);
}

/*
** duplicate a string
*/
char *mystrdup (string)
char
    *string;
{
    char
        *tmp;

    if (string == (char *) NULL)
        return ((char *) NULL);

    /* malloc twice as much memory.  */
    tmp = (char *) malloc ((int) strlen(string)*2);

    if (tmp == (char *) NULL)
    {
        StringImage("memory allocation problem mystrdup()");
        exit(0);
        return ((char *) NULL);
    }
    /* it's safe to copy this way */
    (void) strcpy(tmp, string);
    return (tmp);
}



/*
** from wusage 2.3
*/
void RemoveTrailingSp (string)
char
    *string;
{
    while (True)
    {
        int
            l;

        l = (int) strlen (string);
        if (!l)
            return;

        if (string[l-1] == ' ')
            string[l-1] = '\0';
        else
            return;
    }
}


/*
**  allocCountdown()
**  allocate memory for the Countdown struct and initializes the members
**
**  Parameters:
**      none
**
**  Return Values:
**      Countdown   *       if succeeds
**      NULL                if fails
**
**  Limitations and Comments:
**      none
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   May-10-1998    first cut
*/

Countdown *allocCountdown()
{
    Countdown
        *cd;

    cd=(Countdown *) malloc(sizeof(Countdown));
    if (cd == (Countdown *) NULL)
        return ((Countdown *) NULL);

    cd->days=0;
    cd->hours=0;
    cd->minutes=0;
    cd->seconds=0;

    return (cd);
}

/*
**  calculate countdown time from a specific time and returns a pointer
**  to the struct Countdown
**
**  Parameters:
**      time_t  from_sec    From time in seconds
**
**  Return Values:
**      pointer to struct Countdown in success
**      NULL    in failure
**
**  Limitations and Comments:
**      none
**
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   May-10-1998    first cut for EJV
**                           Dec-13-1998    in Count 2.4
*/
Countdown *countDown(t_year,t_mon,t_day,t_hr,t_min,t_sec,t1)
int
    t_year,
    t_mon,
    t_day,
    t_hr,
    t_min,
    t_sec;

time_t
    t1;
{
    Countdown
        *cdown=(Countdown *) NULL;

    struct tm
        tm;

    time_t
        t2,
        tdiff;

    cdown=allocCountdown();
    if (cdown == NULL)
        return ((Countdown *) NULL);

    tm.tm_year=t_year-1900;        /* Year Since 1900 */
    tm.tm_mday=t_day;
    tm.tm_hour=t_hr;
    tm.tm_min=t_min;
    tm.tm_sec=t_sec;
    tm.tm_mon=t_mon-1;
    tm.tm_isdst=(-1);
    t2=mktime(&tm);

    tdiff=t2-t1;

    cdown->days=tdiff/(60*60*24);
    cdown->hours=(tdiff % (60*60*24))/3600;
    cdown->minutes=(tdiff % 3600)/60;
    cdown->seconds=(tdiff % 60);

    return (cdown);
}


/*
 *  checkTimezone - parses the GMT+-hhmm and returns the time in seconds
 *
 *  RCS:
 *      $Revision: 2.3 $
 *      $Date: 1996/05/03 02:20:22 $
 *
 *  Security:
 *      Unclassified
 *
 *  Description:
 *      This funcions parses a string like GMT+-hhmm and returns the time
 *      in seconds.
 *
 *  Input Parameters:
 *      type    identifier  description
 *
 *      char    str[]
 *          str holds a string like GMT+-hhmm
 *
 *  Output Parameters:
 *      type    identifier  description
 *
 *      text
 *
 *  Return Values:
 *      value   description
 *      int     diff
 *          hhmm is converted to seconds before returning. the value will
 *          be positive or negative based on GMT+ or GMT-
 *
 *  Side Effects:
 *      text
 *
 *  Limitations and Comments:
 *      text
 *
 *  Development History:
 *      who                 when        why
 *      muquit@semcor.com   26-Mar-96   first cut
 */


long checkTimezone(str)
char
    str[];
{
    int
        hr,
        min;
    int
        d;
    int
        counter=0;

    char
        buf[10];

    long
        diff=0L;

    int
        i;

    *buf='\0';
    hr=0;
    min=0;
    

    if (*str != '\0')
    {
        if (strlen(str) == 8)
        {
            if ((str[3] == '-') || (str[3] == '+'))
            {
                (void) strcpy(buf,&str[4]);   
                for (i=0; i < strlen(buf); i++)
                {
                    d=buf[i];
                    if ((d >= '0') && (d <= '9'))
                    {
                        if (counter < 2)
                            hr=hr*10 +(d-'0');
                        else
                            min=min*10+(d-'0');
                        counter++;
                    }
                }
                diff=(hr*3600)+min*60;
                if (str[3] == '-')
                    diff = -(diff);
            }
        }
    }
    return (diff);
}
/*
**  getLocalTime()
**  calculate time like: Mon-Day-Year hr:min:sec AM/PM GMT+xyzw
**  Parameters:
**  char *timezone_str      a string withe GMT offset, e.g. GMT+0500
**                          GMT+0000 GMT-1030 etc
**  int add_from            if 1, add From: before the time string
**
**  Return Values:
**  char *time  on success
**  NULL on failure
**
**  Limitations and Comments:
**  returns pointer to a malloc'd space, caller is responsible to free it.
**  If gmtime() is not available in the system, it will call **  localtime() 
**  instead of gmtime()
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Jan-24-1999    first cut
*/

char  *getLocalTime(timezone_str,add_from)
char
    *timezone_str;
int
    add_from;
{
    char
        *s=(char *) NULL;

    struct tm
        *ptm;

    time_t
        now;

    long
        diff;

    now=time(NULL);

    if (timezone_str == NULL || *timezone_str == '\0')
    {
        ptm=localtime(&now);
    }
    else
    {
        diff=checkTimezone(timezone_str);
        now += diff;
#ifdef HAVE_GMTIME
        ptm=gmtime(&now);
#else
        ptm=localtime(&now);
#endif /* HAVE_GMTIME */
    }
    s=cMalloc(512);
    if (add_from)
    {
        if (timezone_str == NULL || *timezone_str == '\0')
        {
            (void) sprintf(s,"%s %s %d %d, %d:%d:%d",
            "From:", month_name[ptm->tm_mon], ptm->tm_mday, 1900+ptm->tm_year,
            ptm->tm_hour, ptm->tm_min, ptm->tm_sec);
        }
        else
            (void) sprintf(s,"%s %s %d %d, %d:%d:%d %s",
            "From:", month_name[ptm->tm_mon], ptm->tm_mday, 1900+ptm->tm_year,
            ptm->tm_hour, ptm->tm_min, ptm->tm_sec,timezone_str);
    }
    else
    {
        if (timezone_str == NULL || *timezone_str == '\0')
        {
            (void) sprintf(s,"%s %d %d, %d:%d:%d",
            month_name[ptm->tm_mon], ptm->tm_mday, 1900+ptm->tm_year,
            ptm->tm_hour, ptm->tm_min, ptm->tm_sec);
        }
        else
        {
            (void) sprintf(s,"%s %d %d, %d:%d:%d %s",
            month_name[ptm->tm_mon], ptm->tm_mday, 1900+ptm->tm_year,
            ptm->tm_hour, ptm->tm_min, ptm->tm_sec,timezone_str);
        }
    }

    return (s);
}

void makeGIFPath(path,path_len,base,digit_info)
char
    *path;

int
    path_len;

char
    *base;

DigitInfo
    *digit_info;
{
            /* make the gif filepath*/
            mutilsSafeStrcpy(path,base,path_len-1);
#ifndef __VMS
            mutilsSafeStrcat(path,"/",path_len-1,path_len,strlen(path));
#endif

            mutilsSafeStrcat(path,digit_info->ddhead,path_len-1,
                    path_len,strlen(path));
#ifndef __VMS
            mutilsSafeStrcat(path,"/",path_len-1,path_len, strlen(path));
#else
/*          For VMS add a ] before adding name of .gif file  */
            mutilsSafeStrcat(path,"]",path_len-1,path_len,
                    strlen(path));
#endif
            mutilsSafeStrcat(path,digit_info->gif_file,path_len-1,
                    path_len,strlen(path));
}




char *mystrtok(s, delim)
char
    *s;
char
    *delim;
{
    register char
        *spanp;

    register int
        c,
        sc;

    char
        *tok;

    static char
        *last;


    if (s == (char *) NULL && (s = last) == (char *) NULL)
        return ((char *) NULL);

     /*
     ** Skip (span) leading delimiters (s += strspn(s, delim), sort of).
     */
cont:
    c = *s++;
    for (spanp = (char *)delim; (sc = *spanp++) != 0;)
    {
        if (c == sc)
            goto cont;
    }

    if (c == 0)
    {       /* no non-delimiter characters */
        last = (char *) NULL;
        return ((char *)NULL);
    }
    tok = s - 1;

    /*
     * Scan token (scan for delimiters: s += strcspn(s, delim), sort of).
     * Note that delim must have one NUL; we stop if we see that, too.
     */

    for (;;)
    {
        c = *s++;
        spanp = (char *)delim;
        do
        {
            if ((sc = *spanp++) == c)
            {
                if (c == 0)
                    s = (char *) NULL;
                else
                    s[-1] = '\0';
                last = s;
                return (tok);
            }
        } while (sc != 0);
    }
      /* NOTREACHED */
}

#ifdef USE_DATABASE

/*
**  grepForStringInDb()
**  check if a string is in the database
**
**  Parameters:
**  char    *db_path        full path of the database
**  char    *str            the string
**  cahr    *errbuf         error message if it occurs
**
**  Return Values:
**  True    if the string found
**  False   if not
**
**  Limitations and Comments:
**  the caller should check if any error message is in errbuf first.
**  errbuf should have enough space to hold the message.
**  search is case in-sensitive. 
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Aug-13-1999    first cut
*/


int grepForStringInDb(db_path,str,errbuf)
char
    *db_path;
char
    *str;
char
    *errbuf;
{
    LDBM
        dbp=NULL;

    Datum
        key_data,
        return_data;

    int
        rc=False;

    if (*str == '\0' || str == NULL)
        return (False);

    /* open the database */
    dbp=ldbm_open(db_path,LDBM_READER,00644,0);
    if (dbp == NULL)
    {
        (void) sprintf(errbuf,"Could not open database:\n%s",db_path);
        return(False);
    }

    while (return_data.dptr != NULL)
    {
        key_data=return_data;
        return_data=ldbm_fetch(dbp,key_data);
        if (mutilsStristr(key_data.dptr,str)) /* found */
        {
            if (return_data.dptr)
                ldbm_datum_free(dbp,return_data);
            rc=True;
            goto ExitProcessing;
        }
        if (return_data.dptr)
            ldbm_datum_free(dbp,return_data);
        return_data=ldbm_nextkey(dbp,key_data);
    }

ExitProcessing:

    if (dbp)
        ldbm_close(dbp);

    return (rc);
}

/*
**  grepForKeyInDb()
**  check if a key is in the database
**
**  Parameters:
**  char    *db_path        full path of the database
**  char    *str            the string
**  cahr    *errbuf         error message if it occurs
**
**  Return Values:
**      char *val       if the key is found
**      NULL            if the key is not found or if value is NULL
**
**  Limitations and Comments:
**  the caller should check if any error message is in errbuf first.
**  errbuf should have enough space to hold the message. search is 
**  case in-sensitive. 
**  Returns pointer to a malloc'd space, the caller is responsible to
**  free it.
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Aug-13-1999    first cut
**`                                         returns value if key is found
*/


char *grepForKeyInDb(db_path,key,errbuf)
char
    *db_path;
char
    *key;
char
    *errbuf;
{
    LDBM
        dbp=NULL;

    Datum
        key_data,
        return_data;

    char
        *val=NULL;

    if (*key == '\0' || key == NULL)
        return (NULL);

    /* open the database */
    dbp=ldbm_open(db_path,LDBM_READER,00644,0);
    if (dbp == NULL)
    {
        (void) sprintf(errbuf,"Could not open database:\n%s",db_path);
        return(NULL);
    }

    key_data.dptr=key;
    key_data.dsize=strlen(key)+1;
    return_data=ldbm_fetch(dbp,key_data);
    if (return_data.dptr != NULL)
    {
        if (return_data.dptr)
            val=mystrdup(return_data.dptr);
        ldbm_datum_free(dbp,return_data);
    }

    if (dbp)
        ldbm_close(dbp);

    return (val);
}

#endif /* USE_DATABASE */
