/*
**  routines to handle Count 2.5 configuration file
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Dec-27-1998    first cut
*/

#include <combine.h>
#include "count.h"
#include "cdebug.h"

#ifdef SYS_WIN32
#include "configNT.h"
#elif defined( __VMS )
#include "configVMS.h"
#else
#include "config.h"
#endif


#ifdef __VMS
#define DIR_SEPARATOR "" /* Already have ] at end of _DIR macros */
#else
#define DIR_SEPARATOR "/"
#endif


/*
** global in this file. member of this struct is accessible via some function
** calls. a way to avoid globals.
*/
enum
{
    BUFR_INC=1024
};
static Cfg
    *s_cfg=(Cfg *) NULL;
static char
    *s_cursec=(char *) NULL;
static char
    *s_bufr=(char *) NULL;
static int
    s_bsize=0;


    

/* private protos */
/*
static void printConfigError _Declare ((int err_type));
*/

static int  handlecfgParam   _Declare ((char *param,char *value));
static Cfg  *initCfg         _Declare ((void));
static void setCfgBoolean    _Declare ((char *value,int *set));
static void setCfgInteger    _Declare ((char *value,int *set));
static int  cfgContinuation  _Declare((char *line,int pos));
static char *locRealloc      _Declare((char *p,int size));
static int  cfgSection       _Declare((FILE *fp));
static int  cfgParameter     _Declare((FILE *fp,int c));


static int cfgContinuation(line,pos)
char
    *line;
int
    pos;
{
    pos--;
    while ((pos >= 0) && isspace(line[pos]))
        pos--;

    return (((pos >= 0) && (line[pos] == '\\')) ? pos : -1);
}

/*
** expand a pointer to be a particular size
*/
static char *locRealloc(p,size)
char
    *p;
int
    size;
{
    char
        *ret=NULL;

    if (size == 0)
    {
        if (p)
        {
            (void) free(p);
            (void) fprintf(stderr,"locRealloc() asked for 0 bytes\n");
            return (NULL);
        }
    }

    if (!p)
        ret=(char *) malloc(size);
    else
        ret=(char *) realloc(p,size);

    if (!ret)
        (void) fprintf(stderr,"malloc problem, failed to expand to %d bytes\n",
                       size);
    return (ret);
}

/*
**  cfgParameter()
**  scan a parameter name (or name and value pair)
**
**  Parameters:
**      fp      - open FILE pointer
**      c       - the first character of the parameter name, which would
**                have been read by Parse(). unlike comment line or a section
**                header, there's no lead-in character can be discarded.
**
**      style   - the style of the config file. it can be MS_STYLE, that is
**                parameter must follows by = and the value. If it is
**                NOT_MS_STYLE, then parameter does not follows by = or the
**                value.
**                  Example of MS_STYLE config file:
**                      [section]
**                          version = 2.4
**                      [foo]
**                          bar=hello
**                          foobar=world
**
**                  Example of NOT_MS_STYLE config file:
**                      [section]
**                           2.4
**                      [foo]
**                          hello
**                          world
**
**  Return Values:
**      0       on success
**      -1      on failure
**
**  Limitations and Comments:
**
**
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Apr-08-1998    first cut
*/

static int cfgParameter (fp,c)
FILE
    *fp;
int
    c;
{
    int
        i=0;      /* position withing s_bufr */
    int
        end=0;    /* s_bufr[end] is current end-of-string */
    int
        vstart=0; /* starting position of the parameter */

    char
        *func="cfg.c:cfgParameter() -";

    while ((c != EOF) && (c > 0))
    {
        if (i > (s_bsize-2))
        {
            s_bsize += BUFR_INC;
            s_bufr=locRealloc(s_bufr,s_bsize);
            if (s_bufr == NULL)
            {
                (void) fprintf(stderr,"%s malloc failed\n",func);
                return(-1);
            }
        }

        switch(c)
        {
            case '\r':
            {
                c=getc(fp);
                break;
            }
            case '\n':
            {
                i=cfgContinuation(s_bufr,i);
                if (i < 0)
                    c=0;
                else
                {
                    for(end=i; (end >= 0) && isspace(s_bufr[end]); end--)
                        ;
                    c=getc(fp);
                }
                break;
            }

            default:
            {
                s_bufr[i++]=c;
                if (!isspace(c))
                    end=i;
                c=getc(fp);
                break;
            }

        }
    }
    s_bufr[end]='\0';
    return (handlecfgParam(s_bufr,&s_bufr[vstart]));
}



/*
**  cfgSection()
**  scan a section name and pass the name to the function sfunc()
**
**  Parameters:
**      fp      - open FILE pointer
**
**  Return Values:
**      0   if the section name was read and 0 was returned from <sfunc>
**      -1  if a lexical error was encountered.
**
**  Limitations and Comments:
**      from samba source code
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   Apr-10-1998    first cut
*/

static int cfgSection(fp)
FILE
    *fp;
{
    int
        c,
        i,
        end;

    char
        *func="cfg.c:cfgSection() -";

    i=0;
    end=0;

    c=mutilsEatWhitespace(fp);    /* 
                            ** we've already got the '['. scan past initial
                            ** white space
                            */

    while ((c != EOF) && (c > 0))
    {
        if (i > (s_bsize-2))
        {
            s_bsize += BUFR_INC;
            s_bufr=locRealloc(s_bufr,s_bsize);
            if (s_bufr == NULL)
            {
                (void) fprintf(stderr,"%s malloc failed\n",func);
                return(-1);
            }
        }
        switch (c)
        {
            case ']':           /* found the closing bracked */
            {
                s_bufr[end]='\0';
                if (end == 0)
                {
                    (void) fprintf(stderr,"%s empty section name\n",func);
                    return (-1);
                }

                if (s_cursec != NULL)
                    (void) free((char *) s_cursec);
                s_cursec=mystrdup(s_bufr);
                (void) mutilsEatComment(fp);
                return (0);
                break;
            }

            case '\n':
            {
                i=cfgContinuation(s_bufr,i);
                if (i < 0)
                {
                    s_bufr[end]='\0';
                    (void) fprintf(stderr,"%s badly formed line in cfg file\n",
                                   func);
                    return (-1);
                }
                end=((i > 0) && (s_bufr[i-1] == ' ')) ? (i-1): (i);
                c=getc(fp);
                break;
            }
            default:
            {
                if (isspace(c))
                {
                    s_bufr[end]=' ';
                    i=end + 1;
                    c=mutilsEatWhitespace(fp);
                }
                else
                {
                    s_bufr[i++]=c;
                    end=i;
                    c=getc(fp);
                }
                break;
            }
        }
    }

    return (0);
}

/* returns 0 on success, -1 on failure */
static int handlecfgParam(param,value)
char
    *param;
char
    *value;
{
    char
        *p,
        *v;

    if (s_cursec == NULL || s_cfg == NULL || param == NULL)
        return (-1);

    if (mutilsStrcasecmp("version",s_cursec) == 0)
    {
        s_cfg->version=mystrdup(param);
        if (s_cfg->version == (char *) NULL)
            return (-1);
    }
    else if (mutilsStrcasecmp("options",s_cursec) == 0)
    {
        p=param;
        v=mutilsStrtok(p,"=");
        if (v != NULL)
        {
            if (mutilsStrcasecmp(v,"auto_file_creation") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->auto_file_creation);
            }
            else if (mutilsStrcasecmp(v,"strict_mode") == 0)
            {
                /* left out, added Feb-01-1999 */
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->strict_mode);
            }
            else if (mutilsStrcasecmp(v,"allow_rgb_database") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->allow_rgb_database);
            }
            else if (mutilsStrcasecmp(v,"count_reload") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->count_reload);
            }
            else if (mutilsStrcasecmp(v,"log_error_messages") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->log_error_messages);
            }
            else if (mutilsStrcasecmp(v,"log_visitor_info") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->log_visitor_info);
            }
            else if (mutilsStrcasecmp(v,"log_rotation_interval") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgInteger(v,&s_cfg->log_rotation_interval);
            }
            else if (mutilsStrcasecmp(v,"show_error_messages_to_browsers") == 0)
            {
                p=(char *) NULL;
                v=mutilsStrtok(p,"=");
                if (v != (char *) NULL)
                    setCfgBoolean(v,&s_cfg->show_error_messages_to_browsers);
            }


        }
    }
    else if (mutilsStrcasecmp("ignore ips",s_cursec) == 0)
        additemToList(&s_cfg->ignore_ips,param);
    else if (mutilsStrcasecmp("authorized",s_cursec) == 0)
        additemToList(&s_cfg->auth_hosts,param);

    return(0);
}

/*
static void printCfg         _Declare ((void));
*/


void parseConfigFile ()
{
    FILE
        *fp=(FILE *) NULL;

    int
        c;

    char
        tmpbuf[BUFSIZ],
        szcfgfile[MaxLineLength+1];

    *szcfgfile='\0';

#ifdef SYS_WIN32

    /* g_sZ_vbase has a trailing \ at the end */
    (void) strcpy(szcfgfile,g_sZ_vbase);
    (void) strcat(szcfgfile,CONFIG_DIR);
    (void) strcat(szcfgfile,"/");
    (void) strcat(szcfgfile,CONFIG_FILE);

#else

    (void) strcpy(szcfgfile,CONFIG_DIR);
    (void) strcat(szcfgfile,DIR_SEPARATOR);
    (void) strcat(szcfgfile,CONFIG_FILE);
#endif /* SYS_WIN32 */


    /* initialize Cfg struct */
    s_cfg=initCfg();
    if (s_cfg == (Cfg *) NULL)
    {
        StringImage("malloc failed at cfg.c:ParseConfig()");
        exit(0);
    }

    /* try to open the cfg file */
    fp=fopen(szcfgfile,"r");
    if (fp == (FILE *) NULL)
    {
       /* 
       ** we should not have any overflow, if the admin is stupid enough
       ** to have a path longer than BUFSIZ, he deserves it. I don't
       ** care
       */
       (void) sprintf(tmpbuf,"Unable to open config file for reading: %s\nIt is a permission problem! Make sure all the counter directories are accessible\nand the file count.cfg is readable by your web server.",szcfgfile);

           multilineStringImage(tmpbuf,ALIGN_CENTER,0,0,0,0,255,0);
           exit(0);

    }
    c=mutilsEatWhitespace(fp);
    while ((c != EOF) && (c > 0))
    {
        switch(c)
        {
            case '\n':                  /* blank line */
            {
                c=mutilsEatWhitespace(fp);
                break;
            }

            case ';':                   /* comment line */
            case '#':
            {
                c=mutilsEatComment(fp);
                break;
            }

            case '[':                   /* section header */
            {
                if (cfgSection(fp) < 0)
                {
                    (void) sprintf(tmpbuf,"Error reading section in cfg file:\n%s",szcfgfile);
                    multilineStringImage(tmpbuf,ALIGN_CENTER,0,0,0,0,255,0);
                    exit(0);
                }

                c=mutilsEatWhitespace(fp);
                break;
            }

            case '\\':                  /* bogus backslash */
            {
                c=mutilsEatWhitespace(fp);
                break;
            }

            default:                    /* parameter line */
            {
                if (cfgParameter(fp,c) < 0)
                {

                    (void) sprintf(tmpbuf,"Error reading parameter in cfg file:\n%s",szcfgfile);
                    multilineStringImage(tmpbuf,ALIGN_CENTER,0,0,0,0,255,0);
                }
                c=mutilsEatWhitespace(fp);
            }
        }

    }

   /*
   printCfg();
   */

    if (fp != (FILE *) NULL)
        (void) fclose(fp);
}


/*
** initialize the Cfg structure
*/
static Cfg *initCfg()
{
    Cfg
        *cfg;

    char
        *func="cfg.c:initCfg() -";

    cfg=(Cfg *) malloc(sizeof(Cfg));
    if (cfg == (Cfg *) NULL)
    {
        (void) fprintf(stderr,"malloc failed at: %s\n",func);
        return ((Cfg *) NULL);
    }

    cfg->version=(char *) NULL;
    cfg->auto_file_creation=False;
    cfg->strict_mode=True;
    cfg->allow_rgb_database=False;
    cfg->count_reload=False;
    cfg->log_error_messages=True;
    cfg->log_visitor_info=False;
    cfg->log_rotation_interval=0;
    cfg->show_error_messages_to_browsers=True;
    cfg->ignore_ips=NULL;
    cfg->auth_hosts=NULL;

    return (cfg);
}


/*
** ignoreIP()
**    returns True if ignore, False otherwise
*/
int ignoreIP(remote_ip)
char
    *remote_ip;
{
    Sllist
        *ig_ip,
        *llp;

    int
        rc=False;

    if (s_cfg == (Cfg *) NULL)
    {
        /*
        ** should write some message
        */
        return (rc);
    }
    ig_ip=getcfgIgips();
    for (llp=ig_ip;llp;llp=llp->next)
    {
        rc=CheckRemoteIP(remote_ip,llp->item);
        if (rc == True)
            break;
    }

    return (rc);
}

/*
** hostAuthorized()
**    returns True if Authorized, False otherwise
*/
int hostAuthorized(host)
char
    *host;
{
    Sllist
        *auth_hosts,
        *llp;

    int
        rc=False;

    if (s_cfg == (Cfg *) NULL)
    {
        /*
        ** should write some message
        */
        return (rc);
    }
    auth_hosts=getcfgAhs();
    for (llp=auth_hosts;llp;llp=llp->next)
    {
        rc=mutilsIsinname(host,llp->item);
        if (rc == True)
            break;
    }

    return (rc);
}

/*
** print the Cfg struct
*/

#if 0
static void printCfg()
{
    Sllist
        *ll,
        *ip;

    (void) fprintf(stderr,"[version]\n");
    (void) fprintf(stderr," %s\n",getcfgVersion());
    (void) fprintf(stderr,"[options]\n");
    (void) fprintf(stderr," auto_file_creation=%d\n",getcfgAutoFileCreation());
    (void) fprintf(stderr," strict_mode=%d\n",getcfgStrictMode());
    (void) fprintf(stderr," allow_rgb_database=%d\n",getcfgArgbDatabase());
    (void) fprintf(stderr," count_reload=%d\n",getcfgCountReload());
    (void) fprintf(stderr," log_error_messages=%d\n",getcfgLogErrmsg());
    (void) fprintf(stderr," log_visitor_info=%d\n",getcfgLogVisitorInfo());
    (void) fprintf(stderr," show_error_messages_to_browsers=%d\n",
                                getcfgShowErrmsg());                            
    (void) fprintf(stderr,"[ignore ips]\n");
    ip=getcfgIgips();
    for (ll=ip;ll;ll=ll->next)
    {
        (void) fprintf(stderr," %s\n",ll->item);
    }
    (void) fprintf(stderr,"[authorized]\n");
    ip=getcfgAhs();
    for (ll=ip;ll;ll=ll->next)
    {
        (void) fprintf(stderr," %s\n",ll->item);
    }

}
#endif /* 0 */



/*
** setCfgBoolen()
*/
static void setCfgBoolean(value,set)
char
    *value;
int
    *set;
{
    if (value != (char *) NULL)
    {
        if ((mutilsStrcasecmp(value,"y") == 0)      ||
            (mutilsStrcasecmp(value,"yes") == 0)    ||
            (mutilsStrcasecmp(value,"1") == 0)      ||
            (mutilsStrcasecmp(value,"true") == 0))
        {
            *set=True;
        }
        else
        {
            if ((mutilsStrcasecmp(value,"n") == 0)      ||
                (mutilsStrcasecmp(value,"no") == 0)    ||
                (mutilsStrcasecmp(value,"0") == 0)      ||
                (mutilsStrcasecmp(value,"false") == 0))
            {
                *set=False;
            }
        }
    }
}

/*
** sets an positive integer. if value string does not contain a number
** the int will be set to 0
*/
static void setCfgInteger(value,set)
char
    *value;
int
    *set;
{
    if (value != (char *) NULL)
    {
        *set=atoi(value);
        if (*set < 0)
            *set=0;
    }
}

/*
** functions to access static cfg struct from outside, better than globals
** anyway.
*/

char *getcfgVersion()
{
    if (s_cfg)
        return (s_cfg->version);
    else
        return (NULL);
}

int getcfgAutoFileCreation()
{
    if (s_cfg)
        return (s_cfg->auto_file_creation);
    else
        return (False);
}

int getcfgStrictMode()
{
    if (s_cfg)
        return (s_cfg->strict_mode);
    else
        return (True);
}

int getcfgArgbDatabase()
{
    if (s_cfg)
        return (s_cfg->allow_rgb_database);
    else
        return (False);
}

int getcfgCountReload()
{
    if (s_cfg)
        return (s_cfg->count_reload);
    else
        return (True);
}

int getcfgLogErrmsg()
{
    if (s_cfg)
        return (s_cfg->log_error_messages);
    else
        return(True);

}

int getcfgLogVisitorInfo()
{
    if (s_cfg)
        return (s_cfg->log_visitor_info);
    else
        return(False);
}

int getcfgLogRotationInterval()
{
    if (s_cfg)
        return (s_cfg->log_rotation_interval);
    else
        return(0);
}

int getcfgShowErrmsg()
{
    if (s_cfg)
        return (s_cfg->show_error_messages_to_browsers);
    else
        return(True);
}

Sllist *getcfgIgips()
{
    if (s_cfg)
        return (s_cfg->ignore_ips);
    else
        return(NULL);
}

Sllist *getcfgAhs()
{
    if (s_cfg)
        return (s_cfg->auth_hosts);
    else
        return(NULL);
}


