/*
 *  String drawing routines
 *
 *  RCS:
 *      $Revision: 2.3 $
 *      $Date: 1996/05/03 02:21:34 $
 *
 *  Security:
 *      Unclassified
 *
 *  Limitations and Comments:
 *      text
 *
 *  Development History:
 *      who                 when        why
 *      muquit@semcor.com   30-Aug-95   first cut
 *      muquit, May-21-1999, TrueType Font support, gets activated
 *                           if HAS_TRUE_TYPE is defined
 */

#include "combine.h"
#include "gdfonts.h"

void ImageChar(image,f,x,y,c,font_info)
Image
    *image;
gdFontPtr
    f;
int
    x,
    y;
char
    c;
SFontInfo
    *font_info;
{
    int
        cx,
        cy,
        px,
        py;

    int
        fline;

    cx=0;
    cy=0;

    if ((c < f->offset) || (c >= (f->offset + f->nchars)))
        return;

    fline=(c - f->offset) * f->h * f->w;
    for (py = y; (py < (y + f->h)); py++)
    {
        for (px = x; (px < (x + f->w)); px++)
        {
            if (font_info->do_bg == True)
            {
                if (f->data[fline + cy * f->w + cx])
                    SetPixel(image,px,py,font_info,0);
                else
                    SetPixel(image,px,py,font_info,1);
            }
            else
            {
                if (f->data[fline + cy * f->w + cx])
                    SetPixel(image,px,py,font_info,0);
            }                
            cx++;
        }
        cx=0;
        cy++;
    }
}

void ImageString(image,f,x,y,s,font_info)
Image
    *image;
gdFontPtr
    f;
int
    x,
    y;
char
    *s;
SFontInfo
    *font_info;
{
    int
        i,
        l;

    l=(int) strlen(s);

    for (i=0; i < l; i++)
    {
        ImageChar(image,f,x,y,s[i],font_info);
        x += f->w;
    }
}

#ifdef HAS_TRUE_TYPE

#define MAX_GLYPHS  1024

typedef struct
{
    short
        x1,
        y1,
        x2,
        y2;
} MSegment;


/*
**  TryType font support
**
**  Limitations and Comments:
**  Uses TrueType font library from http://www.freetype.org/
**  Code adapted from ImageMagick 4.2.5 and freetype demo utils
**  Yeap John Cristy of ImageMagick is my Guru.
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   May-21-1999    first cut
*/

#include <freetype.h>

static void renderGlyph(canvas,character,glyph,x_off,y_off,glyph_metrics)
TT_Raster_Map
    *canvas,
    *character;
TT_Glyph
    glyph;
int
    x_off,
    y_off;
TT_Glyph_Metrics
    *glyph_metrics;
{
    int
        x,
        y,
        i;

    unsigned char
        *p,
        *q;

    MSegment
        bounds;


    /* render Glyph */
    q=(unsigned char *) character->bitmap;
    for (i=0; i < character->size; i++)
        *q++=0;

    TT_Get_Glyph_Pixmap(glyph,character,-(glyph_metrics->bbox.xMin & -64),
            -(glyph_metrics->bbox.yMin & -64));

    /* composite chararactr on canvas */
    x_off=((glyph_metrics->bbox.xMin & -64)/64)+x_off;
    y_off=(-(glyph_metrics->bbox.yMin & -64)/64)-y_off;

    bounds.x1=x_off < 0 ? -x_off : 0;
    bounds.y1=y_off < 0 ? -y_off : 0;

    bounds.x2=(int) canvas->cols-x_off;
    if (bounds.x2 > character->cols)
        bounds.x2=character->cols;

    bounds.y2=(int) canvas->rows-y_off;
    if (bounds.y2 > character->rows)
        bounds.y2=character->rows;

    if (bounds.x1 >= bounds.x2)
        return;

    for (y=bounds.y1; y < bounds.y2; y++)
    {
        p=((unsigned char *) character->bitmap)+y*character->cols+bounds.x1;
        q=((unsigned char *) canvas->bitmap)
            +(y+y_off)*canvas->cols+bounds.x1+x_off;
        for (x=bounds.x1; x < bounds.x2; x++)
            *q++|=(*p++);
    }
}    

/*
**  ttfStringImage()
**  render a string as image
**
**  Parameters:
**  char    *str        the string to convert to image
*   char    *font_file  the full path of the TTF font file
*   int     bgr,bgg,bgb the red, green and blue component of background
*   int     fgr,fgg,fgb the red,green and blue component of foreground
*   int     border      the border space in pixel around the text
*   int     smooth      1 or 0, do anti alias or not
*   int     point_size  the point size (i pt = 1/72 inch)
*   int     x_res       the X resolution in dot/inch
*   int     y_res       the Y resolution in dot/inh
*   int     errstr      the error message (returns) if it occurs
**
**  Return Values:
**
**  Limitations and Comments:
**  errstr can be overflowed if not careful.
**
**
**  Development History:
**      who                  when           why
**      ma_muquit@fccc.edu   May-21-1999    first cut
*/

Image *ttfStringImage(str,font_file,
        bgr,bgg,bgb,fgr,fgg,fgb,
        border,smooth,point_size,x_res,y_res,
        errstr)
char
    *str;

char
    *font_file;

int
    bgr,
    bgg,
    bgb;

int fgr,
    fgg,
    fgb;

int
    border;

int
    smooth;

int
    point_size;

int
    x_res,
    y_res;

char
    *errstr;

{
    Image
        *base_image,
        *image;

    unsigned char
        *p;

    Runlength
        *q;

    TT_Engine
        engine;

    TT_Face
        face;

    TT_Error
        error;

    TT_Face_Properties
        face_properties;

    TT_Instance
        instance;

    TT_UShort
        code;

    TT_Glyph
        *glyphs=NULL;

    TT_Glyph_Metrics
        glyph_metrics;

    TT_CharMap
        char_map;

    TT_Instance_Metrics
        instance_metrics;

    TT_Raster_Map
        canvas,
        character;

    unsigned short
        idx;

    int
        i,
        x,
        y,
        str_len,
        character_map,
        number_glyphs,
        width,
        height,
        ascent,
        descent,
        upm;

    unsigned short
        encoding,
        platform;

    *errstr='\0';
    image=NULL;
    base_image=NULL;

    if (str == NULL || *str == '\0')
    {
        (void) strcpy(errstr,"NULL string supplied for TTF image");
        return ((Image *) NULL);
    }

    str_len=strlen(str);

    /* initialize font engine */
    error=TT_Init_FreeType(&engine);
    if (error)
    {
        (void) strcpy(errstr,"Failed to initialize TrueType font engine");
        return ((Image *) NULL);
    }

    /* open the TTF font file */
    error=TT_Open_Face(engine,font_file,&face);
    if (error)
    {
        (void) sprintf(errstr,"Could not open TTF font: %s",font_file);
        return ((Image *) NULL);
    }

    TT_Get_Face_Properties(face,&face_properties);
    error=TT_New_Instance(face,&instance);
    if (error)
    {
        (void) strcpy(errstr,"Could not create new TTF Instance");
        return ((Image *) NULL);
    }

    error|=TT_Set_Instance_Resolutions(instance,x_res,y_res);
    error|=TT_Set_Instance_CharSize(instance,point_size*64);
    if (error)
    {
        (void) strcpy(errstr,"Could not initialize TTF Instance");
        return ((Image *) NULL);
    }

    for (code=0; code < (int) face_properties.num_CharMaps; code++)
    {
        TT_Get_CharMap_ID(face,code,&platform,&encoding);
        if (((platform == 3) && (encoding == 1)) ||
            ((platform == 0) && (encoding == 0)))
        {
            TT_Get_CharMap(face,code,&char_map);
            break;
        }
    }

    number_glyphs=0;
    character_map=True;
    if (code == face_properties.num_CharMaps)
    {
        TT_Get_Face_Properties(face,&face_properties);
        number_glyphs=face_properties.num_Glyphs;
        character_map=False;
    }

    glyphs=(TT_Glyph *) malloc(MAX_GLYPHS*sizeof(TT_Glyph));
    if (glyphs == (TT_Glyph *) NULL)
    {
        (void) strcpy(errstr,"Memory allocation failed for TT_Glyph");
        return ((Image *) NULL);
    }

    for (i=0; i< MAX_GLYPHS; i++)
        glyphs[i].z=(TT_Glyph *) NULL;

    for (i=0; i < str_len; i++)
    {
        if (glyphs[(unsigned char) str[i]].z != (TT_Glyph *) NULL)
            continue;
        if (character_map)
            code=TT_Char_Index(char_map,(unsigned char) str[i]);
        else
        {
            code=((int) str[i]-' '+1) < 0 ? 0 : (str[i]-' '+1);
            if (code >= number_glyphs)
                code=0;
        }
        error=TT_New_Glyph(face,&glyphs[(unsigned char) str[i]]);
        error|=TT_Load_Glyph(instance,glyphs[(unsigned char) str[i]],code,
                TTLOAD_SCALE_GLYPH | TTLOAD_HINT_GLYPH);
        if (error)
        {
            (void) strcpy(errstr,"Could not Initialize TTF Glyph");
            image=(Image *) NULL;
            goto ExitProcessing;
        }
    }
    TT_Get_Face_Properties(face,&face_properties);
    TT_Get_Instance_Metrics(instance,&instance_metrics);

    upm = face_properties.header->Units_Per_EM;
    ascent=(face_properties.horizontal->Ascender*instance_metrics.y_ppem)/upm;
    descent=(face_properties.horizontal->Descender*instance_metrics.y_ppem)/upm;

    /*
    height=((int) (face_properties.horizontal->Ascender*
        instance_metrics.y_ppem)/ (int) face_properties.header->Units_Per_EM)-
        ((int) (face_properties.horizontal->Descender*instance_metrics.y_ppem)/
         (int) face_properties.header->Units_Per_EM)+1;
    */

    width=border*2;
    height=2*border+ascent-descent+1;

    for (i=0; i < str_len; i++)
    {
        if (glyphs[(unsigned char) str[i]].z == (TT_Glyph *) NULL)
            continue;
        TT_Get_Glyph_Metrics(glyphs[(unsigned char) str[i]],&glyph_metrics);
        width+=glyph_metrics.advance/64;
    }

    canvas.rows=height;
    canvas.width=(width+3) & -4;
    canvas.flow=TT_Flow_Down;
    canvas.cols=canvas.width;
    canvas.size=canvas.rows*canvas.width;
    canvas.bitmap=(void *) malloc(canvas.size);
    if (!canvas.bitmap)
    {
        (void) strcpy(errstr,"Memory allocation problem for TTF canvas.bitmap");
        goto ExitProcessing;
    }

    p=(unsigned char *) canvas.bitmap;
    for (i=0; i < canvas.size; i++)
        *p++=0;

    character.rows=height;
    character.width=(instance_metrics.x_ppem+32+3) & -4;
    character.flow=TT_Flow_Down;
    character.cols=character.width;
    character.size=character.rows*character.width;
    character.bitmap=(void *) malloc(character.size);
    if (!character.bitmap)
    {
     (void) strcpy(errstr,"Memory allocation problem for TTF character.bitmap");
     goto ExitProcessing;
    }

    x=border;
    y=((int) -(face_properties.horizontal->Descender*instance_metrics.y_ppem)/
            (int) face_properties.header->Units_Per_EM);
    y=border-descent;
    for (i=0; i < str_len; i++)
    {
        if (glyphs[(unsigned char) str[i]].z == (TT_Glyph *) NULL)
            continue;
        TT_Get_Glyph_Metrics(glyphs[(unsigned char) str[i]],&glyph_metrics);
        renderGlyph(&canvas,&character,glyphs[(unsigned char) str[i]],x,y,
                &glyph_metrics);
        x+=glyph_metrics.advance/64;
    }

    /* malloc for the image */
    image=AllocateImageStruct();
    if (image == (Image *) NULL)
    {
        (void) strcpy(errstr,"Memory allocation problem with TTF AllocateImageStruct()\n");
        goto ExitProcessing;
    }

    image->columns=canvas.width;
    image->rows=canvas.rows;
    image->packets=image->columns*image->rows;
    image->pixels=(Runlength *)
         malloc(image->packets*sizeof(Runlength));
    if (image->pixels == (Runlength *) NULL)
    {
     (void) strcpy(errstr,"Memory allocation problem with TTF image->pixels\n");
      goto ExitProcessing;
    }

    image->alpha=True;
    image->class=DirectClass;
    p=(unsigned char *) canvas.bitmap;
    q=image->pixels;
    x=0;
    for (i=0; i < (int) image->packets; i++)
    {
        q->red=fgr;
        q->green=fgg;
        q->blue=fgb;
        if (smooth)
            q->index=(int) (MaxRGB*Min(*p,4))/4;
        else
            q->index=(*p) > 1 ? MaxRGB : 0;

        if (q->index == 0)  /* transparent */
        {
            q->red=(~q->red);
            q->green=(~q->green);
            q->blue=(~q->blue);
        }
        q->length=0;
        x++;
        if (x == (int) image->columns)
        {
            if ((image->columns % 2) != 0)
                p++;
            x=0;
        }
        p++;
        q++;
    }

    /* create the base image */
    base_image=CreateBaseImage(image->columns,image->rows,
            bgr,bgg,bgb,DirectClass);
    if (base_image == (Image *) NULL)
    {
        (void) strcpy(errstr,"Could not create TTF Base Image");
        goto ExitProcessing;
    }

    /* now flatten the TTF char image on the base image */
    FlattenImage(base_image,image,AnnotateCompositeOp, 0,0);

    if (canvas.bitmap)
        (void) free ((char *) canvas.bitmap);

    if (character.bitmap)
        (void) free ((char *) character.bitmap);

ExitProcessing:
    if (glyphs)
    {
        for (i=0; i < MAX_GLYPHS; i++)
            TT_Done_Glyph(glyphs[i]);

        (void) free ((char *) glyphs);
    }

    TT_Done_Instance(instance);
    TT_Close_Face(face);
    TT_Done_FreeType(engine);

    if (image != (Image *) NULL)
    {
        DestroyAnyImageStruct(&image);
    }
    return (base_image);

}

#endif /* HAS_TRUE_TYPE */
