/* 
 * PHP3 Internationalization support program.
 *
 * Copyright (c) 1999,2000 by the PHP3 internationalization team.
 * All rights reserved.
 *
 * This program is free software. You can use, redistribute and/or modify
 * without fee under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY including implied or express warranty of
 * marchantability or fitness for a particular purpose.
 *
 * Currently, the "PHP3 internationalization team" has no relationship with
 * the "PHP Development Team". But we hope these code will be integrated
 * into the PHP3, and it will be distributed as a part of PHP3.
 *
 * See README_i18n for more detail.
 *
 * Authors:
 *    Hironori Sato <satoh@jpnnet.com>
 *    Shigeru Kanemoto <sgk@happysize.co.jp>
 *    Tsukada Takuya <tsukada@fminn.nagano.nagano.jp>
 */

/* 
 * PHP4 multibyte regular expression module
 * Authors:
 *    Tsukada Takuya <tsukada@fminn.nagano.nagano.jp>
 */

/* $Id$ */


#include "php.h"
#include "php_ini.h"
#include "php_config.h"
#include "config.h"
#include "mbregex.h"
#include "php_mbregex.h"
#include "php_mbregex_globals.h"

#if HAVE_MBREGEX

static unsigned char third_argument_force_ref[] = { 3, BYREF_NONE, BYREF_NONE, BYREF_FORCE };

function_entry mbregex_functions[] = {
	PHP_FE(mbregex_info,		NULL)
	PHP_FE(mbregex_encoding,	NULL)
	PHP_FE(mbereg,			third_argument_force_ref)
	PHP_FE(mberegi,			third_argument_force_ref)
	PHP_FE(mbereg_replace,			NULL)
	PHP_FE(mberegi_replace,			NULL)
	PHP_FE(mbsplit,					NULL)
	PHP_FE(mbereg_match,			NULL)
	PHP_FE(mbereg_search,			NULL)
	PHP_FE(mbereg_search_pos,		NULL)
	PHP_FE(mbereg_search_regs,		NULL)
	PHP_FE(mbereg_search_init,		NULL)
	PHP_FE(mbereg_search_getregs,	NULL)
	PHP_FE(mbereg_search_getpos,	NULL)
	PHP_FE(mbereg_search_setpos,	NULL)
	{ NULL, NULL, NULL }
};

zend_module_entry mbregex_module_entry = {
	"mbregex",
	mbregex_functions,
	PHP_MINIT(mbregex),
	PHP_MSHUTDOWN(mbregex),
	NULL,
	PHP_RSHUTDOWN(mbregex),
	PHP_MINFO(mbregex),
	STANDARD_MODULE_PROPERTIES
};

ZEND_DECLARE_MODULE_GLOBALS(mbregex)

#ifdef COMPILE_DL_MBREGEX
ZEND_GET_MODULE(mbregex)
#endif


/*
 * string buffer
 */
struct strbuf {
	unsigned char* buffer;
	int length;
	int pos;
	int allocsz;
};

static void
php_mbregex_strbuf_init(struct strbuf *pd)
{
	if (pd) {
		pd->buffer = (char*)0;
		pd->length = 0;
		pd->pos = 0;
		pd->allocsz = 64;
	}
}

static int
php_mbregex_strbuf_ncat(struct strbuf *pd, const unsigned char *psrc, int len)
{
	if (pd == NULL || psrc == NULL) {
		return -1;
	}

	if ((pd->pos + len) >= pd->length) {
		/* reallocate buffer */
		int newlen = pd->length + pd->allocsz + len;
		unsigned char *tmp = (unsigned char*)erealloc((void*)pd->buffer, newlen);
		if (tmp == NULL) {
			return -1;
		}
		pd->length = newlen;
		pd->buffer = tmp;
	}

	while (len > 0) {
		pd->buffer[pd->pos++] = *psrc++;
		len--;
	}

	return len;
}


/*
 * encoding name resolver
 */
static int
php_mbregex_name2mbctype(const char *pname)
{
	int mbctype;

	mbctype = -1;
	if (pname != NULL) {
		if (strcasecmp("EUC-JP", pname) == 0) {
			mbctype = MBCTYPE_EUC;
		} else if (strcasecmp("UTF-8", pname) == 0) {
			mbctype = MBCTYPE_UTF8;
		} else if (strcasecmp("SJIS", pname) == 0) {
			mbctype = MBCTYPE_SJIS;
		} else if (strcasecmp("ascii", pname) == 0) {
			mbctype = MBCTYPE_ASCII;
		} else if (strcasecmp("euc", pname) == 0) {
			mbctype = MBCTYPE_EUC;
		} else if (strcasecmp("eucJP", pname) == 0) {
			mbctype = MBCTYPE_EUC;
		} else if (strcasecmp("EUC_JP", pname) == 0) {
			mbctype = MBCTYPE_EUC;
		} else if (strcasecmp("Shift_JIS", pname) == 0) {
			mbctype = MBCTYPE_SJIS;
		}
	}

	return mbctype;
}

static const char*
php_mbregex_mbctype2name(int mbctype)
{
	const char *p;

	if (mbctype == MBCTYPE_EUC) {
		p = "EUC-JP";
	} else if(mbctype == MBCTYPE_UTF8) {
		p = "UTF-8";
	} else if(mbctype == MBCTYPE_SJIS) {
		p = "SJIS";
	} else if(mbctype == MBCTYPE_ASCII) {
		p = "ascii";
	} else {
		p = "unknown";
	}

	return p;
}


/*
 * regex cache
 */
static int
php_mbregex_compile_pattern(mb_regex_t *pre, const char *pattern, int patlen, int options, int mbctype)
{
	int res = 0;
	const char *err_str = NULL;
	mb_regex_t *rc = NULL;
	MBREGEXLS_FETCH();

	if(zend_hash_find(&MBREGEXG(ht_rc), (char *)pattern, patlen+1, (void **) &rc) == FAILURE ||
			rc->options != options || rc->mbctype != mbctype) {
		memset(pre, 0, sizeof(*pre));
		pre->fastmap = (char*)malloc((1 << MBRE_BYTEWIDTH)*sizeof(char));
		pre->options = options;
		pre->mbctype = mbctype;
		err_str = mbre_compile_pattern(pattern, patlen, pre);
		if (!err_str) {
			zend_hash_update(&MBREGEXG(ht_rc), (char *) pattern, patlen+1,
					(void *) pre, sizeof(*pre), NULL);
		} else {
			free(pre->fastmap);
			pre->fastmap = (char*)0;
			php_error(E_WARNING, "mbregex compile err: %s", err_str);
			res = 1;
		}
	} else {
		memcpy(pre, rc, sizeof(*pre));
	}

	return res;
}

static void
php_mbregex_free_cache(mb_regex_t *pre) 
{
	mbre_free_pattern(pre);
}


/*
 * php.ini
 */
static PHP_INI_MH(OnUpdate_mbregex_encoding)
{
	int mbctype;
	MBREGEXLS_FETCH();

	if (new_value != NULL) {
		mbctype = php_mbregex_name2mbctype(new_value);
		if (mbctype < 0) {
			return FAILURE;
		}
		MBREGEXG(default_mbctype) = mbctype;
		MBREGEXG(current_mbctype) = mbctype;
	}

	return SUCCESS;
}


PHP_INI_BEGIN()
	PHP_INI_ENTRY("mbregex.encoding", NULL, PHP_INI_ALL, OnUpdate_mbregex_encoding)
PHP_INI_END()


/*
 * initialize
 */
static void
php_mbregex_init_globals(zend_mbregex_globals *pglobals)
{
	pglobals->default_mbctype = MBCTYPE_ASCII;
	pglobals->current_mbctype = MBCTYPE_ASCII;
	zend_hash_init(&(pglobals->ht_rc), 0, NULL, (void (*)(void *)) php_mbregex_free_cache, 1);
	pglobals->search_str_val = (unsigned char*)0;
	pglobals->search_str_len = 0;
	pglobals->search_re = (mb_regex_t*)0;
	pglobals->search_pos = 0;
	pglobals->search_regs = (struct mbre_registers*)0;
}


PHP_MINIT_FUNCTION(mbregex)
{
	ZEND_INIT_MODULE_GLOBALS(mbregex, php_mbregex_init_globals, NULL);

	REGISTER_INI_ENTRIES();

	return SUCCESS;
}


PHP_MSHUTDOWN_FUNCTION(mbregex)
{
	MBREGEXLS_FETCH();

	zend_hash_destroy(&MBREGEXG(ht_rc));
	UNREGISTER_INI_ENTRIES();

	return SUCCESS;
}


PHP_RSHUTDOWN_FUNCTION(mbregex)
{
	MBREGEXLS_FETCH();

	MBREGEXG(current_mbctype) = MBREGEXG(default_mbctype);
	if (MBREGEXG(search_str_val)) {
		efree(MBREGEXG(search_str_val));
		MBREGEXG(search_str_val) = (unsigned char*)0;
	}
	MBREGEXG(search_str_len) = 0;
	MBREGEXG(search_pos) = 0;
	if (MBREGEXG(search_re)) {
		efree(MBREGEXG(search_re));
		MBREGEXG(search_re) = (mb_regex_t *)0;
	}
	if (MBREGEXG(search_regs)) {
		mbre_free_registers(MBREGEXG(search_regs));
		efree(MBREGEXG(search_regs));
		MBREGEXG(search_regs) = (struct mbre_registers*)0;
	}
	if (zend_hash_num_elements(&MBREGEXG(ht_rc)) > PHP_MBREGEX_MAXCACHE) {
		zend_hash_clean(&MBREGEXG(ht_rc));
	}

	return SUCCESS;
}


PHP_MINFO_FUNCTION(mbregex)
{
	DISPLAY_INI_ENTRIES();
}


static void
php_mbregex_init_option(const char *parg, int narg, int *option, int *eval) 
{
	int n;
	char c;

	if (parg) {
		n = 0;
		while(n < narg) {
			c = parg[n++];
			if (option) {
				switch (c) {
				case 'i':
					*option |= MBRE_OPTION_IGNORECASE;
					break;
				case 'x':
					*option |= MBRE_OPTION_EXTENDED;
					break;
				case 'm':
					*option |= MBRE_OPTION_MULTILINE;
					break;
				case 's':
					*option |= MBRE_OPTION_SINGLELINE;
					break;
				case 'p':
					*option |= MBRE_OPTION_POSIXLINE;
					break;
				case 'l':
					*option |= MBRE_OPTION_LONGEST;
					break;
				default:
					break;
				}
			}
			if (eval && (c == 'e')) {
				*eval = 1;
			}
		}
	}
}


/*
 * php funcions
 */

PHP_FUNCTION(mbregex_info)
{
	int n;
	MBREGEXLS_FETCH();

	n = zend_hash_num_elements(&MBREGEXG(ht_rc));
	php_printf("num cache: %d<br>", n);

#if ZEND_DEBUG
	zend_hash_display(&MBREGEXG(ht_rc));
#endif
}


/* {{{ proto string mbregex_encoding([string encoding])
   Returns the current encoding as a string. */
PHP_FUNCTION(mbregex_encoding)
{
	pval **arg1;
	int mbctype;
	MBREGEXLS_FETCH();

	if (ARG_COUNT(ht) == 0) {
		RETVAL_STRING((char*)php_mbregex_mbctype2name(MBREGEXG(current_mbctype)), 1);
	} else if (ARG_COUNT(ht) == 1 && zend_get_parameters_ex(1, &arg1) != FAILURE) {
		convert_to_string_ex(arg1);
		mbctype = php_mbregex_name2mbctype((*arg1)->value.str.val);
		if (mbctype < 0) {
			php_error(E_WARNING, "unknown encoding \"%s\"", (*arg1)->value.str.val);
			RETVAL_FALSE;
		} else {
			MBREGEXG(current_mbctype) = mbctype;
			RETVAL_TRUE;
		}
	} else {
		WRONG_PARAM_COUNT;
	}
}
/* }}} */


/* regex match */
static void
php_mbereg_exec(INTERNAL_FUNCTION_PARAMETERS, int icase)
{
	pval **arg_pattern, **arg_string, **array = NULL;
	mb_regex_t re;
	struct mbre_registers regs = {0, 0, 0, 0};
	int i, err, match_len, string_len, option, beg, end;
	char *str;
	MBREGEXLS_FETCH()

	switch(ARG_COUNT(ht)) {
	case 2:
		if (zend_get_parameters_ex(2, &arg_pattern, &arg_string) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;

	case 3:
		if (zend_get_parameters_ex(3, &arg_pattern, &arg_string, &array) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		if (!ParameterPassedByReference(ht, 3)) {
			php_error(E_WARNING, "Array to be filled with values must be passed by reference.");
			RETURN_FALSE;
		}
		break;

	default:
		WRONG_PARAM_COUNT;
		break;
	}

	option = 0;
	if (icase) {
		option |= MBRE_OPTION_IGNORECASE;
	}

	/* compile the regular expression from the supplied regex */
	if ((*arg_pattern)->type == IS_STRING) {
		option |= MBRE_OPTION_EXTENDED;
	} else {
		/* we convert numbers to integers and treat them as a string */
		if ((*arg_pattern)->type == IS_DOUBLE) {
			convert_to_long_ex(arg_pattern);	/* get rid of decimal places */
		}
		convert_to_string_ex(arg_pattern);
		/* don't bother doing an extended regex with just a number */
	}
	err = php_mbregex_compile_pattern(
	     &re,
	     (*arg_pattern)->value.str.val,
	     (*arg_pattern)->value.str.len,
	     option, MBREGEXG(current_mbctype));
	if (err) {
		RETURN_FALSE;
	}

	/* actually execute the regular expression */
	convert_to_string_ex(arg_string);
	err = mbre_search(
	     &re,
	     (*arg_string)->value.str.val,
	     (*arg_string)->value.str.len,
	      0, (*arg_string)->value.str.len,
	     &regs);
	if (err < 0) {
		mbre_free_registers(&regs);
		RETURN_FALSE;
	}

	match_len = 1;
	str = (*arg_string)->value.str.val;
	if (array) {
		match_len = regs.end[0] - regs.beg[0];
		string_len = (*arg_string)->value.str.len;
		pval_destructor(*array);	/* start with clean array */
		array_init(*array);
		for (i = 0; i < regs.num_regs; i++) {
			beg = regs.beg[i];
			end = regs.end[i];
			if (beg >= 0 && beg < end && end <= string_len) {
				add_index_stringl(*array, i, &str[beg], end - beg, 1);
			} else {
				add_index_bool(*array, i, 0);
			}
		}
	}

	mbre_free_registers(&regs);
	if (match_len == 0) {
		match_len = 1;
	}
	RETVAL_LONG(match_len);
}

/* {{{ proto int mbereg(string pattern, string string [, array registers])
   Regular expression match for multibyte string */
PHP_FUNCTION(mbereg)
{
	php_mbereg_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}
/* }}} */

/* {{{ proto int mberegi(string pattern, string string [, array registers])
   Case-insensitive regular expression match for multibyte string */
PHP_FUNCTION(mberegi)
{
	php_mbereg_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */



/* regex replacement */
static void
php_mbereg_replace_exec(INTERNAL_FUNCTION_PARAMETERS, int option)
{
	pval **arg_pattern, **arg_replace, **arg_string, **arg_option;
	char *p, *string, *replace;
	mb_regex_t re;
	struct mbre_registers regs = {0, 0, 0, 0};
	struct strbuf outdev, evaldev, *pdevice;
	int i, n, err, pos, replace_len, string_len, eval;
	char *description = NULL;
	zval retval;
	MBREGEXLS_FETCH();
	CLS_FETCH();
	ELS_FETCH();

	eval = 0;
	switch(ARG_COUNT(ht)) {
	case 3:
		if (zend_get_parameters_ex(3, &arg_pattern, &arg_replace, &arg_string) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;

	case 4:
		if (zend_get_parameters_ex(4, &arg_pattern, &arg_replace, &arg_string, &arg_option) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		convert_to_string_ex(arg_option);
		option = 0;
		php_mbregex_init_option((*arg_option)->value.str.val, (*arg_option)->value.str.len, &option, &eval);
		break;

	default:
		WRONG_PARAM_COUNT;
		break;
	}

	convert_to_string_ex(arg_pattern);
	/* create regex pattern buffer */
	err = php_mbregex_compile_pattern(
	    &re,
	    (*arg_pattern)->value.str.val,
	    (*arg_pattern)->value.str.len,
	    option, MBREGEXG(current_mbctype));
	if (err) {
		RETURN_FALSE;
	}

	convert_to_string_ex(arg_replace);
	replace = (*arg_replace)->value.str.val;
	replace_len = (*arg_replace)->value.str.len;

	convert_to_string_ex(arg_string);
	string = (*arg_string)->value.str.val;
	string_len = (*arg_string)->value.str.len;

	/* initialize string buffer (auto reallocate buffer) */
	php_mbregex_strbuf_init(&outdev);
	php_mbregex_strbuf_init(&evaldev);
	outdev.allocsz = (string_len >> 2) + 8;

	if (eval) {
		pdevice = &evaldev;
		description = zend_make_compiled_string_description("mbregex replace");
	} else {
		pdevice = &outdev;
		description = NULL;
	}

	/* do the actual work */
	err = 0;
	pos = 0;
	while (err >= 0) {
		err = mbre_search(&re, string, string_len, pos, string_len - pos, &regs);
		if (err <= -2) {
			php_error(E_WARNING, "mbregex search failure in php_mbereg_replace_exec()");
			break;
		}
		if (err >= 0) {
			/* copy the part of the string before the match */
			php_mbregex_strbuf_ncat(&outdev, &string[pos], regs.beg[0] - pos);
			/* copy replacement and backrefs */
			i = 0;
			p = replace;
			while (i < replace_len) {
				n = -1;
				if (p[0] == '\\' && p[1] >= '0' && p[1] <= '9') {
					n = p[1] - '0';
				}
				if (n >= 0 && n < regs.num_regs) {
					if (regs.beg[n] >= 0 && regs.beg[n] < regs.end[n] && regs.end[n] <= string_len) {
						php_mbregex_strbuf_ncat(pdevice, &string[regs.beg[n]], regs.end[n] - regs.beg[n]);
					}
					p += 2;
					i += 2;
				} else {
					php_mbregex_strbuf_ncat(pdevice, p, 1);
					p++;
					i++;
				}
			}
			if (eval) {
				/* null terminate buffer */
				php_mbregex_strbuf_ncat(&evaldev, "\0", 1);
				/* do eval */
				zend_eval_string(evaldev.buffer, &retval, description CLS_CC ELS_CC);
				/* result of eval */
				convert_to_string(&retval);
				php_mbregex_strbuf_ncat(&outdev, retval.value.str.val, retval.value.str.len);
				/* Clean up */
				evaldev.pos = 0;
				zval_dtor(&retval);
			}
			n = regs.end[0];
			if (pos < n) {
				pos = n;
			} else {
				pos++;
			}
		} else { /* nomatch */
			/* stick that last bit of string on our output */
			php_mbregex_strbuf_ncat(&outdev, &string[pos], string_len - pos);
		}
	}

	if (description) {
		efree(description);
	}
	mbre_free_registers(&regs);
	if (evaldev.buffer) {
		efree((void*)evaldev.buffer);
	}
	n = outdev.pos;
	php_mbregex_strbuf_ncat(&outdev, "\0", 1);
	if (err <= -2) {
		if (outdev.buffer) {
			efree((void*)outdev.buffer);
		}
		RETVAL_FALSE;
	} else {
		RETVAL_STRINGL(outdev.buffer, n, 0);
	}
}

/* {{{ proto string mbereg_replace(string pattern, string replacement, string string [, string option])
   Replace regular expression for multibyte string */
PHP_FUNCTION(mbereg_replace)
{
	php_mbereg_replace_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, MBRE_OPTION_EXTENDED);
}
/* }}} */

/* {{{ proto string embregi_replace(string pattern, string replacement, string string)
   Case insensitive replace regular expression for multibyte string */
PHP_FUNCTION(mberegi_replace)
{
	php_mbereg_replace_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, MBRE_OPTION_EXTENDED | MBRE_OPTION_IGNORECASE);
}
/* }}} */


/* {{{ proto array mbsplit(string pattern, string string [, int limit])
   split multibyte string into array by regular expression */
PHP_FUNCTION(mbsplit)
{
	pval **arg_pat, **arg_str, **arg_count = NULL;
	mb_regex_t re;
	struct mbre_registers regs = {0, 0, 0, 0};
	char *string;
	int n, err, count, string_len, pos;
	MBREGEXLS_FETCH();

	count = -1;
	switch (ARG_COUNT(ht)) {
	case 2:
		if (zend_get_parameters_ex(2, &arg_pat, &arg_str) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;

	case 3:
		if (zend_get_parameters_ex(3, &arg_pat, &arg_str, &arg_count) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		convert_to_long_ex(arg_count);
		count = (*arg_count)->value.lval;
		break;

	default:
		WRONG_PARAM_COUNT;
		break;
	}

	if (array_init(return_value) == FAILURE) {
		RETURN_FALSE;
	}

	convert_to_string_ex(arg_pat);
	convert_to_string_ex(arg_str);

	/* create regex pattern buffer */
	err = php_mbregex_compile_pattern(
	     &re,
	     (*arg_pat)->value.str.val,
	     (*arg_pat)->value.str.len,
	     MBRE_OPTION_EXTENDED, MBREGEXG(current_mbctype));
	if (err) {
		RETURN_FALSE;
	}

	string = (*arg_str)->value.str.val;
	string_len = (*arg_str)->value.str.len;
	pos = 0;
	err = 0;
	/* churn through str, generating array entries as we go */
	while ((count != 0) &&
		   (err = mbre_search(&re, string, string_len, pos, string_len - pos, &regs)) >= 0) {
		n = regs.beg[0];
		if (n == pos) {
			/* match is at start of string, return empty string */
			add_next_index_stringl(return_value, empty_string, 0, 1);
		} else {
			/* On a real match */
			/* add it to the array */
			if (n < string_len) {
				n -= pos;
				add_next_index_stringl(return_value, &string[pos], n, 1);
			} else {
				err = -2;
				break;
			}
		}
		/* point at our new starting point */
		n = regs.end[0];
		if (pos < n) {
			pos = n;
		} else {
			pos++;
		}
		/* if we're only looking for a certain number of points,
		   stop looking once we hit it */
		if (count > 0) {
			count--;
		}
	}

	mbre_free_registers(&regs);

	/* see if we encountered an error */
	if (err <= -2) {
		php_error(E_WARNING, "mbregex search failure in mbsplit()");
		pval_destructor(return_value);
		RETURN_FALSE;
	}

	/* otherwise we just have one last element to add to the array */
	n = string_len - pos;
	if (n > 0) {
		add_next_index_stringl(return_value, &string[pos], n, 1);
	} else {
		add_next_index_stringl(return_value, empty_string, 0, 1);
	}
}
/* }}} */


/* {{{ proto bool mbereg_match(string pattern, string string [,string option])
   Regular expression match for multibyte string */
PHP_FUNCTION(mbereg_match)
{
	pval **arg_pattern, **arg_str, **arg_option;
	mb_regex_t re;
	int option, err;
	MBREGEXLS_FETCH()

	option = MBRE_OPTION_EXTENDED;
	switch (ARG_COUNT(ht)) {
	case 2:
		if (zend_get_parameters_ex(2, &arg_pattern, &arg_str) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;
	case 3:
		if (zend_get_parameters_ex(3, &arg_pattern, &arg_str, &arg_option) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		convert_to_string_ex(arg_option);
		option = 0;
		php_mbregex_init_option((*arg_option)->value.str.val, (*arg_option)->value.str.len, &option, NULL);
		break;
	default:
		WRONG_PARAM_COUNT;
		break;
	}

	/* create regex pattern buffer */
	convert_to_string_ex(arg_pattern);
	err = php_mbregex_compile_pattern(
	    &re,
	    (*arg_pattern)->value.str.val,
	    (*arg_pattern)->value.str.len,
	    option, MBREGEXG(current_mbctype));
	if (err) {
		RETURN_FALSE;
	}

	/* match */
	convert_to_string_ex(arg_str);
	err = mbre_match(&re, (*arg_str)->value.str.val, (*arg_str)->value.str.len, 0, NULL);
	if (err >= 0) {
		RETVAL_TRUE;
	}else{
		RETVAL_FALSE;
	}
}
/* }}} */


/* regex search */
static void
php_mbereg_search_exec(INTERNAL_FUNCTION_PARAMETERS, int mode)
{
	pval **arg_pattern, **arg_option;
	int n, i, err, pos, len, beg, end, option;
	unsigned char *str;
	MBREGEXLS_FETCH()

	option = MBRE_OPTION_EXTENDED;
	switch (ARG_COUNT(ht)) {
	case 0:
		break;
	case 1:
		if (zend_get_parameters_ex(1, &arg_pattern) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;
	case 2:
		if (zend_get_parameters_ex(2, &arg_pattern, &arg_option) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		convert_to_string_ex(arg_option);
		option = 0;
		php_mbregex_init_option((*arg_option)->value.str.val, (*arg_option)->value.str.len, &option, NULL);
		break;
	default:
		WRONG_PARAM_COUNT;
		break;
	}
	if (ARG_COUNT(ht) > 0) {
		/* create regex pattern buffer */
		convert_to_string_ex(arg_pattern);
		if (!MBREGEXG(search_re)) {
			MBREGEXG(search_re) = (mb_regex_t*)ecalloc(1, sizeof(mb_regex_t));
		}
		err = php_mbregex_compile_pattern(
		    MBREGEXG(search_re),
		    (*arg_pattern)->value.str.val,
		    (*arg_pattern)->value.str.len,
		    option, MBREGEXG(current_mbctype));
		if (err) {
			efree(MBREGEXG(search_re));
			MBREGEXG(search_re) = (mb_regex_t*)0;
			RETURN_FALSE;
		}
	}

	pos = MBREGEXG(search_pos);
	str = MBREGEXG(search_str_val);
	len = MBREGEXG(search_str_len);

	if (!MBREGEXG(search_re)) {
		php_error(E_WARNING, "no regex for search");
		RETURN_FALSE;
	}
	if (!MBREGEXG(search_str_val)) {
		php_error(E_WARNING, "no string for search");
		RETURN_FALSE;
	}
	if (MBREGEXG(search_regs)) {
		mbre_free_registers(MBREGEXG(search_regs));
		memset(MBREGEXG(search_regs), 0, sizeof(struct mbre_registers));
	} else {
		MBREGEXG(search_regs) = (struct mbre_registers*)ecalloc(1, sizeof(struct mbre_registers));
	}

	err = mbre_search(MBREGEXG(search_re), str, len, pos, len - pos, MBREGEXG(search_regs));

	if (err <= -2) {
		php_error(E_WARNING, "mbregex search failure in mbregex_search()");
		RETVAL_FALSE;
	} else if (err < 0) {
		MBREGEXG(search_pos) = len;
		RETVAL_FALSE;
	} else {
		switch (mode) {
		case 1:
			if (array_init(return_value) != FAILURE) {
				beg = MBREGEXG(search_regs)->beg[0];
				end = MBREGEXG(search_regs)->end[0];
				add_next_index_long(return_value, beg);
				add_next_index_long(return_value, end - beg);
			} else {
				RETVAL_FALSE;
			}
			break;
		case 2:
			if (array_init(return_value) != FAILURE) {
				n = MBREGEXG(search_regs)->num_regs;
				for (i = 0; i < n; i++) {
					beg = MBREGEXG(search_regs)->beg[i];
					end = MBREGEXG(search_regs)->end[i];
					if (beg >= 0 && beg <= end && end <= len) {
						add_index_stringl(return_value, i, &str[beg], end - beg, 1);
					} else {
						add_index_bool(return_value, i, 0);
					}
				}
			} else {
				RETVAL_FALSE;
			}
			break;
		default:
			RETVAL_TRUE;
			break;
		}
		end = MBREGEXG(search_regs)->end[0];
		if (pos < end) {
			MBREGEXG(search_pos) = end;
		} else {
			MBREGEXG(search_pos) = pos + 1;
		}
	}

	if (err < 0) {
		mbre_free_registers(MBREGEXG(search_regs));
		efree(MBREGEXG(search_regs));
		MBREGEXG(search_regs) = (struct mbre_registers*)0;
	}
}
/* }}} */


/* {{{ proto bool mbereg_search([string pattern, string option])
   Regular expression search for multibyte string */
PHP_FUNCTION(mbereg_search)
{
	php_mbereg_search_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, 0);
}
/* }}} */


/* {{{ proto array mbereg_search_pos([string pattern, string option])
   Regular expression search for multibyte string */
PHP_FUNCTION(mbereg_search_pos)
{
	php_mbereg_search_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, 1);
}
/* }}} */


/* {{{ proto array mbereg_search_regs([string pattern, string option])
   Regular expression search for multibyte string */
PHP_FUNCTION(mbereg_search_regs)
{
	php_mbereg_search_exec(INTERNAL_FUNCTION_PARAM_PASSTHRU, 2);
}
/* }}} */


/* {{{ proto bool mbereg_search_init(string string [, string pattern, string option])
   Initialize string and regular expression for search. */
PHP_FUNCTION(mbereg_search_init)
{
	pval **arg_str, **arg_pattern, **arg_option;
	int err, option;
	MBREGEXLS_FETCH()

	option = MBRE_OPTION_EXTENDED;
	switch (ARG_COUNT(ht)) {
	case 1:
		if (zend_get_parameters_ex(1, &arg_str) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;
	case 2:
		if (zend_get_parameters_ex(2, &arg_str, &arg_pattern) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		break;
	case 3:
		if (zend_get_parameters_ex(3, &arg_str, &arg_pattern, &arg_option) == FAILURE) {
			WRONG_PARAM_COUNT;
		}
		convert_to_string_ex(arg_option);
		option = 0;
		php_mbregex_init_option((*arg_option)->value.str.val, (*arg_option)->value.str.len, &option, NULL);
		break;
	default:
		WRONG_PARAM_COUNT;
		break;
	}
	if (ARG_COUNT(ht) > 1) {
		/* create regex pattern buffer */
		convert_to_string_ex(arg_pattern);
		if (!MBREGEXG(search_re)) {
			MBREGEXG(search_re) = (mb_regex_t*)ecalloc(1, sizeof(mb_regex_t));
		}
		err = php_mbregex_compile_pattern(
		    MBREGEXG(search_re),
		    (*arg_pattern)->value.str.val,
		    (*arg_pattern)->value.str.len,
		    option, MBREGEXG(current_mbctype));
		if (err) {
			efree(MBREGEXG(search_re));
			MBREGEXG(search_re) = (mb_regex_t*)0;
			RETURN_FALSE;
		}
	}

	if (MBREGEXG(search_str_val)) {
		efree(MBREGEXG(search_str_val));
	}
	convert_to_string_ex(arg_str);
	MBREGEXG(search_str_val) = (unsigned char*)estrndup((*arg_str)->value.str.val, (*arg_str)->value.str.len);
	MBREGEXG(search_str_len) = (*arg_str)->value.str.len;
	MBREGEXG(search_pos) = 0;
	if (MBREGEXG(search_regs)) {
		mbre_free_registers(MBREGEXG(search_regs));
		efree(MBREGEXG(search_regs));
		MBREGEXG(search_regs) = (struct mbre_registers*)0;
	}

	RETURN_TRUE;
}
/* }}} */


/* {{{ proto array mbereg_search_getregs(void)
   Get matched substring of the last time */
PHP_FUNCTION(mbereg_search_getregs)
{
	int n, i, len, beg, end;
	unsigned char *str;
	MBREGEXLS_FETCH()

	if (MBREGEXG(search_regs) && MBREGEXG(search_str_val) &&
	    array_init(return_value) != FAILURE) {
		str = MBREGEXG(search_str_val);
		len = MBREGEXG(search_str_len);
		n = MBREGEXG(search_regs)->num_regs;
		for (i = 0; i < n; i++) {
			beg = MBREGEXG(search_regs)->beg[i];
			end = MBREGEXG(search_regs)->end[i];
			if (beg >= 0 && beg <= end && end <= len) {
				add_index_stringl(return_value, i, &str[beg], end - beg, 1);
			} else {
				add_index_bool(return_value, i, 0);
			}
		}
	} else {
		RETVAL_FALSE;
	}
}
/* }}} */


/* {{{ proto int mbereg_search_getpos(void)
   Get search start position */
PHP_FUNCTION(mbereg_search_getpos)
{
	MBREGEXLS_FETCH()

	RETVAL_LONG(MBREGEXG(search_pos));
}
/* }}} */


/* {{{ proto bool mbereg_search_setpos(int position)
   Set search start position */
PHP_FUNCTION(mbereg_search_setpos)
{
	pval **arg_pos;
	int n;
	MBREGEXLS_FETCH()

	if (ARG_COUNT(ht) != 1 || zend_get_parameters_ex(1, &arg_pos) == FAILURE) {
		WRONG_PARAM_COUNT;
	}
	convert_to_long_ex(arg_pos);
	n = (*arg_pos)->value.lval;
	if (n < 0) {
		php_error(E_WARNING, "position is minus value");
		MBREGEXG(search_pos) = 0;
		RETVAL_FALSE;
	} else if (n >= MBREGEXG(search_str_len)) {
		php_error(E_WARNING, "position not contained in string");
		MBREGEXG(search_pos) = MBREGEXG(search_str_len);
		RETVAL_FALSE;
	} else {
		MBREGEXG(search_pos) = n;
		RETVAL_TRUE;
	}
}
/* }}} */

#endif	/* HAVE_MBREGEX */
