/*
 * PS3 Media Server, for streaming any medias to your PS3.
 * Copyright (C) 2008  A.Brochard
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.pms.formats;

import java.util.ArrayList;
import java.util.StringTokenizer;

import net.pms.configuration.RendererConfiguration;
import net.pms.dlna.DLNAResource;
import net.pms.dlna.DLNAMediaInfo;
import net.pms.dlna.InputFile;
import net.pms.encoders.Player;
import net.pms.network.HTTPResource;
import net.pms.PMS;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class Format implements Cloneable {
	private static final Logger logger = LoggerFactory.getLogger(Format.class);

	public int getType() {
		return type;
	}

	public static final int ISO = 32;
	public static final int PLAYLIST = 16;
	public static final int UNKNOWN = 8;
	public static final int VIDEO = 4;
	public static final int AUDIO = 1;
	public static final int IMAGE = 2;

	protected String matchedId;

	public String getMatchedId() {
		return matchedId;
	}
	protected int type = UNKNOWN;
	protected Format secondaryFormat;

	public Format getSecondaryFormat() {
		return secondaryFormat;
	}

	public void setSecondaryFormat(Format secondaryFormat) {
		this.secondaryFormat = secondaryFormat;
	}

	public void setType(int type) {
		if (isUnknown()) {
			this.type = type;
		}
	}
	public void setTypeForce(int type) {  //regzam
		this.type = type;
	}

	public abstract String[] getId();
	public abstract boolean ps3compatible();
	public abstract boolean transcodable();
	public abstract ArrayList<Class<? extends Player>> getProfiles();

	public String mimeType() {
		return new HTTPResource().getDefaultMimeType(type);
	}

	public boolean match(String filename) {
		boolean match = false;
		if (filename == null) {
			return match;
		}
		filename = filename.toLowerCase();
		for (String singleid : getId()) {
			String id = singleid.toLowerCase();
			match = filename.endsWith("." + id) || filename.startsWith(id + "://");
			if (match) {
				matchedId = singleid;
				return true;
			}
			else {	//regzamod, for special suffix
				match= filename.endsWith("." + id+ "_d")
					|| filename.endsWith("." + id+ "_r")
					|| filename.endsWith("." + id+ "_t");
				if (match) {
					matchedId = singleid;
					return true;
				}
			}
		}
		return match;
	}

	public boolean isVideo() {
		return (type & VIDEO) == VIDEO;
	}

	public boolean isAudio() {
		return (type & AUDIO) == AUDIO;
	}

	public boolean isImage() {
		return (type & IMAGE) == IMAGE;
	}

	public boolean isUnknown() {
		return (type & UNKNOWN) == UNKNOWN;
	}

	@Override
	protected Object clone() {
		Object o = null;
		try {
			o = super.clone();
		} catch (CloneNotSupportedException e) {
			logger.error(null, e);
		}
		return o;
	}

	public Format duplicate() {
		return (Format) this.clone();
	}

	// method which allow us the fine tuning of parsing with different formats in the future
	public void parse(DLNAMediaInfo media, InputFile file, int type) {
		parse(media, file, type, null,null);
	}

	// 10/02/03 now this is useful :p
	public void parse(DLNAMediaInfo media, InputFile file, int type, RendererConfiguration renderer,DLNAResource dlna) {
		//if (renderer != null && renderer.isMediaParserV2()) { //origin
		if(PMS.rz_debug>1) PMS.dbg("Format.parse:renderer="+renderer);
			
		if(renderer != null && renderer.getRZ_MediaParseType()==1) {
			if(PMS.rz_debug>1) PMS.dbg("Format.parse: exec media.parse()");
			media.parse(file, this, type, false,dlna);
		}
		else if (renderer != null && (renderer.isMediaParserV2() || renderer.getRZ_MediaParseType()==2)) { //regzam
			if(PMS.rz_debug>1) PMS.dbg("Format.parse: exec FormatConfiguration.parse()");
			renderer.getFormatConfiguration().parse(media, file, this, type, dlna);
		} else {
			if(PMS.rz_debug>1) PMS.dbg("Format.parse: exec media.parse()");
			media.parse(file, this, type, false,dlna);
		}
		if(PMS.rz_debug>1) logger.trace("Parsing results: " + file + " / " + media);
	}

	public boolean skip(String extensions, String another_set_of_extensions) {
		if (extensions != null && extensions.length() > 0) {
			StringTokenizer st = new StringTokenizer(extensions, ",");
			while (st.hasMoreTokens()) {
				String id = st.nextToken().toLowerCase();
				if (matchedId != null && matchedId.toLowerCase().equals(id)) {
					return true;
				}
			}
		}

		if (another_set_of_extensions != null && another_set_of_extensions.length() > 0) {
			StringTokenizer st = new StringTokenizer(another_set_of_extensions, ",");
			while (st.hasMoreTokens()) {
				String id = st.nextToken().toLowerCase();
				if (matchedId != null && matchedId.toLowerCase().equals(id)) {
					return true;
				}
			}
		}
		return false;
	}
}
