/*
 * PS3 Media Server, for streaming any medias to your PS3.
 * Copyright (C) 2008  A.Brochard
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.pms.encoders;

import java.awt.Font;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;

import javax.swing.JCheckBox;
import javax.swing.JComponent;

import net.pms.configuration.PmsConfiguration;
import net.pms.configuration.RendererConfiguration;
import net.pms.dlna.DLNAMediaAudio;
import net.pms.dlna.DLNAMediaInfo;
import net.pms.dlna.DLNAResource;
import net.pms.dlna.InputFile;
import net.pms.dlna.rz_ScriptExecutor;
import net.pms.formats.Format;
import net.pms.io.OutputParams;
import net.pms.io.PipeIPCProcess;
import net.pms.io.PipeProcess;
import net.pms.io.ProcessWrapper;
import net.pms.io.ProcessWrapperImpl;
import net.pms.io.StreamModifier;
import net.pms.Messages;
import net.pms.PMS;
import net.pms.util.CodecUtil;
import net.pms.util.PMSUtil;	// regzamod
import net.pms.encoders.MEncoderVideo;

import com.jgoodies.forms.builder.PanelBuilder;
import com.jgoodies.forms.factories.Borders;
import com.jgoodies.forms.layout.CellConstraints;
import com.jgoodies.forms.layout.FormLayout;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


public class TSMuxerVideo extends Player {
	private static final Logger logger = LoggerFactory.getLogger(TSMuxerVideo.class);
	public static final String ID = "tsmuxer";
	private PmsConfiguration configuration;

	public TSMuxerVideo(PmsConfiguration configuration) {
		this.configuration = configuration;
	}

	public boolean excludeFormat(Format extension) {
		String m = extension.getMatchedId();
		//return m != null && !m.equals("mp4") && !m.equals("mkv") && !m.equals("ts") && !m.equals("tp") && !m.equals("m2ts") && !m.equals("m2t") && !m.equals("mpg") && !m.equals("evo") && !m.equals("mpeg")
		//	&& !m.equals("vob") && !m.equals("m2v") && !m.equals("mts") && !m.equals("mov");
		return m != null && !m.equals("mp4") && !m.equals("mkv") && !m.equals("ts") && !m.equals("tp") && !m.equals("m2ts") && !m.equals("m2t") && !m.equals("mpg") && !m.equals("evo") && !m.equals("mpeg")
			&& !m.equals("vob") && !m.equals("m2v") && !m.equals("mts") && !m.equals("mov")&& !m.equals("flv");
	}

	@Override
	public int purpose() {
		return VIDEO_SIMPLEFILE_PLAYER;
	}

	@Override
	public String id() {
		return ID;
	}

	@Override
	public boolean isTimeSeekable() {
		return true;
	}

	@Override
	public String[] args() {
		return null;
	}

	@Override
	public String executable() {
		return configuration.getTsmuxerPath();
	}

	@Override
	public ProcessWrapper launchTranscode(
		String fileName,
		DLNAResource dlna,
		DLNAMediaInfo media,
		OutputParams params) throws IOException {
		
		if(media==null) { //regzamod
			logger.error("TSMuxerVideo.launchTranscode: media=null, return noop");
			return null;
		}

		setAudioAndSubs(fileName, dlna, media, params, configuration);
		boolean singleMediaAudio = media != null && media.getAudioCodes().size() <= 1; //regzamod, moved to top

		PipeIPCProcess ffVideoPipe = null;
		ProcessWrapperImpl ffVideo = null;

		PipeIPCProcess ffAudioPipe[] = null;
		ProcessWrapperImpl ffAudio[] = null;

		String fps = media.getValidFps(false);
		String videoType = "V_MPEG4/ISO/AVC";
		if (media != null && media.getCodecV() != null && media.getCodecV().equals("mpeg2video")) {
			videoType = "V_MPEG-2";
		}
		dlna.setEncMode(PMS.ENC_REMUX);
			
		//--- regzamod start
		int	remux_type=333;
		boolean do_all_by_ffmpeg=false;		// do all(demux&mux) by ffmpeg 
											// --> best A/V sync, so important for part_play
		boolean do_all_by_tsmuxer=false;	// do all(demux&mux) by tsmuxer (accurate but slow) 
		boolean extract_by_ffmpeg=false; 	// test
		boolean extract_by_ffmpeg_v=false; 	// test
		boolean extract_by_ffmpeg_a=false; 	// test
		boolean cut_by_muxer=false; 		//seek(partial_cut_out) by tsmuxer
		
		double timeseek_v=0;	//for extractor of video
		double timeseek_a=0;	//for extractor of audio
		double audio_delay=0;
		double timeseek_t=0;	//for muxer (valid when cut_by_muxer is true)
		double endpos_v=0;		//for extractor of video
		double endpos_a=0;		//for extractor of audio
		double endpos_t=0;		//for muxer (valid when cut_by_muxer is true)
		double sound_delay=0;
		double sound_delay_mc=0;
		double sound_delay_ts=0;	//for muxer
		int sound_volume= -1;		
		int sound_sync= -1;		
		double rewind=0;
		String track_v="1";
		String track_a="2";
		String format_a_ts="A_AC3";	//for tsmuxer, default
		int numAudioTracks=1;  //equal to realAudioTracks, only when configuration.isMuxAllAudioTracks() is true
		int realAudioTracks=1;
			
		String tsmuxerout_fmt="tsmuxerout.ts";
		String ffmpeg_addopts="";
		rz_ScriptExecutor.medAddOpts spec_addopts=null;
		
		String ffmpeg_path=configuration.getFfmpegPath();
		
		//-----------------------------------------------------------
		// process additional options from pxf/pxm/renderer.conf
		//-----------------------------------------------------------
		//if(dlna.getForcedRemuxProg()!=null) ffmpeg_path=dlna.getForcedRemuxProg();
		//if(dlna.getForcedFFmpegPath()!=null) {
		//	ffmpeg_path=dlna.getForcedFFmpegPath();
		//}
		
		//String script=params.sess.scex.getRz_TransOptScript(2); //get only normal scripts
		if(PMS.rz_is_rend_script_enabled && params.sess.scex!=null) { // from script
			spec_addopts=new rz_ScriptExecutor.medAddOpts();
			int remux_format=params.mediaRenderer.getRZ_RemuxVFormat();
			//---- override by rz_MetaTransFormat in pxm/pxf 
			if(dlna.getForcedRemuxFormat()>=0) {
				remux_format=dlna.getForcedRemuxFormat();
			}
			spec_addopts.player=this.toString();
			spec_addopts.trans_type=rz_ScriptExecutor.trans_type_int2string(PMS.ENC_REMUX);
			spec_addopts.remux_format=rz_ScriptExecutor.trans_format_int2string(remux_format);

			//execScripts(type,when,taget,dlna,..)
			// int type;	//=1:load,=2:exec
			// int when;	//=1:list,=2:play,  =3:list|play,  =4:oneshot
			// int target;	//=1:file,=2:folder,=3:file|folder,=4:other

			params.sess.scex.execScripts(2,2,1,dlna,dlna.getSrcPath(),params,spec_addopts);
			//---- apply ffmpeg opts(dlna.rz_MetaFFmpegOpt)
			if(dlna.rz_MetaFFmpegOpt[1]!=null) {
				ffmpeg_addopts+=dlna.rz_MetaFFmpegOpt[1];
			}
		}
		
		//PMS.dbg("TSMuxerVideo: getForcedRemuxProg="+dlna.getForcedRemuxProg());
			
		if(dlna.getForcedRemuxProg()!=null) ffmpeg_path=dlna.getForcedRemuxProg();
		if(dlna.rz_MetaFFmpegOpt[0]!=null) {  //from pxm
			ffmpeg_addopts+=dlna.rz_MetaFFmpegOpt[0];
		}
		if(PMS.rz_debug>1) PMS.dbg("TsMuxerVideo: ffmpeg_addopts="+ffmpeg_addopts+",spec_addopts="+spec_addopts);

		//-----------------------------------------------------------
		// Judge remux_type
		//-----------------------------------------------------------
		if (media != null && media.getAudioCodes() != null && media.getAudioCodes().size() > 1) {
			realAudioTracks=media.getAudioCodes().size();
			if(configuration.isMuxAllAudioTracks()) {  //default is false
				numAudioTracks = realAudioTracks;
			}
		}
		
		//do_all_by_ffmpeg=params.mediaRenderer.isRZ_RemuxExtractByFfmpeg();
		//override by Individual Metafile params
		remux_type=params.mediaRenderer.getRZ_RemuxType();
		
		if(dlna.getForcedRemuxProc()>0) {
			remux_type=dlna.getForcedRemuxProc();	//overrride
		}
		if(dlna.rz_MetaSoundVolume>=0) {
			sound_volume=dlna.rz_MetaSoundVolume;
			if(sound_volume>1000) {
				logger.warn("sound_vlume="+sound_volume+"(%) :Danger! too BigSound --> Cutoff to 1000% (x10:+20db) for safety");
				sound_volume=1000;
			}
		}
		if(dlna.rz_MetaSoundSync>=0) {
			sound_sync=dlna.rz_MetaSoundSync;
		}
		if(remux_type==113) {		// MMT (Original)
			//PMS.dbg("Remux_type="+remux_type+", MMT (Extract all by mencoder (Old Default)"); 
			//defaults
		}
		else if(remux_type==213) {	// FMT
			//PMS.dbg("Remux_type="+remux_type+", FMT (Extract video by ffmpeg"); 
			extract_by_ffmpeg_v=true;
		}
		else if(remux_type==223) {	// FFT
			//PMS.dbg("Remux_type="+remux_type+", FFT (Extract all by ffmpeg"); 
			extract_by_ffmpeg_v=true;
			extract_by_ffmpeg_a=true;
		}
		else if(remux_type==222) {	// FFF
			//PMS.dbg("Remux_type="+remux_type+", FFF (Do all by ffmpeg (New Default)"); 
			do_all_by_ffmpeg=true;
			extract_by_ffmpeg_v=true;
			extract_by_ffmpeg_a=true;
		}
		else if(remux_type==333) {	// TTT
			//PMS.dbg("Remux_type="+remux_type+", TTT (Do all by Tsmuxer)"); 
			do_all_by_tsmuxer=true;
			cut_by_muxer=true; 
		}
		
		if(do_all_by_ffmpeg||do_all_by_tsmuxer||extract_by_ffmpeg_a) {
			if(numAudioTracks>1) {
				logger.warn("remux_type="+remux_type+": can't handle multi-audio tracks.");
				logger.warn("--> use other remux_type, if you want handle all tracks");
			}
		}
		//PMS.dbg("getRZ_RemuxAudioDelay="+params.mediaRenderer.getRZ_RemuxAudioDelay());
		if(params.mediaRenderer.getRZ_RemuxAudioDelay()!=0) {	
			//---- constant delay both on start/restart, per renderer, for any remux type 
			audio_delay += params.mediaRenderer.getRZ_RemuxAudioDelay();
		}
		
		//PMS.dbg("--> audio_delay="+audio_delay);
		//PMS.dbg("dlna.rz_SoundDelay="+dlna.rz_SoundDelay);
		if(dlna.rz_SoundDelay[0]!=0) {
			//---- constant delay both on start/restart, per file, for any remux type 
			audio_delay +=dlna.rz_SoundDelay[0];
		}
		if(dlna.rz_SoundDelay[0]!=0) {
			//---- constant delay both on start/restart, per file, for any remux type 
			audio_delay +=dlna.rz_SoundDelay[0];
		}
		//PMS.dbg("--> audio_delay="+audio_delay);
		
		if(do_all_by_ffmpeg) {
			//-----------------------------------------------------------
			// Process type A: New way, Process all by ffmpeg 
			// (from audio/video extract to remuxed output)
			//-----------------------------------------------------------
			//---- judge Start pos (seek time) ---------------
			if(params.timeseek>0) {
				// tuning restart rewind by renderer.conf
				rewind=params.mediaRenderer.getRZ_RemuxRestartRewind();
				timeseek_v=params.timeseek-rewind;
				timeseek_a=params.timeseek-rewind;
				
				// tuning sound delay by renderer.conf
				if(media.isMpegTS()) {	
					// no need to tuning sound delay ?
				}
				else {
					if(dlna.rz_PartPlaySttp != params.timeseek) {
						//NOT from top of the part
						sound_delay=params.mediaRenderer.getRZ_RemuxAudioDelayRestartFm();
						audio_delay += sound_delay;
					}
				}
			}
			else {
				if(params.timeseek<0) {
					timeseek_v=params.timeseek;
					timeseek_a=params.timeseek;
				}
				float start_skip=params.mediaRenderer.getRZ_RemuxStartSkip();
				if(start_skip!=0) {
					timeseek_v=params.timeseek+start_skip;
					timeseek_a=params.timeseek+start_skip;
				}
				if(PMS.rz_debug>2) PMS.dbg("TsMuxerVideo: start_skip="+start_skip+", timeseek="+timeseek_v);
			}
			//---- judge end pos(duration) ------------------
			if(dlna.rz_isPartPlay) {
				//endpos_v=dlna.getSplitRange().getEnd();
				endpos_v=dlna.rz_PartPlayEndp - timeseek_v;
				endpos_a=endpos_v;
			}
		}
		else {
			//-----------------------------------------------------------
			// Process type B:  Original way, Tsmuxer & extracter(ffmpeg/mencoder)
			//-----------------------------------------------------------
			// set defaults of track_No. only examples: really, many different values used 
			if(media.isMpegTS()) {	//for container ts,m2ts
				if(media.getCodecV().startsWith("h264")) {
					track_v="4113";	//0x1011
					track_a="4352";	//0x1100
				}
				//else if(fileName.endsWith(".ts")) {
				if(media.getCodecV().startsWith("mpeg")) {
					track_v="68";	//not constant: 0x44 or 0x111 or ...
					track_a="69";	//not constant: 0x45 or 0x112 or ...
				}
			}
			else {	//for container mp4,flv,mkv ?
				track_v="1";
				track_a="2";
			}
			//override individually, if according Metafile params exists
			if(dlna.rz_MetaChannel_v>0) track_v=""+dlna.rz_MetaChannel_v;
			if(dlna.rz_MetaChannel_a>0) track_a=""+dlna.rz_MetaChannel_a;

			//-----------------------------------------------
			// for partial play
			//-----------------------------------------------
			if(dlna.rz_isPartPlay) {
				if(cut_by_muxer) {
					//---- partial_cut_out by muxer (TsMuxer)
					if(params.timeseek>0) {
						// tuning param for video delay on pause/restart/skip on REMUX
						rewind=params.mediaRenderer.getRZ_RemuxRestartRewind();
						//timeseek_v=params.timeseek-rewind;
						//timeseek_a=params.timeseek-rewind;
						timeseek_t=params.timeseek-rewind;
					}
					//endpos_v=dlna.getSplitRange().getEnd();
					//endpos_v=dlna.rz_PartPlayDuration;
					endpos_t=dlna.rz_PartPlayEndp - timeseek_t;
					endpos_v=endpos_t; // need??
					endpos_a=endpos_t;
				}
				else {
					//---- partial_cut_out by extractor (ffmpeg/mencoder)
					//----> mencoder's audio extraction cause terrible A/V async!!
					if(params.timeseek>0) {
						// tuning param for video delay on pause/restart/skip on REMUX
						rewind=params.mediaRenderer.getRZ_RemuxRestartRewind();
						timeseek_v=params.timeseek-rewind;
						timeseek_a=params.timeseek-rewind;
						
						if(extract_by_ffmpeg_a) {  //audio extract by ffmpeg
							sound_delay=params.mediaRenderer.getRZ_RemuxAudioDelayRestartFm();
						}
						else {	//audio extract by mencoder
							sound_delay=params.mediaRenderer.getRZ_RemuxAudioDelayRestart();
						}
						audio_delay += sound_delay;
					}
					//endpos_v=dlna.getSplitRange().getEnd();
					//endpos_v=dlna.rz_PartPlayDuration;
					endpos_v=dlna.rz_PartPlayEndp - timeseek_v;
					endpos_a=endpos_v;
					
					//sound_delay from LinkMetafile params
					double mag=dlna.rz_SeekAudioMagnify;
					if(mag!=0) timeseek_a *=mag;	
					double sd=dlna.rz_SeekAudioDelay;
					if(sd!=0) {
						audio_delay += sd;
					}
				}
			}
			//-----------------------------------------------	
			// for normal play
			//-----------------------------------------------	
			else {	
				if(params.timeseek>0) {
					// tuning param for video delay on pause/restart/skip on REMUX
					rewind=params.mediaRenderer.getRZ_RemuxRestartRewind();
					timeseek_v=params.timeseek-rewind;
					timeseek_a=params.timeseek-rewind;
					timeseek_t=params.timeseek-rewind;
					if(!do_all_by_tsmuxer) {
						sound_delay=params.mediaRenderer.getRZ_RemuxAudioDelayRestart();
						audio_delay += sound_delay;
					}
				}
			}
		}
		
		//Remux container format & specific options
		int remux_format=params.mediaRenderer.getRZ_RemuxVFormat();
		//---- override by rz_MetaTransFormat in pxm/pxf 
		//if(dlna.rz_MetaRemuxFormat>=0) {
		if(dlna.getForcedRemuxFormat()>=0) {
			remux_format=dlna.getForcedRemuxFormat();
		}
		if(remux_format==PMS.TRS_M2TS) {
			tsmuxerout_fmt="tsmuxerout.m2ts";
		}
		else {
			ffmpeg_addopts =" -copytb 1 "+ffmpeg_addopts;
		}
		
		if(do_all_by_tsmuxer) {
			//exec audio_delay by tsMuxeR
			sound_delay_ts = audio_delay;   // negative effect
		}
		else {
			if(do_all_by_ffmpeg || params.mediaRenderer.isRZ_RemuxAudioDelayByTimeSeek()) {
				//exec audio_delay by ffmpeg/mencoder's timeseek param
				if(extract_by_ffmpeg_a) {
					timeseek_a -= audio_delay;
				}
				else { //extract_by_mencoder --> can't handle timeseek <0 (audio_delay >0) 
					sound_delay_ts = audio_delay;   // negative effect
				}
			}
			else {
				//exec audio_delay by tsMuxeR
				sound_delay_ts = audio_delay;   // negative effect
			}
		}

		//---- dump basic play params 
		if(PMS.rz_debug>1) {
			PMS.dbg("========= TsMuxerVideo: "+ dlna.getName()+" ===================");
			PMS.dbg("remux_type="+remux_type
				+", mediaRenderer.getRZ_RemuxType="+params.mediaRenderer.getRZ_RemuxType()
				+", ForcedRemuxProc="+dlna.getForcedRemuxProc());
				
			PMS.dbg("do_all_by_ffmpeg="+do_all_by_ffmpeg
				+", do_all_by_tsmuxer="+do_all_by_tsmuxer
				+", cut_by_muxer=="+cut_by_muxer);
				
			PMS.dbg("extract_by_ffmpeg_v="+extract_by_ffmpeg_v
				+", extract_by_ffmpeg_a="+extract_by_ffmpeg_a);
			
			PMS.dbg("rz_isPartPlay="+dlna.rz_isPartPlay
				+", rz_Bodyfile="+dlna.rz_Bodyfile);
			
			PMS.dbg("rz_PartPlaySttp="+dlna.rz_PartPlaySttp
				+", rz_PartPlayEndp="+dlna.rz_PartPlayEndp
				+", rz_SoundDelay="+dlna.rz_SoundDelay
				+", rz_MetaAspect="+dlna.rz_MetaAspect
				+", rz_MetaSoundVolume="+dlna.rz_MetaSoundVolume);
			
			PMS.dbg("rz_MetaChannel_v="+dlna.rz_MetaChannel_v
				+", rz_MetaChannel_a="+dlna.rz_MetaChannel_a);
			
			PMS.dbg("rz_SeekAudioDelay="+dlna.rz_SeekAudioDelay
				+", RZ_RemuxAudioDelayByTimeSeek="+params.mediaRenderer.isRZ_RemuxAudioDelayByTimeSeek()
				+", RZ_RemuxAudioDelay="+params.mediaRenderer.getRZ_RemuxAudioDelay()
				+", rz_SeekAudioMagnify="+dlna.rz_SeekAudioMagnify);
			
			PMS.dbg("timeseek_v="+timeseek_v
				+", timeseek_a="+timeseek_a
				+", audio_delay="+audio_delay
				+", sound_delay_ts="+sound_delay_ts+" (delay by TsMuxer)"
				+", timeseek_t="+timeseek_t
				+", endpos_v="+endpos_v
				+", endpos_a="+endpos_a
				+", endpos_t="+endpos_t);
			PMS.dbg("(Mencoder's foolish 'endpos' means 'Duration', not 'Position')");
			PMS.dbg("numAudioTracks="+numAudioTracks);
			
			PMS.dbg("=======================");
		}
		

		//-------------------------------------------------------------------------------
		// TsMuxerAudio body
		//-------------------------------------------------------------------------------
		if (this instanceof TsMuxerAudio && media.getFirstAudioTrack() != null) {
			ffVideoPipe = new PipeIPCProcess(System.currentTimeMillis() + "fakevideo", System.currentTimeMillis() + "videoout", false, true);
			String ffmpegLPCMextract[] = new String[]{ffmpeg_path, "-t", "" + params.timeend, "-loop_input", "-i", "/resources/images/fake.jpg", "-qcomp", "0.6", "-qmin", "10", "-qmax", "51", "-qdiff", "4", "-me_range", "4", "-f", "h264", "-vcodec", "libx264", "-an", "-y", ffVideoPipe.getInputPipe()};
			//videoType = "V_MPEG-2";
			videoType = "V_MPEG4/ISO/AVC";
			if (params.timeend < 1) {
				ffmpegLPCMextract[1] = "-title";
				ffmpegLPCMextract[2] = "dummy";
			}

			OutputParams ffparams = new OutputParams(PMS.getConfiguration());
			ffparams.maxBufferSize = 1;
			ffVideo = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams);

			if (fileName.toLowerCase().endsWith(".flac") && media != null && media.getFirstAudioTrack().getBitsperSample() >= 24 && media.getFirstAudioTrack().getSampleRate() % 48000 == 0) {
				ffAudioPipe = new PipeIPCProcess[1];
				ffAudioPipe[0] = new PipeIPCProcess(System.currentTimeMillis() + "flacaudio", System.currentTimeMillis() + "audioout", false, true);
				String flacCmd[] = new String[]{configuration.getFlacPath(), "--output-name=" + ffAudioPipe[0].getInputPipe(), "-d", "-f", "-F", fileName};

				ffparams = new OutputParams(PMS.getConfiguration());
				ffparams.maxBufferSize = 1;
				ffAudio = new ProcessWrapperImpl[1];
				ffAudio[0] = new ProcessWrapperImpl(flacCmd, ffparams);
			} else {
				ffAudioPipe = new PipeIPCProcess[1];
				ffAudioPipe[0] = new PipeIPCProcess(System.currentTimeMillis() + "mlpaudio", System.currentTimeMillis() + "audioout", false, true);
				String depth = "pcm_s16le";
				String rate = "48000";
				if (media != null && media.getFirstAudioTrack().getBitsperSample() >= 24) {
					depth = "pcm_s24le";
				}
				if (media != null && media.getFirstAudioTrack().getSampleRate() > 48000) {
					rate = "" + media.getFirstAudioTrack().getSampleRate();
				}
				String flacCmd[] = new String[]{ffmpeg_path, "-ar", rate, "-i", fileName, "-f", "wav", "-acodec", depth, "-y", ffAudioPipe[0].getInputPipe()};

				ffparams = new OutputParams(PMS.getConfiguration());
				ffparams.maxBufferSize = 1;
				ffAudio = new ProcessWrapperImpl[1];
				ffAudio[0] = new ProcessWrapperImpl(flacCmd, ffparams);
			}
		} 
		//-------------------------------------------------------------------------------
		// TsMuxerVideo body
		//-------------------------------------------------------------------------------
		//---- tuning params for start playing 
		//params.manageFastStart(); 
		params.manageStart(params.mediaRenderer.getRZ_RemuxStartWait()); 
		
		//defaults
		//params.waitbeforestart = configuration.getVideoTranscodeStartDelay()
		//params.minBufferSize   = configuration.getMinMemoryBufferSize();
		//params.secondread_minsize = 100000; 
		
		//-------------------------------------------------------------------------------
		// Type A : New Way --- do all(Demux/Mux) by ffmpeg
		//-------------------------------------------------------------------------------
		if(do_all_by_ffmpeg) {
			if(PMS.rz_debug>1) {
				PMS.dbg("======== Wellcome to New Doom of TsMuxerVideo ==========="); 
				PMS.dbg("---> All Process will be done by FFMPEG without TsMuxer!!");
				PMS.dbg("---> Currentry, support only Single_audio_track");
			}
			
			//-------------------------------------------------------------------------------
			// prepair ffmpeg args 
			//-------------------------------------------------------------------------------

			PipeProcess tsPipe = new PipeProcess(System.currentTimeMillis() + tsmuxerout_fmt);
			
			//int bitrate=CodecUtil.getAC3Bitrate(configuration, params.aid);    //too high for ffmpeg?
			int bitrate=CodecUtil.getFFmpegAudioBps(configuration, params.aid);  //some lower bps
			
			if(PMS.rz_debug>2) PMS.dbg("TsMuxerVideo : ReEncode audio_bps(kbps)="+bitrate);
			String[] ffmpegLPCMextract = new String[]{
				ffmpeg_path,			//0
				"-ss", "0",				//1,2	--- for video stream
				"-itsoffset","0",		//3,4
				"-i", 					//5
				fileName,				//6
				
				"-ss", "0",				//7,8	--- for audio stream
				"-itsoffset","0",		//9,10
				"-i", 					//11
				fileName,				//12
				
				"-y",					//13	reserved for -t
				"-y",					//14 	reserved for -t
				"-y",					//15 	reserved for no message
				"-vcodec", "copy",		//16,17
				"-y","-y",				//18,19 reserved for -vbsf
				"-acodec", "ac3",		//20,21 ac3: default&failsafe
				"-y","-y",				//22,23 reserved for -absf 
				"-async","10",			//24,25 must for audio sync? (really no need)
				"-copyts",				//26    must for audio sync? (really no need)
				"-y","-y",				//27,28 reserved for -vol
				"-map", "0:v",			//29,30 out0 <--- video in input_file0
				"-map", "1:a",			//31,32 out1 <--- audio in input_file1("1:a" means all audio)
				"-loglevel",(PMS.rz_debug>1?"warning":"fatal"),  //33,34
				"-ab", bitrate+"k", 	//35,36
				"-ar", "48000",			//37,38
				tsPipe.getInputPipe(),	//39
			};
			if (params.stdin != null) {
				// make input-0 to stdin 
				ffmpegLPCMextract[6] = "-";
				// cancel input-1: two sunction doesn't work well
				ffmpegLPCMextract[7] = "-y";
				ffmpegLPCMextract[8] = "-y";
				ffmpegLPCMextract[9] = "-y";
				ffmpegLPCMextract[10] = "-y";
				ffmpegLPCMextract[11] = "-y";
				ffmpegLPCMextract[12] = "-y";
				//change audio to input-0
				ffmpegLPCMextract[32] = "0:a";
				if(timeseek_v<0) timeseek_v=0;  //couse error when pipe input
				if(timeseek_a<0) timeseek_a=0;  //couse error when pipe input
			}
			if(timeseek_v!=0) {
				ffmpegLPCMextract[2]="" + timeseek_v;
			}
			if(timeseek_a!=0 && params.stdin == null) {
				ffmpegLPCMextract[8]="" + timeseek_a;
			}
			if(endpos_v!=0) {
				ffmpegLPCMextract[13]="-t";
				ffmpegLPCMextract[14]=""+endpos_v;
			}
			
			//---- container -------------------
			//if(!media.isMpegTS() !media.isMpegTS()) {	//really? :yes, at least for mp4,flv(H.264)
			//if(media.getCodecV().startsWith("h264")) {
			if(!media.isMpegTS() && media.getCodecV()!=null && media.getCodecV().startsWith("h264")) {
				ffmpegLPCMextract[18]="-vbsf";
				ffmpegLPCMextract[19]="h264_mp4toannexb";
			}
			
			//---- video codec-------------------
			if(media.getCodecV()!=null && media.getCodecV().startsWith("h264")) {
				//"-f", "h264", 
			}
			
			//---- audio codec -------------------
			String acodec_str="unknown";
			DLNAMediaAudio audio=params.aid;
			int aid_chg=0;
			int aid= -1;
			if(dlna.rz_MetaChannel_a_ffmpeg>=0) {	//called from #TRANSCODE#
				//aid=dlna.rz_MetaChannel_a_ffmpeg -1;
				aid=dlna.rz_MetaChannel_a_ffmpeg;
			}
			if(dlna.rz_MetaChannel_a>=0) {	// directed by PlayMetafile
				//aid=dlna.rz_MetaChannel_a -1;
				aid=dlna.rz_MetaChannel_a;
			}
			if(aid>=0) {
				if(aid>=media.getAudioCodes().size()) {  //over max
					logger.error("TsMuxerVideo: rz_MetaChannel_a="+dlna.rz_MetaChannel_a
						+" over AudioTracks="+realAudioTracks);
					// no change
				}
				else {
					 audio= media.getAudioCodes().get(aid);
				}
				aid_chg=1;  //audio channel explicitly directed --> mux only this channel
			}
				
			//PMS.dbg("TsMUxerVideo: audio="+audio+", aid_chg="+aid_chg
			//	+", numAudioTracks"+numAudioTracks+",realAudioTracks="+realAudioTracks);
				
			//if(true) {  // all channel will be processed same as target aid --- danger
			//if(audio!=null && (aid_chg>0 || numAudioTracks==1)) {  //audio channel explicitly directed, or single audio
			if(audio!=null && (aid_chg>0 || realAudioTracks==1)) {  //audio channel explicitly directed, or single audio
				if(audio.isAC3() && configuration.isRemuxAC3()) {
					//PMS.dbg("TsMUxerVideo: audio.isAC3() --> encode acodec=copy");
					ffmpegLPCMextract[21]="copy";
				}
				else if(audio.getAudioCodec().equals("aac") ) {
					if(media.isMpegTS() && realAudioTracks==1 && aid==0) {  
						//sometime cause ffmpeg error [aac bitstream not in adts format and extradata missing]
						//PMS.dbg("TsMUxerVideo: media.isMpegTS() && realAudioTracks==1 --> encode acodec=copy");
						ffmpegLPCMextract[21]="copy";
					}
					else {
						//---- REGZA can't play aac extracted by ffmpeg from mpeg4
						//---->ffmpeg changes AAC Level Higher than Realy is !!
						//really? :yes, at least for mp4,flv(H.264)
						ffmpegLPCMextract[21]="ac3";	// force encode
					}
				}
				else if(audio.isDTS()) {
					if(params.mediaRenderer.isMuxDTSToMpeg()) {
						ffmpegLPCMextract[21]="copy";
					}
				}
				else if(audio.isPCM()) {
					if(params.mediaRenderer.isMuxLPCMToMpeg()) {
						ffmpegLPCMextract[21]="copy";
					}
				}
				else if(audio.isTrueHD()) {
					//noop: i.e. conv to ac3
				}
			}
			else {
				// otherwise, all will be -acodec ac3
			}
			//PMS.dbg("TsMUxerVideo: finally --> encode acodec="+ffmpegLPCMextract[21]);
			if(audio!=null) {
				//---- sound_sync value
				if(sound_sync>=0) {
					// -async samples_per_second' 
					// Audio sync method. "Stretches/squeezes" the audio stream to match the timestamps, 
					// the parameter is the maximum samples per second by which the audio is changed. 
					// -async 1 is a special case where only the start of the audio stream is corrected 
					// without any later correction. 

					//PMS.dbg("TsMuxerVideo: fource sound_sync(ffmpeg's -async)="+sound_sync);
					ffmpegLPCMextract[25]=""+ sound_sync;  // -async value
				}
				//---- sound_volume
				if(sound_volume>=0) {
					//force volume (normal=256) 128=-6dB(=x0.5)A512=+6dB(=x2.0),  -1:not setted
					//PMS.dbg("TsMuxerVideo: fource sound_volume="+sound_volume+" (%)");
					//PMS.dbg("---> and, force re-encode for volume tuning");
					ffmpegLPCMextract[21]="ac3";	//need force encode
					ffmpegLPCMextract[27]="-vol";
					ffmpegLPCMextract[28]=""+ (int)(((float)sound_volume)/100*256);
				}
				if(aid_chg>0) {
					//ffmpegLPCMextract[17]="ac3";	// force encode, minor aac in ts will cause error on copy
					if (params.stdin != null) {
						ffmpegLPCMextract[32]="0:a:"+(aid);	//default: "1:a" == all audio of input2
					}
					else {
						ffmpegLPCMextract[32]="1:a:"+(aid);	//default: "1:a" == all audio of input2
					}
					//PMS.dbg("Change Audio trackNo.="+ffmpegLPCMextract[28]);
				}
			}
			else {
				ffmpegLPCMextract[31] = "-y";
				ffmpegLPCMextract[32] = "-y";
			}
			
			//track
			if(dlna.rz_MetaChannel_v>=0) {
				//ffmpegLPCMextract[25]="-map"+dlna.rz_MetaChannel_v;	//default: "0:0" == trackNo.0 of input1
				//ffmpegLPCMextract[30]="0:"+dlna.rz_MetaChannel_v;	//default: "0:0" == trackNo.0 of input1
				ffmpegLPCMextract[30]="0:v:"+dlna.rz_MetaChannel_v;	//default: "0:0" == trackNo.0 of input1
				//PMS.dbg("Change Video trackNo.="+ffmpegLPCMextract[26]);
			}
			
			//---- apply addopts
			Float adelay_add=null;
			Float adelay_rep=null;
			if(ffmpeg_addopts!=null) {
				String[] sArgs=ffmpeg_addopts.split("\\s");
				if(sArgs.length>0) {
					int	length_org=ffmpegLPCMextract.length;
					String last_para_org=ffmpegLPCMextract[length_org-1];	//memory oldlast param(== outpot file(pipe))
					ffmpegLPCMextract = Arrays.copyOf(ffmpegLPCMextract, ffmpegLPCMextract.length + sArgs.length);
					for(int i=0;i<sArgs.length; i++) {
						ffmpegLPCMextract[length_org-1+i]=sArgs[i];
						if(sArgs[i].equals("-adelay_add")) {
							adelay_add=Float.parseFloat(sArgs[i+1]);
						}
						else if(sArgs[i].equals("-adelay_rep")){
							adelay_rep=Float.parseFloat(sArgs[i+1]);
						}
					}
					ffmpegLPCMextract[ffmpegLPCMextract.length-1]=last_para_org;	//recover newlast param(== outpot file(pipe))
				}
			}
			//---- process special opts (adelay_add/adelay_rep) too late?
			if(params.stdin != null) {
				if(adelay_add!=null) ffmpegLPCMextract[8]="" + (timeseek_a - adelay_add);
				if(adelay_rep!=null) ffmpegLPCMextract[8]="" + (- adelay_rep);
			}
			
			//---- shift -ss from before to after input, for accurate seek
			if(false) {	//not good
				/*
				ffmpeg_path,			//0
				"-ss", "0",				//1,2	--- for video stream
				"-itsoffset","0",		//3,4
				"-i", 					//5
				fileName,				//6
				
				"-ss", "0",				//7,8	--- for audio stream
				"-itsoffset","0",		//9,10
				"-i", 					//11
				fileName,				//12
				*/
				String[] para_wk=  new String[]{"","","",""};
				para_wk[0]=ffmpegLPCMextract[1];  	//-ss
				para_wk[1]=ffmpegLPCMextract[2];  	//seek_v
				para_wk[2]=ffmpegLPCMextract[7];	//-ss 
				para_wk[3]=ffmpegLPCMextract[8];	//seek_a
				
				ffmpegLPCMextract[1]=ffmpegLPCMextract[3];
				ffmpegLPCMextract[2]=ffmpegLPCMextract[4];
				ffmpegLPCMextract[3]=ffmpegLPCMextract[5];
				ffmpegLPCMextract[4]=ffmpegLPCMextract[6];
				ffmpegLPCMextract[5]=para_wk[0];
				ffmpegLPCMextract[6]=para_wk[1];

				ffmpegLPCMextract[7]=ffmpegLPCMextract[9];
				ffmpegLPCMextract[8]=ffmpegLPCMextract[10];
				ffmpegLPCMextract[9]=ffmpegLPCMextract[11];
				ffmpegLPCMextract[10]=ffmpegLPCMextract[12];
				ffmpegLPCMextract[11]=para_wk[2];
				ffmpegLPCMextract[12]=para_wk[3];
			}
		
			//-------------------------------------------------------------------------------
			// Run ffmpeg's output pipe_process 
			//-------------------------------------------------------------------------------
			String[] cmdArray = ffmpegLPCMextract;

			cmdArray = finalizeTranscoderArgs(
				this,
				fileName,
				dlna,
				media,
				params,
				cmdArray
			);

			ProcessWrapperImpl p = new ProcessWrapperImpl(cmdArray, params);
			params.maxBufferSize = 100;
			params.input_pipes[0] = tsPipe;
			//params.stdin = null;
			ProcessWrapper pipe_process = tsPipe.getPipeProcess();
			p.attachProcess(pipe_process);
			pipe_process.runInNewThread();
			
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
			}
			tsPipe.deleteLater();

			//-------------------------------------------------------------------------------
			// Run ffmpeg
			//-------------------------------------------------------------------------------
			//p.runInNewThread();
			p.runInNewThread(dlna);	//regzamod
			return p;
			//---- all end for do_all_by_ffmpeg
		}
		//-------------------------------------------------------------------------------
		// Type B : do all(Demux/Mux) by TsMuxer
		//-------------------------------------------------------------------------------
		else if(do_all_by_tsmuxer) {
			if(PMS.rz_debug>1) PMS.dbg("===>> do_all_by_tsmuxer");
			// process all by tsmuxer 
			// nothing to do here.
		}
		//-------------------------------------------------------------------------------
		// Type C: Legacy Process ( Demux by ffmpeg/mencoder & Mux by TsMuxer)
		//-------------------------------------------------------------------------------
		else {
			if(PMS.rz_debug>1) {
				PMS.dbg("======== Wellcome to Legacy Doom of TsMuxerVideo ======"); 
				PMS.dbg("---> Process Demuxer(ffmpeg/mencoder) + Muxer(TsMuxer)");
				PMS.dbg("---> Perhaps You come here because of Multi-Tracks Audio !!"); 
				PMS.dbg("---> (Currently, New Way can't handle Multi-Tracks Audio)"); 
			}
			
			String mencoderPath = configuration.getMencoderPath();

			//-------------------------------------------------------------------------------
			// Prepair Video params
			//-------------------------------------------------------------------------------
			ffVideoPipe = new PipeIPCProcess(System.currentTimeMillis() + "ffmpegvideo.mp4", System.currentTimeMillis() + "videoout", false, true);
			
			String ffmpegLPCMextract[];
			//---- Use ffmpeg for video extract
			if (extract_by_ffmpeg_v) {	//regzamod, use ffmpeg instead of mencoder
					ffmpegLPCMextract = new String[]{
					ffmpeg_path,
					"-ss", "0",
					"-i", 
					fileName,	//4
					"-y",		//5	reserved for -t
					"-y",		//6 reserved for -t
					"-vcodec", "copy",//7,8
					"-y", "-y",//9,10
					"-an",
					//"-absf","aac_adtstoasc",
					"-f", "h264", 
					"-y",
					ffVideoPipe.getInputPipe(),
				};
				//if(media.getCodecV().startsWith("h264")) {
				if(!media.isMpegTS() && media.getCodecV().startsWith("h264")) {
					ffmpegLPCMextract[9] ="-vbsf";
					ffmpegLPCMextract[10] ="h264_mp4toannexb";
				}
				
				if(ffmpeg_addopts!=null) {	//adiitional options by metafile
					int	length_org=ffmpegLPCMextract.length;
					String last_para_org=ffmpegLPCMextract[length_org-1];	//memory oldlast param(== outpot file(pipe))
					String[] opts=ffmpeg_addopts.split("\\s");
					if(opts.length>0) {
						ffmpegLPCMextract = Arrays.copyOf(ffmpegLPCMextract, ffmpegLPCMextract.length + opts.length);
						for(int i=0;i<opts.length; i++) {
							ffmpegLPCMextract[length_org-1+i]=opts[i];
						}
						ffmpegLPCMextract[ffmpegLPCMextract.length-1]=last_para_org;	//recover newlast param(== outpot file(pipe))
					}
					//PMS.dbg("ffmpeg params counts_org="+length_org+", counts_add="+opts.length);
				}
			}
			//---- Use mencoder for video extract
			else {	// original
					ffmpegLPCMextract = new String[]{
					mencoderPath,
					"-ss", "0",
					fileName,			//3
					"-quiet",			//4
					"-quiet",			//5
					"-really-quiet",	//6
					"-msglevel", "statusline=2",
					"-ovc", "copy",
					"-nosound",
					"-mc", "0",
					"-noskip",
					"-of", "rawvideo",
					"-o", ffVideoPipe.getInputPipe()
				};
			}
			//---- play strat/end pos designated in filename_params
			// start pos will be present at params.timeseek: already done
			if(endpos_v!=0) {
					if(PMS.rz_debug>1) PMS.dbg("TsMuxerVideo: set endpos_v="+endpos_v);
					if(extract_by_ffmpeg_v) {
						ffmpegLPCMextract[5] = "-t";	// for ffmpeg
					}
					else {
						ffmpegLPCMextract[5] = "-endpos";	// for mencoder
					}
					ffmpegLPCMextract[6] = ""+endpos_v;
					//PMS.dbg("TsMuxerVideo: ffmpeg -t(duration)="+endpos_v);
			}	

			//if (fileName.toLowerCase().endsWith(".evo")) {
			if (!extract_by_ffmpeg_v && fileName.toLowerCase().endsWith(".evo")) {	//regzamod
				ffmpegLPCMextract[4] = "-psprobe";
				ffmpegLPCMextract[5] = "1000000";
			}

			//logger.info("TsMuerVideo: params.stdin=" +params.stdin);	//regzamod
			if (params.stdin != null) {
				if(extract_by_ffmpeg_v) {	//regzamod
					ffmpegLPCMextract[4] = "-";
				}
				else {
					ffmpegLPCMextract[3] = "-";
				}
			}
			InputFile newInput = new InputFile();
			newInput.setFilename(fileName);
			newInput.setPush(params.stdin);

			if (media != null) {
				if(true) { //regzamod, don't check if not PS3
					if (params.mediaRenderer.isPS3()) {
						boolean compat = (media.isVideoPS3Compatible(newInput,dlna) 
						|| !params.mediaRenderer.isH264Level41Limited());
						if (!compat) {
							logger.info("The video will not play or show a black screen on the ps3...");
						}
					}
				}
				else { //original
					boolean compat = (media.isVideoPS3Compatible(newInput,dlna) || !params.mediaRenderer.isH264Level41Limited());
					if (!compat && params.mediaRenderer.isPS3()) {
						logger.info("The video will not play or show a black screen on the ps3...");
					}
				}
				//if (media.getH264AnnexB() != null && media.getH264AnnexB().length > 0) {
				if (!extract_by_ffmpeg_v && media.getH264AnnexB() != null && media.getH264AnnexB().length > 0) {	//regzamod
					int	len=media.getH264AnnexB().length;
					byte b[]=media.getH264AnnexB();
					logger.info("TSMuxerVideo:getH264AnnexB len=" + len +", data="+ b);
					
					StreamModifier sm = new StreamModifier();
					sm.setHeader(media.getH264AnnexB());
					sm.setH264_annexb(true);
					
					logger.info("TsMuerVideo: ffVideoPipe.setModifier");	//regzamod
					ffVideoPipe.setModifier(sm);
				}
			}

			//logger.info("TsMuerVideo: params.timeseek=" +params.timeseek);	//regzamod
			if (params.timeseek > 0) {
				//ffmpegLPCMextract[2] = "" + params.timeseek;	//origin
				ffmpegLPCMextract[2] = "" + timeseek_v;	//regzamod
			}

			OutputParams ffparams = new OutputParams(PMS.getConfiguration());
			ffparams.maxBufferSize = 1;
			ffparams.stdin = params.stdin;
			
			//ffVideo = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams);	//regzamod
			//for real absorb stdout/stderr, without any buffering or logging
			ffVideo = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams, false,false,true);	//regzamod

			//-----------------------------------------------------------------------------------
			// Prepair Audio params
			//-----------------------------------------------------------------------------------
			// int numAudioTracks = 1; moved top
			//if (media != null && media.getAudioCodes() != null && media.getAudioCodes().size() > 1 && configuration.isMuxAllAudioTracks()) {
			//	numAudioTracks = media.getAudioCodes().size();
			//}

			//boolean singleMediaAudio = media != null && media.getAudioCodes().size() <= 1; //moved upper top
			if (params.aid != null) {
				if(extract_by_ffmpeg_a) {
					//-------------------------------------------------------------------------------
					// use ffmpeg for audio extract, currentry, only when single channel
					//-------------------------------------------------------------------------------
					String acodec_str="unknown";
					DLNAMediaAudio audio=params.aid;
					int	aid_chg=0;
					int aid= -1;
					if(dlna.rz_MetaChannel_a_ffmpeg>0) {	//called from #TRANSCODE#
						aid=dlna.rz_MetaChannel_a_ffmpeg -1;
					}
					if(dlna.rz_MetaChannel_a>0) {	// directed by PlayMetafile
						aid=dlna.rz_MetaChannel_a -1;
					}
					if(aid>=0) {
						if(dlna.rz_MetaChannel_a>=media.getAudioCodes().size()) {
							logger.error("TsMuxerVideo: rz_MetaChannel_a="+dlna.rz_MetaChannel_a
								+" over realAudioTracks="+realAudioTracks);
							//no change
						}
						else {
							 audio= media.getAudioCodes().get(aid);
							 aid_chg=1;
						}
					}
					/* TsMuxer  Names of Acodecs in the meta file:
					A_AC3 - DD (AC3) / DD (E-AC3) / True HD (True HD only tracks with AC3 core inside).
					A_AAC - AAC
					A_DTS - DTS / DTS-HD
					A_MP3 - MPEG audio layer 1/2/3
					A_LPCM - raw pcm data or PCM WAVE file
					S_HDMV / PGS - subtitle format presentation graphic stream.
					S_TEXT/UTF8 - subtitle format SRT. The text file should be in unicode. Any formats: UTF-8, UTF-16 (little-endian, big-endian), UTF-32 (little-endian, big-endian).
					*/
					//String format_a;
					String format_a_ff="ac3";
					String acodec_ff="ac3";
					format_a_ts="A_AC3";
					
					//if(audio!=null) {
					if(aid_chg>0 || numAudioTracks==1) {   //audio channel explicitly directed or single audio
						if(audio.isAC3() && configuration.isRemuxAC3()) {
							format_a_ff="ac3";
							format_a_ts="A_AC3";
							acodec_ff="copy";
						}
						else if(audio.getAudioCodec().equals("aac")) {
							if(media.isMpegTS()) {
								format_a_ff="aac";
								format_a_ts="A_AAC";
								acodec_ff="copy";
							}
							else {
								format_a_ff="ac3";
								format_a_ts="A_AC3";
								acodec_ff="ac3";
							}
						}
						else if(audio.isDTS()) {
							if(params.mediaRenderer.isMuxDTSToMpeg()) {
								format_a_ff="dts";
								format_a_ts="A_DTS";
								acodec_ff="copy";
							}
						}
						else if(audio.isPCM()) {
							if(params.mediaRenderer.isMuxLPCMToMpeg()) {
								format_a_ff="pcm";
								format_a_ts="A_LPCM";
								acodec_ff="copy";
							}
						}
						if(PMS.rz_debug>1) PMS.dbg("format_a_ts="+format_a_ts);
					}
					
					ffAudioPipe = new PipeIPCProcess[numAudioTracks];
					ffAudioPipe[0] = new PipeIPCProcess(System.currentTimeMillis() 
						+ "ffmpegaudio01."+format_a_ff, System.currentTimeMillis() + "audioout", false, true);
					
					ffmpegLPCMextract = new String[]{
						ffmpeg_path,//0
						"-ss", 		//1
						"0",		//2
						"-i",		//3
						fileName,	//4
						"-y",		//5 reserved for -t
						"-y",		//6 reserved for -t
						"-vn", "-y",//7,8
						"-y", "-y",//9,10
						"-acodec", acodec_ff,//11,12
						"-y",	//13
						"-y",	//14
						"-map","0:a",	//15,16
						//"-absf","aac_adtstoasc",	
						//"-f", "h264", 
						//"-channels", "" + sm.getNbchannels(),
						//"-f", (ac.equals("aac")) ? "aac":"ac3",
						ffAudioPipe[0].getInputPipe()
					};
					//if(media.getCodecV().startsWith("h264")) {
					//	ffmpegLPCMextract[9] ="-vbsf";
					//	ffmpegLPCMextract[10] ="h264_mp4toannexb";
					//}
					if(aid_chg>0) {
						ffmpegLPCMextract[16]="0:a:"+(aid);	//default: "1:a" == all audio of input2
						//PMS.dbg("Change Audio trackNo.="+ffmpegLPCMextract[16]);
					}
					//PMS.dbg("set ffmpeg audio, timeseek_a="+timeseek_a);
					if (timeseek_a != 0) {
						//ffmpegLPCMextract[2] = "" + params.timeseek;
						ffmpegLPCMextract[2] = "" + timeseek_a;	//regzamod
					}
					if (params.stdin != null) {
						ffmpegLPCMextract[4] = "-";
					}
					if(endpos_a!=0) {
						if(PMS.rz_debug>1) PMS.dbg("TsMuxerVideo: set endpos_a="+endpos_a);
						ffmpegLPCMextract[5] = "-t";
						ffmpegLPCMextract[6] = "" + endpos_a;
					}
					ffparams = new OutputParams(PMS.getConfiguration());
					ffparams.maxBufferSize = 1;
					ffparams.stdin = params.stdin;
					ffAudio = new ProcessWrapperImpl[numAudioTracks];
					ffAudio[0] = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams);
				}
				else {
					//-------------------------------------------------------------------------------
					// use Mencoder for audio extract
					//-------------------------------------------------------------------------------
					//-------------------------------------------------------------------------------
					// for Single Audio channel
					//-------------------------------------------------------------------------------
					if (numAudioTracks <= 1 || dlna.rz_MetaChannel_a>0) {
						track_a=""+params.aid.getId();
						if(dlna.rz_MetaChannel_a_ffmpeg>0) {	//called from #TRANSCODE#
							track_a=""+dlna.rz_MetaChannel_a_ffmpeg;
						}
						else if(dlna.rz_MetaChannel_a>0) {	//designated by PlayMetafile
							track_a=""+dlna.rz_MetaChannel_a;
						}
						//PMS.dbg("TsMuxerVideo: mencoder track_a="+track_a);
						
						ffAudioPipe = new PipeIPCProcess[numAudioTracks];
						ffAudioPipe[0] = new PipeIPCProcess(System.currentTimeMillis() + "ffmpegaudio01", System.currentTimeMillis() + "audioout", false, true);
						if (
							(
								configuration.isMencoderUsePcm() ||
								configuration.isDTSEmbedInPCM()
							) && (
								params.aid.isLossless() ||
								params.aid.isDTS()
							) &&
							params.mediaRenderer.isDTSPlayable()
						) {
							//-------------------------------------------------------------------------------
							// for audio codec PCM|DTS|LossLess
							//-------------------------------------------------------------------------------
							StreamModifier sm = new StreamModifier();
							sm.setPcm(true);
							sm.setDtsembed(configuration.isDTSEmbedInPCM() && params.aid.isDTS());
							sm.setNbchannels(sm.isDtsembed() ? 2 : CodecUtil.getRealChannelCount(configuration, params.aid));
							sm.setSampleFrequency(params.aid.getSampleRate() < 48000 ? 48000 : params.aid.getSampleRate());
							sm.setBitspersample(16);
							String mixer = CodecUtil.getMixerOutput(!sm.isDtsembed(), sm.getNbchannels());
							ffmpegLPCMextract = new String[]{
								mencoderPath,
								"-ss", "0",
								fileName,
								"-quiet",
								"-quiet",
								"-quiet",	//6
								"-quiet",	//7
								"-quiet",	//8
								"-quiet",	//9
								"-really-quiet",
								"-msglevel", "statusline=2",
								"-channels", "" + sm.getNbchannels(),
								"-ovc", "copy",
								"-of", "rawaudio",
								"-mc", sm.isDtsembed() ? "0.1" : "0",
								"-noskip",
								"-oac", sm.isDtsembed() ? "copy" : "pcm",
								mixer != null ? "-af" : "-quiet", mixer != null ? mixer : "-quiet",
								//singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : ("" + params.aid.getId()),
								singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : (track_a),
								"-srate", "48000",
								"-o", ffAudioPipe[0].getInputPipe()
							};
							if (!params.mediaRenderer.isMuxDTSToMpeg()) {
								ffAudioPipe[0].setModifier(sm);
							}
						} else {
							//-------------------------------------------------------------------------------
							// for audio codec OTHER THAN [PCM|DTS|LossLess], may be AC3
							//-------------------------------------------------------------------------------
							ffmpegLPCMextract = new String[]{
								mencoderPath,
								"-ss", "0",
								fileName,
								"-quiet",
								"-quiet",
								"-quiet",	//6
								"-quiet",	//7
								"-quiet",	//8
								"-quiet",	//9
								"-really-quiet",
								"-msglevel", "statusline=2",
								"-channels", "" + CodecUtil.getAC3ChannelCount(configuration, params.aid),
								"-ovc", "copy",
								"-of", "rawaudio",
								"-mc", "0",
								"-noskip",
								"-oac", (params.aid.isAC3() && configuration.isRemuxAC3()) ? "copy" : "lavc",
								params.aid.isAC3() ? "-fafmttag" : "-quiet", params.aid.isAC3() ? "0x2000" : "-quiet",
								"-lavcopts", "acodec=" + (configuration.isMencoderAc3Fixed() ? "ac3_fixed" : "ac3") + ":abitrate=" + CodecUtil.getAC3Bitrate(configuration, params.aid),
								"-af", "lavcresample=48000",
								"-srate", "48000",
								//singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : ("" + params.aid.getId()),
								singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : (track_a),
								"-o", ffAudioPipe[0].getInputPipe()
							};
						}

						if (fileName.toLowerCase().endsWith(".evo")) {
							ffmpegLPCMextract[4] = "-psprobe";
							ffmpegLPCMextract[5] = "1000000";
						}

						if (params.stdin != null) {
							ffmpegLPCMextract[3] = "-";
						}
						//PMS.dbg("set mencoder single-audio, timeseek_a="+timeseek_a);
						if (timeseek_a != 0) {
							ffmpegLPCMextract[2] = "" + timeseek_a;	//regzamod
						}
						if(endpos_a!=0) {
								if(PMS.rz_debug>1) PMS.dbg("TsMuxerVideo: set endpos_a="+endpos_a);
								ffmpegLPCMextract[6] = "-endpos";
								ffmpegLPCMextract[7] = "" + endpos_a;
						}
						if(dlna.rz_isPartPlay) { //regzamod, test for m2ts partial play --> improve sound duration
								//double fpsw=Double.parseDouble(fps)*2.0;
								if(PMS.rz_debug>1) PMS.dbg("TSMuxerVideo: set -fps for audio fps="+fps);
								ffmpegLPCMextract[8] = "-fps";
								ffmpegLPCMextract[9] = ""+fps;
						}
						ffparams = new OutputParams(PMS.getConfiguration());
						ffparams.maxBufferSize = 1;
						ffparams.stdin = params.stdin;
						ffAudio = new ProcessWrapperImpl[numAudioTracks];
						ffAudio[0] = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams);
					} else {
						//-------------------------------------------------------------------------------
						// for Multi Audio channels
						//-------------------------------------------------------------------------------
						ffAudioPipe = new PipeIPCProcess[numAudioTracks];
						ffAudio = new ProcessWrapperImpl[numAudioTracks];
						for (int i = 0; i < media.getAudioCodes().size(); i++) {
							DLNAMediaAudio audio = media.getAudioCodes().get(i);
							ffAudioPipe[i] = new PipeIPCProcess(System.currentTimeMillis() + "ffmpeg" + i, System.currentTimeMillis() + "audioout" + i, false, true);
							if (
								(
									audio.isLossless() ||
									audio.isDTS()
								) && (
									configuration.isMencoderUsePcm() ||
									configuration.isDTSEmbedInPCM()
								) && params.mediaRenderer.isDTSPlayable()
							) {
								//-------------------------------------------------------------------------------
								// for audio codec PCM|DTS|LossLess
								//-------------------------------------------------------------------------------
								StreamModifier sm = new StreamModifier();
								sm.setPcm(true);
								sm.setDtsembed(configuration.isDTSEmbedInPCM() && audio.isDTS());
								sm.setNbchannels(sm.isDtsembed() ? 2 : CodecUtil.getRealChannelCount(configuration, audio));
								sm.setSampleFrequency(48000);
								sm.setBitspersample(16);
								if (!params.mediaRenderer.isMuxDTSToMpeg()) {
									ffAudioPipe[i].setModifier(sm);
								}
								String mixer = CodecUtil.getMixerOutput(!sm.isDtsembed(), sm.getNbchannels());
								ffmpegLPCMextract = new String[]{
									mencoderPath,
									"-ss", "0",
									fileName,
									"-quiet",
									"-quiet",
									"-quiet",	//6
									"-quiet",	//7
									"-quiet",	//8
									"-quiet",	//9
									"-really-quiet",
									"-msglevel", "statusline=2",
									"-channels", "" + sm.getNbchannels(),
									"-ovc", "copy",
									"-of", "rawaudio",
									"-mc", sm.isDtsembed() ? "0.1" : "0",
									"-noskip",
									"-oac", sm.isDtsembed() ? "copy" : "pcm",
									mixer != null ? "-af" : "-quiet", mixer != null ? mixer : "-quiet",
									singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : ("" + audio.getId()),
									"-srate", "48000",
									"-o", ffAudioPipe[i].getInputPipe()
								};
							} else {
								//-------------------------------------------------------------------------------
								// for audio codec OTHER THAN [PCM|DTS|LossLess], may be AC3
								//-------------------------------------------------------------------------------
								ffmpegLPCMextract = new String[]{
									mencoderPath,
									"-ss", "0",
									fileName,
									"-quiet",
									"-quiet",
									"-quiet",	//6
									"-quiet",	//7
									"-quiet",	//8
									"-quiet",	//9
									"-really-quiet",
									"-msglevel", "statusline=2",
									"-channels", "" + CodecUtil.getAC3ChannelCount(configuration, audio),
									"-ovc", "copy",
									"-of", "rawaudio",
									"-mc", "0",
									"-noskip",
									"-oac", (audio.isAC3() && configuration.isRemuxAC3()) ? "copy" : "lavc",
									audio.isAC3() ? "-fafmttag" : "-quiet", audio.isAC3() ? "0x2000" : "-quiet",
									"-lavcopts", "acodec=" + (configuration.isMencoderAc3Fixed() ? "ac3_fixed" : "ac3") + ":abitrate=" + CodecUtil.getAC3Bitrate(configuration, audio),
									"-af", "lavcresample=48000",
									"-srate", "48000",
									singleMediaAudio ? "-quiet" : "-aid", singleMediaAudio ? "-quiet" : ("" + audio.getId()),
									"-o", ffAudioPipe[i].getInputPipe()
								};
							}

							if (fileName.toLowerCase().endsWith(".evo")) {
								ffmpegLPCMextract[4] = "-psprobe";
								ffmpegLPCMextract[5] = "1000000";
							}

							if (params.stdin != null) {
								ffmpegLPCMextract[3] = "-";
							}
							//PMS.dbg("set mencoder multi-audio, timeseek_a="+timeseek_a);
							if (timeseek_a != 0) {
								//ffmpegLPCMextract[2] = "" + params.timeseek;
								ffmpegLPCMextract[2] = "" + timeseek_a;
							}
							if(endpos_a!=0) {
								//PMS.dbg("TsMuxerVideo: set endpos_a="+endpos_a);
								ffmpegLPCMextract[6] = "-endpos";
								ffmpegLPCMextract[7] = "" + endpos_a;
							}
							if(true) { //regzamod, test for m2ts partial play --> improve sound duration
									//double fpsw=Double.parseDouble(fps)*2.0;
									//PMS.dbg("TSMuxerVideo: Test set double fps for audio fps="+fps+", *2="+fpsw);
									ffmpegLPCMextract[8] = "-fps";
									ffmpegLPCMextract[9] = ""+fps;
							}
							ffparams = new OutputParams(PMS.getConfiguration());
							ffparams.maxBufferSize = 1;
							ffparams.stdin = params.stdin;
							ffAudio[i] = new ProcessWrapperImpl(ffmpegLPCMextract, ffparams);
						}
					}
				}
			}
		}

		if(do_all_by_ffmpeg) {
		}
		else {	//mux by TsMuxer
			//-------------------------------------------------------------------------------
			// Prepair params for TsMuxer
			//-------------------------------------------------------------------------------
			//File f = new File(configuration.getTempFolder(), "pms-tsmuxer.meta");
			File f = new File(configuration.getTempFolder(), "pms-tsmuxer-ff("+params.sess.id+").meta");
			params.log = false;
			PrintWriter pw;
			if(dlna.rz_isPartPlay) {
				//to preserve encode of fileName passed from Metafile
				//given fileName may be shortName for Windos, --> let's use fullname
				if(PMS.rz_debug>1) {
					PMS.dbg("given fileName="+fileName);
					PMS.dbg("-->may be shortned name, use fullName: dlna.rz_Bodyfile="+dlna.rz_Bodyfile);
				}
				//fileName=dlna.rz_Bodyfile.getCanonicalPath(); //get regular full pathname
				//--> rz_Bodyfile may be null, rz_isPartPlay not means allways metafile
				fileName=dlna.getSrcPath(); //get regular full pathname
				//pw = new PrintWriter(f,configuration.getRZ_MetafileEncode());
				pw = new PrintWriter(f,"MS932");   //TsMUxeR understands only MS932, cann't UTF-8
			}
			else {
				//pw = new PrintWriter(f);
				pw = new PrintWriter(f,"MS932"); //TsMUxeR understands only MS932, cann't UTF-8
			}
			
			//------------------------------------------------------
			// TsMuxer general params 
			//------------------------------------------------------
			pw.print("MUXOPT --no-pcr-on-video-pid ");
			pw.print("--new-audio-pes ");
			if (ffVideo != null) {
				pw.print("--no-asyncio ");
			}
			pw.print(" --vbr");
			//if(dlna.rz_isPartPlay && cut_by_muxer) {	// partial_cut_out by tsMuxer 
			if(cut_by_muxer) {	// partial_cut_out by tsMuxer 
				if(timeseek_t>0) pw.print(" --cut-start="+timeseek_t+"s");
				if(endpos_t>0) pw.print(" --cut-end="+(timeseek_t+endpos_t)+"s");
			}
			pw.println(" --vbv-len=500");

			//------------------------------------------------------
			// TsMuxer video channel params 
			//------------------------------------------------------
			if(do_all_by_tsmuxer) {
				String videoparams;
				if(track_v==null) {
					 videoparams= "level=4.1, insertSEI, contSPS";
				}
				else {
					 videoparams= "level=4.1, insertSEI, contSPS, track="+track_v;
				}
				if (this instanceof TsMuxerAudio) {
					videoparams = "track=224";
				}
				if (configuration.isFix25FPSAvMismatch()) {
					fps = "25";
				}
				else if(fps==null) {	//getValidFps failed: not noramal fps value, e.x. 10 fps
					fps=media.getFrameRate();	//regzamod
				}
				pw.println(videoType + ", \"" + fileName + "\", " 
					+ (fps != null ? ("fps=" + fps + ", ") : "") + videoparams);
			}
			else {
				if (ffVideoPipe != null) {
					String videoparams = "level=4.1, insertSEI, contSPS, track=1";
					if (this instanceof TsMuxerAudio) {
						videoparams = "track=224";
					}
					if (configuration.isFix25FPSAvMismatch()) {
						fps = "25";
					}
					else if(fps==null) {	//getValidFps failed: not noramal fps value, e.x. 10 fps
						fps=media.getFrameRate();	//regzamod
					}
					pw.println(videoType + ", \"" + ffVideoPipe.getOutputPipe() + "\", " 
					 	+ (fps != null ? ("fps=" + fps + ", ") : "") + videoparams);
				}
			}
			
			//------------------------------------------------------
			// TsMuxer audio channel params 
			//------------------------------------------------------
			//sound_delay from renderer.conf RZ_RemuxAudioDelayRestart
			long delay_tsm=0;
			if(sound_delay_ts!=0){
				//delay by tsMuxer
				delay_tsm +=(long)(sound_delay_ts*1000);
			}
			
			String str=dlna.rz_MetaTsmuxerOptA[1];
			if(do_all_by_tsmuxer) {
				if(numAudioTracks>0) {
					//currentry, must be single audio channel, but do anyway 
					String timeshift = "";
					String type = "A_AC3";
					if(params.aid.getAudioCodec().equals("aac")) {
						type = "A_AAC";
					}
					long delay_tsmp=delay_tsm;
					if(delay_tsm!=0) {
						timeshift = ", timeshift=" + delay_tsmp + "ms, ";
					}
					
					if(str!=null && str.length()>0) {
						String opta=dlna.getTsmuxerOptAReprace(str,type,fileName,
							timeshift,
							"track="+(track_a==null?"":track_a));
						pw.println(opta);
					}
					
					else if(track_a==null) {
						pw.println(type + ", \"" + fileName + "\"" + timeshift);
					}
					else {
						pw.println(type + ", \"" + fileName + "\"" + timeshift + ", track="+track_a);
					}
				}
			}
			else if(extract_by_ffmpeg_a) {
				//if (ffAudioPipe != null && ffAudioPipe.length == 1) {	//currentry, support only single channel
				if (ffAudioPipe != null) {	//currentry, support only single channel, but do anyway
					String timeshift = "";
					String type = format_a_ts;
					long delay_tsmp=delay_tsm;
					
					if (params.aid != null && params.aid.getDelay() != 0) {
						delay_tsmp +=params.aid.getDelay();
						//timeshift = "timeshift=" + params.aid.getDelay() + "ms, ";
					}
					if(delay_tsm!=0) {
						timeshift = "timeshift=" + delay_tsmp + "ms, ";
					}
					if(str!=null && str.length()>0) {
						String opta=dlna.getTsmuxerOptAReprace(str,type,
							ffAudioPipe[0].getOutputPipe(),
							timeshift,
							"track=2");
						pw.println(opta);
					}
					else {
						pw.println(type + ", \"" + ffAudioPipe[0].getOutputPipe() + "\", " + timeshift + "track=2");
					}
				}
			}
			else {
				if (ffAudioPipe != null && ffAudioPipe.length == 1) {	//single channel
					String timeshift = "";
					String type = "A_AC3";
					long delay_tsmp=delay_tsm;
					if (
						(
							(configuration.isMencoderUsePcm() || configuration.isDTSEmbedInPCM()) &&
							(params.aid.isDTS() || params.aid.isLossless()) &&
							params.mediaRenderer.isDTSPlayable()
						) ||
						this instanceof TsMuxerAudio
					) {
						type = "A_LPCM";
						if (params.mediaRenderer.isMuxDTSToMpeg()) {
							type = "A_DTS";
						}
					}
					if(extract_by_ffmpeg_a && params.aid.getAudioCodec().equals("aac")) {
						type = "A_AAC";
					}
					if (params.aid != null && params.aid.getDelay() != 0) {
						delay_tsmp +=params.aid.getDelay();
						//timeshift = "timeshift=" + params.aid.getDelay() + "ms, ";
					}
					if(delay_tsm!=0) {
						timeshift = "timeshift=" + delay_tsmp + "ms, ";
					}
					if(str!=null && str.length()>0) {
						String opta=dlna.getTsmuxerOptAReprace(str,type,
							ffAudioPipe[0].getOutputPipe(),
							timeshift,
							"track=2");
						pw.println(opta);
					}
					else {
						pw.println(type + ", \"" + ffAudioPipe[0].getOutputPipe() + "\", " + timeshift + "track=2");
					}
				} else if (ffAudioPipe != null) {	//multi-channel
					for (int i = 0; i < media.getAudioCodes().size(); i++) {
						DLNAMediaAudio lang = media.getAudioCodes().get(i);
						String timeshift = "";
						boolean lossless = false;
						long delay_tsmp=delay_tsm;
						
						if ((lang.isDTS() || lang.isLossless()) && (configuration.isMencoderUsePcm() || configuration.isDTSEmbedInPCM()) && params.mediaRenderer.isDTSPlayable()) {
							lossless = true;
						}
						String type = "A_AC3";
						if (lossless) {
							type = "A_LPCM";
							if (params.mediaRenderer.isMuxDTSToMpeg()) {
								type = "A_DTS";
							}
						}
						if (lang.getDelay() != 0) {
							delay_tsmp+=lang.getDelay();
						}
						if(delay_tsmp!=0) {
							timeshift = "timeshift=" + delay_tsmp + "ms, ";
						}
						if(str!=null && str.length()>0) {
							String opta=dlna.getTsmuxerOptAReprace(str,type,
								ffAudioPipe[0].getOutputPipe(),
								"timeshift="+delay_tsmp+"ms",
								"track=" + (2 + i));
							pw.println(opta);
						}
						else {
							pw.println(type + ", \"" + ffAudioPipe[i].getOutputPipe() + "\", " + timeshift + "track=" + (2 + i));
						}
					}
				}
			}
			pw.close();

			//-------------------------------------------------------------------------------
			// Run TsMuxer's output pipe_process 
			//-------------------------------------------------------------------------------
			//PipeProcess tsPipe = new PipeProcess(System.currentTimeMillis() + "tsmuxerout.ts");	//original
			PipeProcess tsPipe= new PipeProcess(System.currentTimeMillis() + tsmuxerout_fmt);	//regzamod, for enable SeekByTime
			String[] cmdArray = new String[]{executable(), f.getAbsolutePath(), tsPipe.getInputPipe()};

			cmdArray = finalizeTranscoderArgs(
				this,
				fileName,
				dlna,
				media,
				params,
				cmdArray
			);

			ProcessWrapperImpl p = new ProcessWrapperImpl(cmdArray, params);
			params.maxBufferSize = 100;
			params.input_pipes[0] = tsPipe;
			params.stdin = null;
			ProcessWrapper pipe_process = tsPipe.getPipeProcess();
			p.attachProcess(pipe_process);
			pipe_process.runInNewThread();

			try {
				Thread.sleep(50);
			} catch (InterruptedException e) {
			}
			tsPipe.deleteLater();

			//-------------------------------------------------------------------------------
			// Run Video processes (video stream extracter)
			//-------------------------------------------------------------------------------
			if (ffVideoPipe != null) {
				ProcessWrapper ff_pipe_process = ffVideoPipe.getPipeProcess();
				p.attachProcess(ff_pipe_process);
				ff_pipe_process.runInNewThread();
				try {
					Thread.sleep(50);
				} catch (InterruptedException e) {
				}
				ffVideoPipe.deleteLater();

				p.attachProcess(ffVideo);
				//PMS.dbg("TsmuxerVideo: kick ffmpeg for video");
				ffVideo.runInNewThread(dlna);
				try {
					Thread.sleep(50);
				} catch (InterruptedException e) {
				}
			}

			//-------------------------------------------------------------------------------
			// run Audio processes (audio streams extracter)
			//-------------------------------------------------------------------------------
			if (ffAudioPipe != null && params.aid != null) {
				for (int i = 0; i < ffAudioPipe.length; i++) {
					ProcessWrapper ff_pipe_process = ffAudioPipe[i].getPipeProcess();
					p.attachProcess(ff_pipe_process);
					ff_pipe_process.runInNewThread();
					try {
						Thread.sleep(50);
					} catch (InterruptedException e) {
					}
					ffAudioPipe[i].deleteLater();
					p.attachProcess(ffAudio[i]);
					ffAudio[i].runInNewThread(dlna);
				}
			}

			//short wait for processes get ready 
			try {
				Thread.sleep(100);
			} catch (InterruptedException e) {
			}

			//-------------------------------------------------------------------------------
			// Run TsMuxer (Streams muxer)
			//-------------------------------------------------------------------------------
			//p.runInNewThread();
			p.runInNewThread(dlna);	//regzamod
			return p;
		}
		return null;
	}

	@Override
	public String mimeType() {
		return "video/mpeg";
	}

	@Override
	public String name() {
		return "tsMuxeR";
	}

	@Override
	public int type() {
		return Format.VIDEO;
	}
	private JCheckBox tsmuxerforcefps;
	private JCheckBox muxallaudiotracks;

	@Override
	public JComponent config() {
		FormLayout layout = new FormLayout(
			"left:pref, 0:grow",
			"p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, p, 3dlu, 0:grow");
		PanelBuilder builder = new PanelBuilder(layout);
		builder.setBorder(Borders.EMPTY_BORDER);
		builder.setOpaque(false);

		CellConstraints cc = new CellConstraints();


		JComponent cmp = builder.addSeparator(Messages.getString("TSMuxerVideo.3"), cc.xyw(2, 1, 1));
		cmp = (JComponent) cmp.getComponent(0);
		cmp.setFont(cmp.getFont().deriveFont(Font.BOLD));

		tsmuxerforcefps = new JCheckBox(Messages.getString("TSMuxerVideo.2"));
		tsmuxerforcefps.setContentAreaFilled(false);
		if (configuration.isTsmuxerForceFps()) {
			tsmuxerforcefps.setSelected(true);
		}
		tsmuxerforcefps.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				configuration.setTsmuxerForceFps(e.getStateChange() == ItemEvent.SELECTED);
			}
		});
		builder.add(tsmuxerforcefps, cc.xy(2, 3));

		muxallaudiotracks = new JCheckBox(Messages.getString("TSMuxerVideo.19"));
		muxallaudiotracks.setContentAreaFilled(false);
		if (configuration.isMuxAllAudioTracks()) {
			muxallaudiotracks.setSelected(true);
		}

		muxallaudiotracks.addItemListener(new ItemListener() {
			public void itemStateChanged(ItemEvent e) {
				configuration.setMuxAllAudioTracks(e.getStateChange() == ItemEvent.SELECTED);
			}
		});
		builder.add(muxallaudiotracks, cc.xy(2, 5));

		return builder.getPanel();
	}

	public boolean isInternalSubtitlesSupported() {
		return false;
	}

	public boolean isExternalSubtitlesSupported() {
		return false;
	}

	@Override
	public boolean isPlayerCompatible(RendererConfiguration mediaRenderer) {
		return mediaRenderer != null && mediaRenderer.isMuxH264MpegTS();
	}
}
