/*
 * PS3 Media Server, for streaming any medias to your PS3.
 * Copyright (C) 2008  A.Brochard
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; version 2
 * of the License only.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package net.pms.dlna.virtual;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import net.pms.PMS;
import net.pms.dlna.DLNAResource;
import net.pms.dlna.RootFolder;
import net.pms.network.HTTPResource;
import net.pms.configuration.RendererConfiguration;

/**
 * Represents a container (folder). This is widely used by the UPNP ContentBrowser service. Child objects are expected in this folder.
 */
public class VirtualFolder extends DLNAResource {
	protected String name;
	protected String thumbnailIcon;
	protected String thumbnailContentType;
	private long prevCallTime;
	private String name0;

	/**Constructor for this class. The constructor does not add any child to the container. This is the only
	 * chance to set the name of this container.
	 * @param name String to be shown in the ContentBrowser service 
	 * @param thumbnailIcon Represents a thumbnail to be shown. The String represents an absolute
	 *        path. Use null if none is available or desired.
	 * @see #addChild(DLNAResource)
	 */
	public VirtualFolder(String name, String name0, String thumbnailIcon) {
		this.name = name;
		this.name0= name0;
		//dname=name;
		this.thumbnailIcon = thumbnailIcon;
		if (thumbnailIcon != null && thumbnailIcon.toLowerCase().endsWith(".png")) {
			thumbnailContentType = HTTPResource.PNG_TYPEMIME;
		} else {
			thumbnailContentType = HTTPResource.JPEG_TYPEMIME;
		}
		sort_enabled=true;	//is default
		doSort=true;
		
		if(rz_MetaSortType<0) { //not forced
			rz_MetaSortType=PMS.SORT_ORIGIN;
		}
	}
	
	public VirtualFolder(String name, String thumbnailIcon) {
		this.name = name;
		this.name0= name;
		//dname=name;
		this.thumbnailIcon = thumbnailIcon;
		if (thumbnailIcon != null && thumbnailIcon.toLowerCase().endsWith(".png")) {
			thumbnailContentType = HTTPResource.PNG_TYPEMIME;
		} else {
			thumbnailContentType = HTTPResource.JPEG_TYPEMIME;
		}
		//sort_enabled=false;	//default
		if(rz_MetaSortType<0) { //not forced
			rz_MetaSortType=PMS.SORT_ORIGIN;  //sort by id: order of created
		}
	}

	/**Because a container cannot be streamed, this function always returns null.
	 * @return null 
	 * @see net.pms.dlna.DLNAResource#getInputStream()
	 */
	@Override
	public InputStream getInputStream() throws IOException {
		return null;
	}
	
	@Override
	public boolean needSort() {	//add, regzamod
		return sort_enabled;
	}
	
	/*
	@Override
	public List<DLNAResource> getChildren() {
		if(sotype==DLNAResource.SO_CONTROLLS) {
			//PMS.dbg("VirtualFolder.getChildren: called, renderer="+getDefaultRenderer()+", child count="+children.size());
			for(DLNAResource ch: children) {
				if((ch.mflags&DLNAResource.MF_HIDE)!=0) {
					//PMS.dbg("VirtualFolder.getChildren: child MF_HIDE is ON, --> set OFF, ch="+ch.getName());
					ch.mflags &=~DLNAResource.MF_HIDE;
				}
			}
		}
		return children;
	}
	*/

	@Override
	public void discoverChildren() {
		//if(sotype==DLNAResource.SO_CONTROLLS) PMS.dbg("VirtualFolder.discoverChildren: called");
		if(PMS.rz_debug>1 && getParent()!=null && getParent().ctlFolder==this) {
			PMS.dbg("ControlFolder: sort_enabled="+sort_enabled+", rz_MetaSortType="+rz_MetaSortType);
			for (DLNAResource ch: getChildren()){
				PMS.dbg("Child name="+ch.getName()+", id="+ch.getId()+", serialId="+ch.getSerialId()+", sort_immune="+ch.sort_immune);
			}
		}
	}

	/**Returns a string representing the container. This string is used in
	 * the UPNP ContentBrowser service.
	 * @see net.pms.dlna.DLNAResource#getName()
	 */
	
	@Override
	public String getName() {
		if(rz_linkobj!=null && (mflags&MF_USE_LINKNAME)!=0) {
			return rz_linkobj.getName();
		}
		return name;
	}
	
	@Override
	public String getName0() {
		if(rz_linkobj!=null && (mflags&MF_USE_LINKNAME)!=0) {
			return rz_linkobj.getName0();
		}
		return name0;
	}

	/*
	@Override
	public String getDisplayName() {
		return getDisplayName(null);
	}
	
	@Override
	public String getDisplayName(RendererConfiguration mediaRenderer) {
		if(rz_linkobj!=null && (mflags&MF_USE_LINKNAME)!=0) {
			return rz_linkobj.getDisplayName(mediaRenderer);
		}
		return super.getDisplayName(mediaRenderer);
	}
	*/

	@Override
	public String getDpName() {	//regzamod
		return name;
	}

	public int activate(int type) {
		//if(sotype==DLNAResource.SO_CONTROLLS) PMS.dbg("CONTROLLS.activate: called");
		long tim1 = System.currentTimeMillis();
		if(type!=1 || (tim1-prevCallTime) <1000) { 
			//if(sotype==DLNAResource.SO_CONTROLLS) PMS.dbg("CONTROLLS.activate: Noop (re-called in short time)");
			return 0;
		}
		RootFolder rt=PMS.getRootFolderByDlna(this);
		//PMS.dbg("VirtualFolder.activate: rz_MenuLock="+rt.rz_MenuLock);
		if(rt.rz_MenuLock==0) {
			//if(videoSettingsRes!=null) videoSettingsRes.mflags&=~MF_HIDE;
			//if(rz_WebQS!=null) rz_WebQS.mflags&=~MF_HIDE;
			if(ClearFolderCahceMenu!=null) ClearFolderCahceMenu.mflags&=~MF_HIDE;
			if(ScriptMenuFolder!=null) ScriptMenuFolder.mflags&=~MF_HIDE;
			if(resumeFolder!=null) resumeFolder.mflags&=~MF_HIDE;
			if(resumeFolder2!=null) resumeFolder2.mflags&=~MF_HIDE;
		}
		else {  //lock-out menu_operations
			//if(sotype==DLNAResource.SO_CONTROLLS) PMS.dbg("CONTROLLS.activate: menu-locked --> Hide Menu");
			//if(videoSettingsRes!=null) videoSettingsRes.mflags|=MF_HIDE;
			//if(rz_WebQS!=null) rz_WebQS.mflags|=MF_HIDE;
			if(ClearFolderCahceMenu!=null) ClearFolderCahceMenu.mflags|=MF_HIDE;
			if(ScriptMenuFolder!=null) ScriptMenuFolder.mflags|=MF_HIDE;
			if(resumeFolder!=null) resumeFolder.mflags|=MF_HIDE;
			if(resumeFolder2!=null) resumeFolder2.mflags|=MF_HIDE;
		}
		prevCallTime=tim1;
		return 0;
	}

	/**Returns true in this case, as this is a folder.
	 * @return true
	 * @see net.pms.dlna.DLNAResource#isFolder()
	 */
	@Override
	public boolean isFolder() {
		return true;
	}

	/**Returns zero as this is a folder (container).
	 * @see net.pms.dlna.DLNAResource#length()
	 */
	@Override
	public long length() {
		return 0;
	}

	/**Containers are likely not to be modified, so this one returns zero.
	 * TODO: (botijo) When is this used then? Is this a prototype?
	 * @return Zero
	 */
	public long lastModified() {
		return 0;
	}

	@Override
	public String getSystemName() {
		return getName();
	}

	/**Returns a {@link InputStream} that represents the thumbnail used.
	 * @see net.pms.dlna.DLNAResource#getThumbnailInputStream()
	 */
	@Override
	public InputStream getThumbnailInputStream() {
		return getResourceInputStream(thumbnailIcon);
	}

	/**Returns the thumbnailContentType associated to the thumbnail associated to this container.
	 * @see net.pms.dlna.DLNAResource#getThumbnailContentType()
	 * @see #thumbnailContentType
	 */
	@Override
	public String getThumbnailContentType() {
		return thumbnailContentType;
	}

	/**Returns true, as a container is always a valid item to add to another container.
	 * @see net.pms.dlna.DLNAResource#isValid()
	 */
	@Override
	public boolean isValid() {
		return true;
	}

	public void setThumbnail(String thumbnailIcon) {
		this.thumbnailIcon = thumbnailIcon;
		
		if (thumbnailIcon != null && thumbnailIcon.toLowerCase().endsWith(".png")) {
			thumbnailContentType = HTTPResource.PNG_TYPEMIME;
		} else {
			thumbnailContentType = HTTPResource.JPEG_TYPEMIME;
		}
		
	}
	
	@Override
	public String toString (RendererConfiguration mediaRenderer) {
		toString_sb=null;
		return super.toString(mediaRenderer);
	}
}
