/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                H   H   AAA   SSSSS  H   H  M   M   AAA   PPPP               %
%                H   H  A   A  SS     H   H  MM MM  A   A  P   P              %
%                HHHHH  AAAAA   SSS   HHHHH  M M M  AAAAA  PPPP               %
%                H   H  A   A     SS  H   H  M   M  A   A  P                  %
%                H   H  A   A  SSSSS  H   H  M   M  A   A  P                  %
%                                                                             %
%                                                                             %
%                  ImageMagick Hash-map and Linked-list Methods               %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                               December 2002                                 %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  This module implements the standard handy hash and linked-list methods for
%  storing and retrieving large numbers of data elements.  It is loosely based
%  on the Java implementation of these algorithms.
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/error.h"
#include "magick/hashmap.h"
#include "magick/memory_.h"
#include "magick/semaphore.h"
#include "magick/signature.h"
#include "magick/string_.h"

/*
  Typedef declarations.
*/
typedef struct _ElementInfo
{
  void
    *datum;

  struct _ElementInfo
    *next;
} ElementInfo;

typedef struct _EntryInfo
{
  size_t
    hash;

  void
    *key,
    *datum;
} EntryInfo;

struct _HashmapInfo
{
  size_t
    (*hash)(const void *);

  void
    *(*liberate_key)(void *),
    *(*liberate_datum)(void *);

  unsigned long
    maximum_entries,
    entries,
    next;

  MagickBooleanType
    head_of_list;

  struct _LinkedListInfo
    **map;

  MagickBooleanType
    debug;

  SemaphoreInfo
    *semaphore;

  unsigned long
    signature;
};

struct _LinkedListInfo
{
  unsigned long
    maximum_elements,
    elements;

  ElementInfo
    *head,
    *tail,
    *next;

  MagickBooleanType
    debug;

  SemaphoreInfo
    *semaphore;

  unsigned long
    signature;
};

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A p p e n d E l e m e n t T o L i n k e d L i s t                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AppendElementToLinkedList() appends an element to the end of the linked-list.
%
%  The format of the AppendElementToLinkedList method is:
%
%      MagickBooleanType AppendElementToLinkedList(LinkedListInfo *list_info,
%        const void *datum)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o datum: The datum.
%
*/
MagickExport MagickBooleanType AppendElementToLinkedList(
  LinkedListInfo *list_info,const void *datum)
{
  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (list_info->elements == list_info->maximum_elements)
    return(MagickFalse);
  next=(ElementInfo *) AcquireMagickMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(MagickFalse);
  next->datum=(void *) datum;
  next->next=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->next == (ElementInfo *) NULL)
    list_info->next=next;
  if (list_info->elements == 0)
    list_info->head=next;
  else
    list_info->tail->next=next;
  list_info->tail=next;
  list_info->elements++;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C l e a r L i n k e d L i s t                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ClearLinkedList() clears all the elements from the linked-list.
%
%  The format of the ClearLinkedList method is:
%
%      void ClearLinkedList(LinkedListInfo *list_info,
%        void *(*liberate_datum)(void *))
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o liberate_datum: The datum deallocation method; typically
%      RelinquishMagickMemory().
%
*/
MagickExport void ClearLinkedList(LinkedListInfo *list_info,
  void *(*liberate_datum)(void *))
{
  ElementInfo
    *element;

  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  AcquireSemaphoreInfo(&list_info->semaphore);
  next=list_info->head;
  while (next != (ElementInfo *) NULL)
  {
    if (liberate_datum != (void *(*)(void *)) NULL)
      next->datum=liberate_datum(next->datum);
    element=next;
    next=next->next;
    element=(ElementInfo *) RelinquishMagickMemory(element);
  }
  list_info->head=(ElementInfo *) NULL;
  list_info->tail=(ElementInfo *) NULL;
  list_info->next=(ElementInfo *) NULL;
  list_info->elements=0;
  RelinquishSemaphoreInfo(&list_info->semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y H a s h m a p                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyHashmap() frees the hash-map and all associated resources.
%
%  The format of the DestroyHashmap method is:
%
%      DestroyHashmap(struct _HashmapInfo *hashmap_info)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
*/
MagickExport void DestroyHashmap(HashmapInfo *hashmap_info)
{
  register EntryInfo
    *entry;

  register long
    i;

  struct _LinkedListInfo
    *list_info;

  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  AcquireSemaphoreInfo(&hashmap_info->semaphore);
  for (i=0; i < (long) hashmap_info->maximum_entries; i++)
  {
    list_info=hashmap_info->map[i];
    if (list_info != (LinkedListInfo *) NULL)
      {
        ResetLinkedListIterator(list_info);
        entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
        while (entry != (EntryInfo *) NULL)
        {
          if (hashmap_info->liberate_key != (void *(*)(void *)) NULL)
            entry->key=hashmap_info->liberate_key(entry->key);
          if (hashmap_info->liberate_datum != (void *(*)(void *)) NULL)
            entry->datum=hashmap_info->liberate_datum(entry->datum);
          entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
        }
      }
    if (list_info != (LinkedListInfo *) NULL)
      DestroyLinkedList(list_info,RelinquishMagickMemory);
  }
  hashmap_info->map=(struct _LinkedListInfo **)
    RelinquishMagickMemory(hashmap_info->map);
  RelinquishSemaphoreInfo(&hashmap_info->semaphore);
  DestroySemaphoreInfo(&hashmap_info->semaphore);
  hashmap_info=(HashmapInfo *) RelinquishMagickMemory(hashmap_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y L i n k e d L i s t                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyLinkedList() frees the linked-list and all associated resources.
%
%  The format of the DestroyLinkedList method is:
%
%      DestroyLinkedList(LinkedListInfo *list_info,
%        void *(*liberate_datum)(void *))
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o liberate_datum: The datum deallocation method;  typically
%      RelinquishMagickMemory().
%
*/
MagickExport void DestroyLinkedList(LinkedListInfo *list_info,
  void *(*liberate_datum)(void *))
{
  ElementInfo
    *entry;

  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  AcquireSemaphoreInfo(&list_info->semaphore);
  next=list_info->head;
  while (next != (ElementInfo *) NULL)
  {
    if (liberate_datum != (void *(*)(void *)) NULL)
      next->datum=liberate_datum(next->datum);
    entry=next;
    next=next->next;
    entry=(ElementInfo *) RelinquishMagickMemory(entry);
  }
  RelinquishSemaphoreInfo(&list_info->semaphore);
  DestroySemaphoreInfo(&list_info->semaphore);
  list_info=(LinkedListInfo *) RelinquishMagickMemory(list_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t E n t r y F r o m H a s h m a p                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetEntryFromHashmap() gets an entry from the hash-map by its key.
%
%  The format of the GetEntryFromHashmap method is:
%
%      void *GetEntryFromHashmap(const HashmapInfo *hashmap_info,void *key)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
%    o key: The key.
%
*/
MagickExport void *GetEntryFromHashmap(const HashmapInfo *hashmap_info,
  const void *key)
{
  struct _LinkedListInfo
    *list_info;

  register EntryInfo
    *entry;

  size_t
    hash;

  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (key == NULL)
    return((void *) NULL);
  hash=hashmap_info->hash(key);
  list_info=hashmap_info->map[hash % hashmap_info->maximum_entries];
  if (list_info != (LinkedListInfo *) NULL)
    {
      ResetLinkedListIterator(list_info);
      entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      while (entry != (EntryInfo *) NULL)
      {
        if (entry->hash == hash)
          return(entry->datum);
        entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      }
    }
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t E l e m e n t F r o m L i n k e d L i s t                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetElementInLinkedList() gets an element from the linked-list by the
%  specified location.
%
%  The format of the GetElementInLinkedList method is:
%
%      void *GetElementInLinkedList(const LinkedListInfo *list_info,
%        const unsigned long index)
%
%  A description of each parameter follows:
%
%    o list_info: The linked_list info.
%
%    o index: The list index.
%
*/
MagickExport void *GetElementFromLinkedList(const LinkedListInfo *list_info,
  const unsigned long index)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (index >= list_info->elements)
    return((void *) NULL);
  if (index == 0)
    return(list_info->head->datum);
  if (index == (list_info->elements-1))
    return(list_info->tail->datum);
  next=list_info->head;
  for (i=0; i < (long) index; i++)
    next=next->next;
  return(next->datum);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t L a s t E n t r y I n L i n k e d L i s t                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetLastElementInLinkedList() gets the last element in the linked-list.
%
%  The format of the GetLastElementInLinkedList method is:
%
%      void *GetLastElementInLinkedList(const LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked_list info.
%
*/
MagickExport void *GetLastElementInLinkedList(const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (list_info->elements == 0)
    return((void *) NULL);
  return(list_info->tail->datum);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N e x t E n t r y I n H a s h m a p                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNextEntryInHashmap() gets the next entry in the hash-map.
%
%  The format of the GetNextEntryInHashmap method is:
%
%      void *GetNextEntryInHashmap(HashmapInfo *hashmap_info)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
%    o key: The key.
%
*/
MagickExport void *GetNextEntryInHashmap(HashmapInfo *hashmap_info)
{
  register EntryInfo
    *entry;

  struct _LinkedListInfo
    *list_info;

  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  while (hashmap_info->next < hashmap_info->maximum_entries)
  {
    list_info=hashmap_info->map[hashmap_info->next];
    if (list_info != (LinkedListInfo *) NULL)
      {
        if (hashmap_info->head_of_list == MagickFalse)
          {
            ResetLinkedListIterator(list_info);
            hashmap_info->head_of_list=MagickTrue;
          }
      entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      if (entry != (EntryInfo *) NULL)
        return(entry->key);
      hashmap_info->head_of_list=MagickFalse;
    }
    hashmap_info->next++;
  }
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N e x t E l e m e n t I n L i n k e d L i s t                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNextElementInLinkedList() gets the next element in the linked-list.
%
%  The format of the GetNextElementInLinkedList method is:
%
%      void *GetNextElementInLinkedList(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
*/
MagickExport void *GetNextElementInLinkedList(LinkedListInfo *list_info)
{
  void
    *datum;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (list_info->next == (ElementInfo *) NULL)
    return((void *) NULL);
  datum=list_info->next->datum;
  list_info->next=list_info->next->next;
  return(datum);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N u m b e r O f E n t r i e s I n H a s h m a p                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNumberOfEntriesInHashmap() returns the number of entries in the hash-map.
%
%  The format of the GetNumberOfEntriesInHashmap method is:
%
%      unsigned long GetNumberOfEntriesInHashmap(const HashmapInfo *hashmap_info)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
*/
MagickExport unsigned long GetNumberOfEntriesInHashmap(
  const HashmapInfo *hashmap_info)
{
  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  return(hashmap_info->entries);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N u m b e r O f E l e m e n t s I n L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNumberOfElementsInLinkedList() returns the number of entries in the
%  linked-list.
%
%  The format of the GetNumberOfElementsInLinkedList method is:
%
%      unsigned long GetNumberOfElementsInLinkedList(
%        const LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
*/
MagickExport unsigned long GetNumberOfElementsInLinkedList(
  const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  return(list_info->elements);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   H a s h P o i n t e r T y p e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Specify the HashPointerType() method in NewHashmap() to find an entry
%  in a hash-map based on the address of a pointer.
%
%  The format of the HashPointerType method is:
%
%      unsigned long HashPointerType(const void *pointer)
%
%  A description of each parameter follows:
%
%    o pointer: compute the hash entry location from this pointer address.
%
*/
MagickExport size_t HashPointerType(const void *pointer)
{
  size_t
    hash;

  hash=(size_t) pointer;
  hash+=(~(hash << 9));
  hash^=(hash >> 14);
  hash+=(hash << 4);
  hash^=(hash >> 10);
  return(hash);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   H a s h S t r i n g T y p e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Specify the HashStringType() method in NewHashmap() to find an entry
%  in a hash-map based on the contents of a string.
%
%  The format of the HashStringType method is:
%
%      size_t HashStringType(const void *string)
%
%  A description of each parameter follows:
%
%    o string: compute the hash entry location from this string.
%
*/
MagickExport size_t HashStringType(const void *string)
{
  register long
    i;

  SignatureInfo
    signature_info;

  size_t
    hash;

  GetSignatureInfo(&signature_info);
  UpdateSignature(&signature_info,(const unsigned char *) string,
    strlen((char *) string));
  FinalizeSignature(&signature_info);
  hash=0;
  for (i=0; i < 8; i++)
    hash^=signature_info.digest[i];
  return(hash);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   H a s h S t r i n g I n f o T y p e                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Specify the HashStringInfoType() method in NewHashmap() to find an entry
%  in a hash-map based on the contents of a string.
%
%  The format of the HashStringInfoType method is:
%
%      size_t HashStringInfoType(const void *string)
%
%  A description of each parameter follows:
%
%    o string: compute the hash entry location from this string.
%
*/
MagickExport size_t HashStringInfoType(const void *string)
{
  register long
    i;

  SignatureInfo
    signature_info;

  size_t
    hash;

  GetSignatureInfo(&signature_info);
  UpdateSignature(&signature_info,((StringInfo *) string)->datum,
    ((StringInfo *) string)->length);
  FinalizeSignature(&signature_info);
  hash=0;
  for (i=0; i < 8; i++)
    hash^=signature_info.digest[i];
  return(hash);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I n s e r t E l e m e n t I n L i n k e d L i s t                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InsertElementInLinkedList() inserts an element in the linked-list at the
%  specified location.
%
%  The format of the InsertElementInLinkedList method is:
%
%      MagickBooleanType InsertElementInLinkedList(ListInfo *list_info,
%        const unsigned long index,const void *datum)
%
%  A description of each parameter follows:
%
%    o list_info: The hashmap info.
%
%    o index: The index.
%
%    o datum: The datum.
%
*/
MagickExport MagickBooleanType InsertElementInLinkedList(
  LinkedListInfo *list_info,const unsigned long index,const void *datum)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (datum == NULL)
    return(MagickFalse);
  if ((index > list_info->elements) ||
      (list_info->elements == list_info->maximum_elements))
    return(MagickFalse);
  next=(ElementInfo *) AcquireMagickMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(MagickFalse);
  next->datum=(void *) datum;
  next->next=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->elements == 0)
    {
      if (list_info->next == (ElementInfo *) NULL)
        list_info->next=next;
      list_info->head=next;
      list_info->tail=next;
    }
  else
    {
      if (index == 0)
        {
          if (list_info->next == list_info->head)
            list_info->next=next;
           next->next=list_info->head;
           list_info->head=next;
        }
      else
        if (index == list_info->elements)
          {
            if (list_info->next == (ElementInfo *) NULL)
              list_info->next=next;
             list_info->tail->next=next;
             list_info->tail=next;
          }
        else
          {
            ElementInfo
              *element;

            element=list_info->head;
            next->next=element->next;
            for (i=1; i < (long) index; i++)
            {
              element=element->next;
              next->next=element->next;
            }
            next=next->next;
            element->next=next;
            if (list_info->next == next->next)
              list_info->next=next;
          }
    }
  list_info->elements++;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I n s e r t E l e m e n t I n S o r t e d L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InsertElementInSortedLinkedList() inserts an element in the sorted
%  linked-list.
%
%  The format of the InsertElementInSortedLinkedList method is:
%
%      MagickBooleanType InsertElementInLinkedSortedList(ListInfo *list_info,
%        int (*compare)(const void *, const void *),void **replace,
%        const void *datum)
%
%  A description of each parameter follows:
%
%    o list_info: The hashmap info.
%
%    o index: The index.
%
%    o compare: The compare method.
%
%    o replace: return previous element here.
%
%    o datum: The datum.
%
*/
MagickExport MagickBooleanType InsertElementInSortedLinkedList(
  LinkedListInfo *list_info,int (*compare)(const void *, const void *),
  void **replace,const void *datum)
{
  ElementInfo
    *element;

  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if ((compare == NULL) || (datum == NULL))
    return(MagickFalse);
  if (list_info->elements == list_info->maximum_elements)
    return(MagickFalse);
  next=(ElementInfo *) AcquireMagickMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(MagickFalse);
  next->datum=(void *) datum;
  element=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  next->next=list_info->head;
  while (next->next != (ElementInfo *) NULL)
  {
    i=compare(datum,next->next->datum);
    if ((i < 0) || ((replace != (void **) NULL) && (i == 0)))
      {
        if (i == 0)
          {
            *replace=next->next->datum;
            next->next=next->next->next;
            if (element != (ElementInfo *) NULL)
              element->next=(ElementInfo *)
                RelinquishMagickMemory(element->next);
            list_info->elements--;
          }
        if (element != (ElementInfo *) NULL)
          element->next=next;
        else
          list_info->head=next;
        break;
      }
    element=next->next;
    next->next=next->next->next;
  }
  if (next->next == (ElementInfo *) NULL)
    {
      if (element != (ElementInfo *) NULL)
        element->next=next;
      else
        list_info->head=next;
      list_info->tail=next;
    }
  list_info->elements++;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s H a s h m a p E m p t y                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsHashmapEmpty() returns MagickTrue if the hash-map is empty.
%
%  The format of the IsHashmapEmpty method is:
%
%      MagickBooleanType IsHashmapEmpty(const HashmapInfo *hashmap_info)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
*/
MagickExport MagickBooleanType IsHashmapEmpty(const HashmapInfo *hashmap_info)
{
  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  return((MagickBooleanType) (hashmap_info->entries == 0));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s L i n k e d L i s t E m p t y                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsLinkedListEmpty() returns MagickTrue if the linked-list is empty.
%
%  The format of the IsLinkedListEmpty method is:
%
%      MagickBooleanType IsLinkedListEmpty(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
*/
MagickExport MagickBooleanType IsLinkedListEmpty(
  const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  return((MagickBooleanType) (list_info->elements == 0));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   L i n k e d L i s t T o A r r a y                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LinkedListToArray() converts the linked-list to an array.
%
%  The format of the LinkedListToArray method is:
%
%      MagickBooleanType LinkedListToArray(LinkedListInfo *list_info,
%        void **array)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o array: The array.
%
*/
MagickExport MagickBooleanType LinkedListToArray(LinkedListInfo *list_info,
  void **array)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (array == (void **) NULL)
    return(MagickFalse);
  next=list_info->head;
  for (i=0; next != (ElementInfo *) NULL; i++)
  {
    array[i]=next->datum;
    next=next->next;
  }
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   N e w H a s h m a p I n f o                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  NewHashmap() returns a pointer to a HashmapInfo structure initialized
%  to default values.  The maximum number of elements that can be stored in the
%  hash is maximum elements squared.
%
%  The format of the NewHashmap method is:
%
%      HashmapInfo *NewHashmap(const unsigned long maximum_elements,
%        size_t (*hash)(const void *),void *(*liberate_key)(void *),
%        void *(*liberate_datum)(void *))
%
%  A description of each parameter follows:
%
%    o maximum_elements: The maximum number elements in the hash-map:
%      SmallHashmapSize, MediumHashmapSize, or LargeHashmapSize.
%
%    o hash: The hash method, typically HashPointerType(), HashStringType(),
%      HashStringInfoType().
%
%    o liberate_key: The key deallocation method, typically
%      RelinquishMagickMemory(), called whenever a key is removed from the
%      hash-map.
%
%    o liberate_datum: The datum deallocation method;  typically
%      RelinquishMagickMemory(), called whenever a datum object is removed from
%      the hash-map.
%
*/
MagickExport HashmapInfo *NewHashmap(const unsigned long maximum_elements,
  size_t (*hash)(const void *),void *(*liberate_key)(void *),
  void *(*liberate_datum)(void *))
{
  struct _HashmapInfo
    *hashmap_info;

  hashmap_info=(struct _HashmapInfo *)
    AcquireMagickMemory(sizeof(*hashmap_info));
  if (hashmap_info == (struct _HashmapInfo *) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  hashmap_info->hash=hash ? hash : HashPointerType;
  hashmap_info->liberate_key=liberate_key;
  hashmap_info->liberate_datum=liberate_datum;
  hashmap_info->entries=0;
  hashmap_info->maximum_entries=maximum_elements;
  hashmap_info->map=(struct _LinkedListInfo **)
    AcquireMagickMemory((size_t) maximum_elements*sizeof(*hashmap_info->map));
  if (hashmap_info->map == (struct _LinkedListInfo **) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  (void) ResetMagickMemory(hashmap_info->map,'\0',(size_t) maximum_elements*
    sizeof(*hashmap_info->map));
  hashmap_info->debug=IsEventLogging();
  hashmap_info->semaphore=(SemaphoreInfo *) NULL;
  hashmap_info->signature=MagickSignature;
  return(hashmap_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   N e w L i n k e d L i s t I n f o                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  NewLinkedList() returns a pointer to a LinkedListInfo structure
%  initialized to default values.
%
%  The format of the Acquirestruct LinkedListInfomethod is:
%
%      LinkedListInfo *NewLinkedList(const unsigned long maximum_elements)
%
%  A description of each parameter follows:
%
%    o maximum_elements: The maximum number of elements in the list.
%
*/
MagickExport LinkedListInfo *NewLinkedList(const unsigned long maximum_elements)
{
  struct _LinkedListInfo
    *list_info;

  list_info=(LinkedListInfo *) AcquireMagickMemory(sizeof(*list_info));
  if (list_info == (LinkedListInfo *) NULL)
    ThrowMagickFatalException(ResourceLimitFatalError,"MemoryAllocationFailed",
      strerror(errno));
  list_info->maximum_elements=
    maximum_elements == 0 ? (unsigned long) (~0) : maximum_elements;
  list_info->elements=0;
  list_info->head=(ElementInfo *) NULL;
  list_info->tail=(ElementInfo *) NULL;
  list_info->next=(ElementInfo *) NULL;
  list_info->debug=IsEventLogging();
  list_info->semaphore=(SemaphoreInfo *) NULL;
  list_info->signature=MagickSignature;
  return(list_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P u t E n t r y I n H a s h m a p                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PutEntryInHashmap() puts an entry in the hash-map.  If the key hash
%  value already exists in the map it is first removed.
%
%  The format of the PutEntryInHashmap method is:
%
%      MagickBooleanType PutEntryInHashmap(HashmapInfo *hashmap_info,
%        const void *key,const void *datum)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
%    o key: The key.
%
%    o datum: The datum.
%
*/
MagickExport MagickBooleanType PutEntryInHashmap(HashmapInfo *hashmap_info,
  const void *key,const void *datum)
{
  EntryInfo
    *entry,
    *next;

  register unsigned long
    i;

  struct _LinkedListInfo
    *list_info;

  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if ((key == (void *) NULL) || (datum == (void *) NULL))
    return(MagickFalse);
  next=(EntryInfo *) AcquireMagickMemory(sizeof(*next));
  if (next == (EntryInfo *) NULL)
    return(MagickFalse);
  AcquireSemaphoreInfo(&hashmap_info->semaphore);
  next->hash=hashmap_info->hash(key);
  next->key=(void *) key;
  next->datum=(void *) datum;
  list_info=hashmap_info->map[next->hash % hashmap_info->maximum_entries];
  if (list_info == (LinkedListInfo *) NULL)
    {
      list_info=NewLinkedList(hashmap_info->maximum_entries);
      if (list_info == (LinkedListInfo *) NULL)
        {
          next=(EntryInfo *) RelinquishMagickMemory(next);
          RelinquishSemaphoreInfo(&hashmap_info->semaphore);
          return(MagickFalse);
        }
      hashmap_info->map[next->hash % hashmap_info->maximum_entries]=list_info;
    }
  else
    {
      ResetLinkedListIterator(list_info);
      entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      for (i=0; entry != (EntryInfo *) NULL; i++)
      {
        if (entry->hash == next->hash)
          {
            (void) RemoveElementFromLinkedList(list_info,i);
            if (hashmap_info->liberate_key != (void *(*)(void *)) NULL)
              entry->key=hashmap_info->liberate_key(entry->key);
            if (hashmap_info->liberate_datum != (void *(*)(void *)) NULL)
              entry->datum=hashmap_info->liberate_datum(entry->datum);
            entry=(EntryInfo *) RelinquishMagickMemory(entry);
            break;
          }
        entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      }
    }
  if (InsertElementInLinkedList(list_info,0,next) == MagickFalse)
    {
      next=(EntryInfo *) RelinquishMagickMemory(next);
      RelinquishSemaphoreInfo(&hashmap_info->semaphore);
      return(MagickFalse);
    }
  hashmap_info->entries++;
  RelinquishSemaphoreInfo(&hashmap_info->semaphore);
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E l e m e n t B y V a l u e F r o m L i n k e d L i s t       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveElementByValueFromLinkedList() removes an element from the linked-list
%  by value.
%
%  The format of the RemoveElementByValueFromLinkedList method is:
%
%      void *RemoveElementByValueFromLinkedList(LinkedListInfo *list_info,
%        const void *datum)
%
%  A description of each parameter follows:
%
%    o list_info: The list info.
%
%    o datum: The datum.
%
*/
MagickExport void *RemoveElementByValueFromLinkedList(LinkedListInfo *list_info,
  const void *datum)
{
  ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if ((list_info->elements == 0) || (datum == NULL))
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (datum == list_info->head->datum)
    {
      if (list_info->next == list_info->head)
        list_info->next=list_info->head->next;
      next=list_info->head;
      list_info->head=list_info->head->next;
      next=(ElementInfo *) RelinquishMagickMemory(next);
    }
  else
    {
      ElementInfo
        *element;

      next=list_info->head;
      while ((next->next != (ElementInfo *) NULL) &&
             (next->next->datum != datum))
        next=next->next;
      if (next->next == (ElementInfo *) NULL)
        {
          RelinquishSemaphoreInfo(&list_info->semaphore);
          return((void *) NULL);
        }
      element=next->next;
      next->next=element->next;
      if (element == list_info->tail)
        list_info->tail=next;
      if (list_info->next == element)
        list_info->next=element->next;
      element=(ElementInfo *) RelinquishMagickMemory(element);
    }
  list_info->elements--;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return((void *) datum);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E l e m e n t F r o m L i n k e d L i s t                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveElementFromLinkedList() removes an element from the linked-list at the
%  specified location.
%
%  The format of the RemoveElementFromLinkedList method is:
%
%      void *RemoveElementFromLinkedList(LinkedListInfo *list_info,
%        const unsigned long index)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o index: The index.
%
*/
MagickExport void *RemoveElementFromLinkedList(LinkedListInfo *list_info,
  const unsigned long index)
{
  ElementInfo
    *next;

  register long
    i;

  void
    *entry;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (index >= list_info->elements)
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (index == 0)
    {
      if (list_info->next == list_info->head)
        list_info->next=list_info->head->next;
      entry=list_info->head->datum;
      next=list_info->head;
      list_info->head=list_info->head->next;
      next=(ElementInfo *) RelinquishMagickMemory(next);
    }
  else
    {
      ElementInfo
        *element;

      next=list_info->head;
      for (i=1; i < (long) index; i++)
        next=next->next;
      element=next->next;
      next->next=element->next;
      if (element == list_info->tail)
        list_info->tail=next;
      if (list_info->next == element)
        list_info->next=element->next;
      entry=element->datum;
      element=(ElementInfo *) RelinquishMagickMemory(element);
    }
  list_info->elements--;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E n t r y F r o m H a s h m a p                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveEntryFromHashmap() removes an entry from the hash-map by its key.
%
%  The format of the RemoveEntryFromHashmap method is:
%
%      void *RemoveEntryFromHashmap(HashmapInfo *hashmap_info,void *key)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
%    o key: The key.
%
*/
MagickExport void *RemoveEntryFromHashmap(HashmapInfo *hashmap_info,
  const void *key)
{
  EntryInfo
    *entry;

  register unsigned long
    i;

  size_t
    hash;

  struct _LinkedListInfo
    *list_info;

  void
    *datum;

  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (key == NULL)
    return((void *) NULL);
  AcquireSemaphoreInfo(&hashmap_info->semaphore);
  hash=hashmap_info->hash(key);
  list_info=hashmap_info->map[hash % hashmap_info->maximum_entries];
  if (list_info != (LinkedListInfo *) NULL)
    {
      ResetLinkedListIterator(list_info);
      entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      for (i=0; entry != (EntryInfo *) NULL; i++)
      {
        if (entry->hash == hash)
          {
            entry=(EntryInfo *) RemoveElementFromLinkedList(list_info,i);
            if (entry == (EntryInfo *) NULL)
              {
                RelinquishSemaphoreInfo(&hashmap_info->semaphore);
                return((void *) NULL);
              }
            if (hashmap_info->liberate_key != (void *(*)(void *)) NULL)
              entry->key=hashmap_info->liberate_key(entry->key);
            datum=entry->datum;
            entry=(EntryInfo *) RelinquishMagickMemory(entry);
            hashmap_info->entries--;
            RelinquishSemaphoreInfo(&hashmap_info->semaphore);
            return(datum);
          }
        entry=(EntryInfo *) GetNextElementInLinkedList(list_info);
      }
    }
  RelinquishSemaphoreInfo(&hashmap_info->semaphore);
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e L a s t E l e m e n t F r o m L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveLastElementFromLinkedList() removes the last element from the
%  linked-list.
%
%  The format of the RemoveLastElementFromLinkedList method is:
%
%      void *RemoveLastElementFromLinkedList(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
*/
MagickExport void *RemoveLastElementFromLinkedList(LinkedListInfo *list_info)
{
  void
    *entry;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  if (list_info->elements == 0)
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->next == list_info->tail)
    list_info->next=(ElementInfo *) NULL;
  if (list_info->elements == 1UL)
    {
      entry=list_info->head->datum;
      list_info->head=(ElementInfo *) NULL;
      list_info->tail=(ElementInfo *) NULL;
    }
  else
    {
      ElementInfo
        *next;

      entry=list_info->tail->datum;
      next=list_info->head;
      while (next->next != list_info->tail)
        next=next->next;
      list_info->tail=(ElementInfo *) RelinquishMagickMemory(list_info->tail);
      list_info->tail=next;
      next->next=(ElementInfo *) NULL;
    }
  list_info->elements--;
  RelinquishSemaphoreInfo(&list_info->semaphore);
  return(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e s e t H a s h m a p I t e r a t o r                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ResetHashmapIterator() resets the hash-map iterator.  Use it in conjunction
%  with GetNextEntryInHashmap() to iterate over all the keys in the hash-map.
%
%  The format of the ResetHashmapIterator method is:
%
%      ResetHashmapIterator(HashmapInfo *hashmap_info)
%
%  A description of each parameter follows:
%
%    o hashmap_info: The hashmap info.
%
*/
MagickExport void ResetHashmapIterator(HashmapInfo *hashmap_info)
{
  assert(hashmap_info != (HashmapInfo *) NULL);
  assert(hashmap_info->signature == MagickSignature);
  if (hashmap_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  hashmap_info->next=0;
  hashmap_info->head_of_list=MagickFalse;
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e s e t L i n k e d L i s t I t e r a t o r                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ResetLinkedListIterator() resets the lined-list iterator.  Use it in
%  conjunction with GetNextElementInLinkedList() to iterate over all the values
%  in the linked-list.
%
%  The format of the ResetLinkedListIterator method is:
%
%      ResetLinkedListIterator(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
*/
MagickExport void ResetLinkedListIterator(LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),"...");
  list_info->next=list_info->head;
}
