/*
 * Copyright 2003 Digi International (www.digi.com)
 *      Scott H Kilau <Scott_Kilau at digi dot com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *	NOTE: THIS IS A SHARED HEADER. DO NOT CHANGE CODING STYLE!!!
 *
 *************************************************************************
 *
 * Driver includes
 *
 *************************************************************************/

#ifndef __DGAP_DRIVER_H
#define __DGAP_DRIVER_H

#define TRC_TO_CONSOLE 1

#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/errno.h>
#include <linux/socket.h>
#include <linux/in.h>
#include <linux/ioport.h>
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <asm/irq.h>

#include <linux/pci.h>

#include <linux/major.h>
#include <linux/signal.h>

#include <linux/mm.h>

#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/stddef.h>
#include <asm/atomic.h>
#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/byteorder.h>
#include <asm/system.h>

#include <linux/tty.h>
#include <linux/termios.h>
#include <linux/poll.h>
#include <linux/types.h>

#include <linux/serial.h>	/* for struct async_serial */

#include "dgap_types.h"		/* Additional types needed by header files */
#include "digi.h"

#include "dgap_pci.h"
#include "dgap_fep5.h"
#include "dgap_conf.h"

/*************************************************************************
 *
 * Driver defines
 *
 *************************************************************************/

/*
 * Driver identification, error and debugging statments
 *
 * In theory, you can change all occurances of "digi" in the next
 * three lines, and the driver printk's will all automagically change.
 *
 * APR((fmt, args, ...));	Always prints message
 * DPR((fmt, args, ...));	Only prints if DGAP_TRACER is defined at
 *				  compile time and dgap_debug!=0
 */
#define	PROCSTR		"dgap"			/* /proc entries	 */
#define	DEVSTR		"/dev/dg/dgap"		/* /dev entries		 */
#define	DRVSTR		"dgap"			/* Driver name string 
						 * displayed by APR	 */
#define	APR(args)	do { PRINTF_TO_KMEM(args); printk(DRVSTR": "); printk args; \
			   } while (0)
#define	RAPR(args)	do { PRINTF_TO_KMEM(args); printk args; } while (0)

/*
 * Debugging levels can be set using debug insmod variable
 * They can also be compiled out completely.
 */

#define	DBG_INIT		(dgap_debug & 0x01)
#define	DBG_BASIC		(dgap_debug & 0x02)
#define	DBG_CORE		(dgap_debug & 0x04)

#define	DBG_OPEN		(dgap_debug & 0x08)
#define	DBG_CLOSE		(dgap_debug & 0x10)
#define	DBG_READ		(dgap_debug & 0x20)
#define	DBG_WRITE		(dgap_debug & 0x40)

#define	DBG_IOCTL		(dgap_debug & 0x80)

#define	DBG_PROC		(dgap_debug & 0x100)
#define	DBG_PARAM		(dgap_debug & 0x200)
#define	DBG_PSCAN		(dgap_debug & 0x400)
#define	DBG_EVENT		(dgap_debug & 0x800)

#define	DBG_DRAIN		(dgap_debug & 0x1000)
#define	DBG_CARR		(dgap_debug & 0x2000)

#define	DBG_MGMT		(dgap_debug & 0x4000)


#if defined(DGAP_TRACER)

# if defined(TRC_TO_KMEM)
/* Choose one: */
#  define TRC_ON_OVERFLOW_WRAP_AROUND
#  undef  TRC_ON_OVERFLOW_SHIFT_BUFFER
# endif //TRC_TO_KMEM

# define TRC_MAXMSG		1024
# define TRC_OVERFLOW		"(OVERFLOW)"
# define TRC_DTRC		"/usr/bin/dtrc"

#if defined TRC_TO_CONSOLE
#define PRINTF_TO_CONSOLE(args) { printk(DRVSTR": "); printk args; }
#else //!defined TRACE_TO_CONSOLE
#define PRINTF_TO_CONSOLE(args)
#endif

#if defined TRC_TO_KMEM
#define PRINTF_TO_KMEM(args) dgap_tracef args 
#else //!defined TRC_TO_KMEM
#define PRINTF_TO_KMEM(args)
#endif

#define	TRC(args)	{ PRINTF_TO_KMEM(args); PRINTF_TO_CONSOLE(args) }

# define DPR_INIT(ARGS)		if (DBG_INIT) TRC(ARGS)
# define DPR_BASIC(ARGS)	if (DBG_BASIC) TRC(ARGS)
# define DPR_CORE(ARGS)		if (DBG_CORE) TRC(ARGS)
# define DPR_OPEN(ARGS)		if (DBG_OPEN)  TRC(ARGS)
# define DPR_CLOSE(ARGS)	if (DBG_CLOSE)  TRC(ARGS)
# define DPR_READ(ARGS)		if (DBG_READ)  TRC(ARGS)
# define DPR_WRITE(ARGS)	if (DBG_WRITE) TRC(ARGS)
# define DPR_IOCTL(ARGS)	if (DBG_IOCTL) TRC(ARGS)
# define DPR_PROC(ARGS)		if (DBG_PROC)  TRC(ARGS)
# define DPR_PARAM(ARGS)	if (DBG_PARAM)  TRC(ARGS)
# define DPR_PSCAN(ARGS)	if (DBG_PSCAN)  TRC(ARGS)
# define DPR_EVENT(ARGS)	if (DBG_EVENT)  TRC(ARGS)
# define DPR_DRAIN(ARGS)	if (DBG_DRAIN)  TRC(ARGS)
# define DPR_CARR(ARGS)		if (DBG_CARR)  TRC(ARGS)
# define DPR_MGMT(ARGS)		if (DBG_MGMT)  TRC(ARGS)

# define DPR(ARGS)		if (dgap_debug) TRC(ARGS)
# define P(X)			dgap_tracef(#X "=%p\n", X)
# define X(X)			dgap_tracef(#X "=%x\n", X)

#else//!defined DGAP_TRACER

#define PRINTF_TO_KMEM(args)
# define TRC(ARGS)
# define DPR_INIT(ARGS)
# define DPR_BASIC(ARGS)
# define DPR_CORE(ARGS)
# define DPR_OPEN(ARGS)
# define DPR_CLOSE(ARGS)
# define DPR_READ(ARGS)
# define DPR_WRITE(ARGS)
# define DPR_IOCTL(ARGS)
# define DPR_PROC(ARGS)
# define DPR_PARAM(ARGS)
# define DPR_PSCAN(ARGS)
# define DPR_EVENT(ARGS)
# define DPR_DRAIN(ARGS)
# define DPR_CARR(ARGS)
# define DPR_MGMT(ARGS)

# define DPR(args)

#endif//DGAP_TRACER

/* Number of boards we support at once. */
#define	MAXBOARDS	32
#define	MAXPORTS	224
#define MAXTTYNAMELEN	200

/* Our 3 magic numbers for our board, channel and unit structs */
#define DGAP_BOARD_MAGIC	0x5c6df104
#define DGAP_CHANNEL_MAGIC	0x6c6df104
#define DGAP_UNIT_MAGIC		0x7c6df104

/* Serial port types */
#define DGAP_SERIAL		0
#define DGAP_PRINT		1

#define	SERIAL_TYPE_NORMAL	1

/* 4 extra for alignment play space */
#define WRITEBUFLEN		((4096) + 4)
#define MYFLIPLEN		N_TTY_BUF_SIZE

#define SBREAK_TIME 0x25
#define U2BSIZE 0x400
#define dgap_jiffies_from_ms(a) (((a) * HZ) / 1000)

/*
 * Our major for the mgmt devices.
 *
 * We can use 22, because Digi was allocated 22 and 23 for the epca driver.
 * 22 has now become obsolete now that the "cu" devices have 
 * been removed from 2.6.
 * Also, this *IS* the epca driver, just PCI only now.
 */
#ifndef DIGI_DGAP_MAJOR
# define DIGI_DGAP_MAJOR         22
#endif

/*
 * The parameters we use to define the periods of the moving averages.
 */
#define		MA_PERIOD	(HZ / 10)
#define		SMA_DUR		(1 * HZ)
#define		EMA_DUR		(1 * HZ)
#define		SMA_NPERIODS	(SMA_DUR / MA_PERIOD)
#define		EMA_NPERIODS	(EMA_DUR / MA_PERIOD)

/*
 * Define a local default termios struct. All ports will be created
 * with this termios initially.  This is the same structure that is defined
 * as the default in tty_io.c with the same settings overriden as in serial.c
 *
 * In short, this should match the internal serial ports' defaults.
 */
#define	DEFAULT_IFLAGS	(ICRNL | IXON)
#define	DEFAULT_OFLAGS	(OPOST | ONLCR)
#define	DEFAULT_CFLAGS	(B9600 | CS8 | CREAD | HUPCL | CLOCAL)
#define	DEFAULT_LFLAGS	(ISIG | ICANON | ECHO | ECHOE | ECHOK | \
			ECHOCTL | ECHOKE | IEXTEN)


/*************************************************************************
 *
 * Utility defines - should be defined elsewhere, but be paranoid
 *
 *************************************************************************/

#if !defined(KERNEL_VERSION) /* defined in version.h in later kernels */
# define KERNEL_VERSION(a,b,c)	(((a) << 16) + ((b) << 8) + (c))
#endif

/* The number of elements in an array */
#if !defined(asizeof)
# define asizeof(X)	(sizeof(X) / sizeof(X[0]))
#endif

/*
 * All the possible states the driver can be while being loaded.
 */
enum {
	DRIVER_INITIALIZED = 0,
	DRIVER_NEED_CONFIG_LOAD,
	DRIVER_REQUESTED_CONFIG,
	DRIVER_READY
};

/*
 * All the possible states the board can be while booting up.
 */
enum {
	BOARD_FAILED = 0,
	CONFIG_NOT_FOUND,
	BOARD_FOUND,
	NEED_RESET,
	FINISHED_RESET,
	NEED_CONFIG,
	FINISHED_CONFIG,
	NEED_DEVICE_CREATION,
	REQUESTED_DEVICE_CREATION,
	FINISHED_DEVICE_CREATION,
	NEED_BIOS_LOAD,
	REQUESTED_BIOS,
	WAIT_BIOS_LOAD,
	FINISHED_BIOS_LOAD,
	NEED_FEP_LOAD,
	REQUESTED_FEP,
	WAIT_FEP_LOAD,
	FINISHED_FEP_LOAD,
	BOARD_READY
};

/*
 * All the possible states that a requested concentrator image can be in.
 */
enum {
	NO_PENDING_CONCENTRATOR_REQUESTS = 0,
	NEED_CONCENTRATOR,
	REQUESTED_CONCENTRATOR
};

extern char *dgap_state_text[];
extern char *dgap_driver_state_text[];


/*************************************************************************
 *
 * Some API's changed at linux version 2.1.x.  I'm not sure exactly
 * when they changed during the 2.1.x development, but it doesn't
 * matter to us.  Create some macros to preserve compatibility.
 *
 * Here you can see what has changed between 2.0 and 2.1+. If you
 * are reading/writing this file, you should be well aware of these
 * macros or you won't be able to follow whats going on.  The
 * alternative was #ifdef spaghetti.
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
# include <asm/irq.h>
# include <asm/bitops.h>
# include <asm/io.h>
# include <asm/uaccess.h>
# include <linux/gfp.h>

# define DGAP_get_free_page(x)		(get_zeroed_page(GFP_ATOMIC))
# define DGAP_VMALLOC(x)		(kmalloc((x), GFP_ATOMIC))
# define DGAP_VFREE(x)			(kfree(x))

# define DGAP_MAJOR(x)			(imajor(x))
# define DGAP_MINOR(x)			(iminor(x))
# define DGAP_TTY_MAJOR(x)		(MAJOR(tty_devnum(tty)))
# define DGAP_TTY_MINOR(tty)		(MINOR(tty_devnum(tty)))

# define DGAP_NUM_CPUS			(num_online_cpus())

# define DGAP_MOD_INC_USE_COUNT(rtn)	(rtn = 1)
# define DGAP_MOD_DEC_USE_COUNT		

# define DGAP_IRQ_RETURN(x)		return x;


# define DGAP_SPINLOCK_INIT(x)		spin_lock_init(&(x))

# define DGAP_IOREMAP(ADDR, LEN)	ioremap(ADDR, LEN)
# define DGAP_IOUNMAP(ADDR)		iounmap(ADDR)
# define GET_USER(A1,A2)		get_user(A1,(unsigned int *)A2)
# define PUT_USER(A1,A2)		put_user(A1,(unsigned long *)A2)

# define READ_PROTO	struct file *file, char *buf, size_t count,loff_t *ppos
# define READ_ARGS	file, buf, count, ppos
# define READ_RETURN_T	ssize_t

# define WRITE_PROTO	struct file *file, const char *buf, size_t count, \
			loff_t *ppos
# define WRITE_ARGS	file, buf, count, ppos
# define WRITE_RETURN_T	ssize_t

# define CLOSE_RETURN_T		int
# define CLOSE_RETURN(X)	return(X)

# define PARM_I_RDEV		file->f_dentry->d_inode->i_rdev
# define GET_POS()		*ppos
# define ADD_POS(AMT)		*ppos += (AMT)
# define SET_POS(AMT)		*ppos = (AMT)

# define DGAP_PCI_IRQ_TYPE	uint
# define PCI_PRESENT()		pci_present()

# define PARM_STR(VAR, INIT, DESC) \
		static char *VAR = INIT; \
		char *dgap_##VAR; \
		MODULE_PARM(VAR, "s"); \
		MODULE_PARM_DESC(VAR, DESC);

# define PARM_INT(VAR, INIT, DESC) \
		static int VAR = INIT; \
		int dgap_##VAR; \
		MODULE_PARM(VAR, "i"); \
		MODULE_PARM_DESC(VAR, DESC);

# define PARM_ULONG(VAR, INIT, DESC) \
		static ulong VAR = INIT; \
		ulong dgap_##VAR; \
		MODULE_PARM(VAR, "l"); \
		MODULE_PARM_DESC(VAR, DESC);

# define DGAP_LOCK(x,y)		spin_lock_irqsave(&(x), y)
# define DGAP_UNLOCK(x,y)	spin_unlock_irqrestore(&(x), y)
//# define DGAP_LOCK(x,y)		spin_lock(&(x))
//# define DGAP_UNLOCK(x,y)	spin_unlock(&(x))
# define DGAP_TRYLOCK(x,y)	spin_trylock(&(x))

# define DGAP_PROCID()		smp_processor_id()

/*
 *  In some revisions within the 2.4 kernel series, there is a patch
 *  which changes the behavior of a failed open.  In 2.0 and 2.2 and
 *  most 2.4 revisions, a failed open will be followed _always_ by
 *  a call to the driver's close routine.  In some 2.4 revisions,
 *  the close will not be called (and therefore driver use counts
 *  will be too high, and ports may become unusable).
 *
 *  In the case that the ALT_FAIL_OPEN code is allowed to exist,
 *  some new parameters are required, like ALT_FAIL_OPEN_DEFAULT.
 */
# define ALLOW_ALT_FAIL_OPEN 0
# define ALT_FAIL_OPEN_DEFAULT 0

# define DGAP_GET_TTY_COUNT(x) (x->count)


#elif LINUX_VERSION_CODE >= KERNEL_VERSION(2,4,0)

# include <asm/uaccess.h>

# define DGAP_get_free_page(x)		(get_free_page(GFP_KERNEL))
# define DGAP_VMALLOC(x)		(vmalloc(x))
# define DGAP_VFREE(x)			(vfree(x))

# define DGAP_MAJOR(x)			(MAJOR((x)->i_rdev))
# define DGAP_MINOR(x)			(MINOR((x)->i_rdev))
# define DGAP_TTY_MAJOR(tty)		(MAJOR(tty->device))
# define DGAP_TTY_MINOR(tty)		(MINOR(tty->device))

# define DGAP_NUM_CPUS			(smp_num_cpus)

# define DGAP_MOD_INC_USE_COUNT(rtn) \
	{ \
		MOD_INC_USE_COUNT; \
		rtn = 1; \
	}

# define DGAP_MOD_DEC_USE_COUNT MOD_DEC_USE_COUNT

# define DGAP_IRQ_RETURN(x)	return;

# define DGAP_SPINLOCK_INIT(x)		spin_lock_init(&(x))

# define DGAP_IOREMAP(ADDR, LEN)	ioremap(ADDR, LEN)
# define DGAP_IOUNMAP(ADDR)		iounmap(ADDR)
# define GET_USER(A1,A2)		get_user(A1,(unsigned int *)A2)
# define PUT_USER(A1,A2)		put_user(A1,(unsigned long *)A2)

# define READ_PROTO	struct file *file, char *buf, size_t count,loff_t *ppos
# define READ_ARGS	file, buf, count, ppos
# define READ_RETURN_T	ssize_t

# define WRITE_PROTO	struct file *file, const char *buf, size_t count, \
			loff_t *ppos
# define WRITE_ARGS	file, buf, count, ppos
# define WRITE_RETURN_T	ssize_t

# define CLOSE_RETURN_T		int
# define CLOSE_RETURN(X)	return(X)

# define PARM_I_RDEV		file->f_dentry->d_inode->i_rdev
# define GET_POS()		*ppos
# define ADD_POS(AMT)		*ppos += (AMT)
# define SET_POS(AMT)		*ppos = (AMT)

# define DGAP_PCI_IRQ_TYPE	uint
# define PCI_PRESENT()		pci_present()

# define PARM_STR(VAR, INIT, DESC) \
		static char *VAR = INIT; \
		char *dgap_##VAR; \
		MODULE_PARM(VAR, "s"); \
		MODULE_PARM_DESC(VAR, DESC);

# define PARM_INT(VAR, INIT, DESC) \
		static int VAR = INIT; \
		int dgap_##VAR; \
		MODULE_PARM(VAR, "i"); \
		MODULE_PARM_DESC(VAR, DESC);

# define PARM_ULONG(VAR, INIT, DESC) \
		static ulong VAR = INIT; \
		ulong dgap_##VAR; \
		MODULE_PARM(VAR, "l"); \
		MODULE_PARM_DESC(VAR, DESC);

# define DGAP_LOCK(x,y)		spin_lock_irqsave(&(x), y);
# define DGAP_UNLOCK(x,y)	spin_unlock_irqrestore(&(x), y);

# define DGAP_PROCID()		smp_processor_id()

/*
 *  In some revisions within the 2.4 kernel series, there is a patch
 *  which changes the behavior of a failed open.  In 2.0 and 2.2 and
 *  most 2.4 revisions, a failed open will be followed _always_ by
 *  a call to the driver's close routine.  In some 2.4 revisions,
 *  the close will not be called (and therefore driver use counts
 *  will be too high, and ports may become unusable).
 *
 *  In the case that the ALT_FAIL_OPEN code is allowed to exist,
 *  some new parameters are required, like ALT_FAIL_OPEN_DEFAULT.
 */
# define ALLOW_ALT_FAIL_OPEN 1
# define ALT_FAIL_OPEN_DEFAULT 0

/*
 * Contrary to what I have been led to believe by some of the
 * kernel guru's, not all 2.4.x kernels have min/max macro's
 * in the kernel.h header.
 * I call your attention to the Red Hat AS 2.4.9-e.27 kernel.
 */
# ifndef min
#  define min(a,b) (((a)<(b))?(a):(b))
# endif
# ifndef max
#  define max(a,b) (((a)>(b))?(a):(b))
# endif

#if defined(REDHAT_AS_30)  || defined(REDHAT_ES_30) || defined(REDHAT_WS_30) || \
    defined(REDHAT_AS_3)   || defined(REDHAT_ES_3)  || defined(REDHAT_WS_3)  || \
    defined(REDHATAS30)    || defined(REDHATES30)   || defined(REDHATWS30)   || \
    defined(REDHATAS3)     || defined(REDHATES3)    || defined(REDHATWS3)    || \
    defined(REDHAT_FEDORA) || defined(FEDORA)       || \
    defined(REDHAT_10)     || defined(REDHAT10)

# define DGAP_GET_TTY_COUNT(x) (atomic_read(&(x->count)))

#else

# define DGAP_GET_TTY_COUNT(x) (x->count)

#endif

#else

# error "this driver does not support anything below the 2.4 kernel series."

#endif


/* 
 * Modem line constants are defined as macros because DSR and
 * DCD are swapable using the ditty altpin option.
 */
#define D_CD(ch)        ch->ch_cd       /* Carrier detect       */
#define D_DSR(ch)       ch->ch_dsr      /* Data set ready       */
#define D_RTS(ch)       DM_RTS          /* Request to send      */
#define D_CTS(ch)       DM_CTS          /* Clear to send        */
#define D_RI(ch)        DM_RI           /* Ring indicator       */
#define D_DTR(ch)       DM_DTR          /* Data terminal ready  */


/*************************************************************************
 *
 * Structures and closely related defines.
 *
 *************************************************************************/


/*
 * A structure to hold a statistics counter.  We also
 * compute moving averages for this counter.
 */
struct macounter
{
	u32		cnt;	/* Total count */
	ulong		accum;	/* Acuumulator per period */
	ulong		sma;	/* Simple moving average */
	ulong		ema;	/* Exponential moving average */
};


/*
 *	Per-board information
 */
struct board_t
{
	int		magic;		/* Board Magic number.  */
	int		boardnum;	/* Board number: 0-3 */
	int		firstminor;	/* First minor, e.g. 0, 30, 60 */

	int		type;		/* Type of board */
	char		*name;		/* Product Name */
	u16		vendor;		/* PCI vendor ID */
	u16		device;		/* PCI device ID */
	u16		subvendor;	/* PCI subsystem vendor ID */
	u16		subdevice;	/* PCI subsystem device ID */
	uchar		rev;		/* PCI revision ID */
	u16		maxports;	/* MAX ports this board can handle */

	spinlock_t	bd_lock;	/* Used to protect board */

	u32		state;		/* State of card. */
	wait_queue_head_t state_wait;	/* Place to sleep on for state change */

	struct		tasklet_struct helper_tasklet; /* Poll helper tasklet */

	u32		wait_for_bios;
	u32		wait_for_fep;

	struct cnode *  bd_config;	/* Config of board */

	u16		nasync;		/* Number of ports on card */

	u32		use_interrupts;	/* Should we be interrupt driven? */
	u32		irq;		/* Interrupt request number */
	u32		intr_count;	/* Count of interrupts */
	u32		intr_used;	/* Non-zero if using interrupts */
	u32		intr_running;	/* Non-zero if FEP knows its doing interrupts */

	ulong		port;		/* Start of base io port of the card */
	ulong		port_end;	/* End of base io port of the card */
	ulong		membase;	/* Start of base memory of the card */
	ulong		membase_end;	/* End of base memory of the card */

	uchar 		*re_map_port;	/* Remapped io port of the card */
	uchar		*re_map_membase;/* Remapped memory of the card */

	uchar		runwait;	/* # Processes waiting for FEP  */
	uchar		inhibit_poller; /* Tells  the poller to leave us alone */

	struct channel_t *channels[MAXPORTS]; /* array of pointers to our channels. */

	struct tty_driver	SerialDriver;
	char		SerialName[200];
	struct tty_driver	PrintDriver;
	char		PrintName[200];

	u32		dgap_Major_Serial_Registered;
	u32		dgap_Major_TransparentPrint_Registered;

	u32		dgap_Serial_Major;
	u32		dgap_TransparentPrint_Major;

	u32		TtyRefCnt;

	struct bs_t	*bd_bs;			/* Base structure pointer       */

	char	*flipbuf;		/* Our flip buffer, alloced if board is found */

	u16		dpatype;	/* The board "type", as defined by DPA */
	u16		dpastatus;	/* The board "status", as defined by DPA */
	wait_queue_head_t kme_wait;	/* Needed for DPA support */

	u32		conc_dl_status;	/* Status of any pending conc download */
	/*
	 *	Mgmt data.
	 */
        char		*msgbuf_head;
        char		*msgbuf;

};



/************************************************************************ 
 * Unit flag definitions for un_flags.
 ************************************************************************/
#define UN_ISOPEN	0x0001		/* Device is open		*/
#define UN_CLOSING	0x0002		/* Line is being closed		*/
#define UN_IMM		0x0004		/* Service immediately		*/
#define UN_BUSY		0x0008		/* Some work this channel	*/
#define UN_BREAKI	0x0010		/* Input break received		*/
#define UN_PWAIT	0x0020		/* Printer waiting for terminal	*/
#define UN_TIME		0x0040		/* Waiting on time		*/
#define UN_EMPTY	0x0080		/* Waiting output queue empty	*/
#define UN_LOW		0x0100		/* Waiting output low water mark*/
#define UN_EXCL_OPEN	0x0200		/* Open for exclusive use	*/
#define UN_WOPEN	0x0400		/* Device waiting for open	*/
#define UN_WIOCTL	0x0800		/* Device waiting for open	*/
#define UN_HANGUP	0x8000		/* Carrier lost			*/


/************************************************************************
 * Structure for terminal or printer unit. 
 ************************************************************************/
struct un_t {
	int	magic;		/* Unit Magic Number.			*/
	struct	channel_t *un_ch;
	u32	un_time;
	u32	un_type;
	u32	un_open_count;	/* Counter of opens to port		*/
	struct tty_struct *un_tty;/* Pointer to unit tty structure	*/
	u32	un_flags;	/* Unit flags				*/
	wait_queue_head_t un_flags_wait; /* Place to sleep to wait on unit */
	u32	un_dev;		/* Minor device number			*/
	tcflag_t un_oflag;	/* oflags being done on board		*/
	tcflag_t un_lflag;	/* lflags being done on board		*/
};


/************************************************************************ 
 * Device flag definitions for ch_flags.
 ************************************************************************/
#define CH_PRON         0x0001          /* Printer on string                */
#define CH_OUT          0x0002          /* Dial-out device open             */
#define CH_STOP         0x0004          /* Output is stopped                */
#define CH_STOPI        0x0008          /* Input is stopped                 */
#define CH_CD           0x0010          /* Carrier is present               */
#define CH_FCAR         0x0020          /* Carrier forced on                */

#define CH_RXBLOCK      0x0080          /* Enable rx blocked flag           */
#define CH_WLOW         0x0100          /* Term waiting low event           */
#define CH_WEMPTY       0x0200          /* Term waiting empty event         */
#define CH_RENABLE      0x0400          /* Buffer just emptied          */
#define CH_RACTIVE      0x0800          /* Process active in xxread()   */
#define CH_RWAIT        0x1000          /* Process waiting in xxread()  */
#define CH_HANGUP       0x8000		/* Hangup received                  */


/************************************************************************ 
 * Channel information structure.
 ************************************************************************/
struct channel_t {
	int magic;			/* Channel Magic Number		*/
	struct bs_t	*ch_bs;		/* Base structure pointer       */
	struct cm_t	*ch_cm;		/* Command queue pointer        */
	struct board_t *ch_bd;		/* Board structure pointer      */
	unsigned char *ch_vaddr;	/* FEP memory origin            */
	unsigned char *ch_taddr;	/* Write buffer origin          */
	unsigned char *ch_raddr;	/* Read buffer origin           */
	struct digi_t  ch_digi;		/* Transparent Print structure  */
	struct un_t ch_tun;		/* Terminal unit info           */
	struct un_t ch_pun;		/* Printer unit info            */

	spinlock_t	ch_lock;	/* provide for serialization */
	wait_queue_head_t ch_flags_wait;

	u32	pscan_state;
	uchar	pscan_savechar;

	u32 ch_portnum;			/* Port number, 0 offset.	*/
	u32 ch_open_count;		/* open count			*/
	u32	ch_flags;		/* Channel flags                */


	u32	ch_close_delay;		/* How long we should drop RTS/DTR for */

	u32	ch_cpstime;		/* Time for CPS calculations    */

	tcflag_t ch_c_iflag;		/* channel iflags               */
	tcflag_t ch_c_cflag;		/* channel cflags               */
	tcflag_t ch_c_oflag;		/* channel oflags               */
	tcflag_t ch_c_lflag;		/* channel lflags               */

	u16  ch_fepiflag;            /* FEP tty iflags               */
	u16  ch_fepcflag;		/* FEP tty cflags               */
	u16  ch_fepoflag;		/* FEP tty oflags               */
	u16  ch_wopen;			/* Waiting for open process cnt */
	u16  ch_tstart;			/* Transmit buffer start        */
	u16  ch_tsize;			/* Transmit buffer size         */
	u16  ch_rstart;			/* Receive buffer start         */
	u16  ch_rsize;			/* Receive buffer size          */
	u16  ch_rdelay;			/* Receive delay time           */

	u16	ch_tlw;			/* Our currently set low water mark */

	u16  ch_cook;			/* Output character mask        */

	uchar   ch_card;		/* Card channel is on           */
	uchar   ch_stopc;		/* Stop character               */
	uchar   ch_startc;		/* Start character              */

	uchar   ch_mostat;		/* FEP output modem status      */
	uchar   ch_mistat;		/* FEP input modem status       */
	uchar   ch_mforce;		/* Modem values to be forced    */
	uchar   ch_mval;		/* Force values                 */
	uchar   ch_fepstopc;		/* FEP stop character           */
	uchar   ch_fepstartc;		/* FEP start character          */

	uchar   ch_astopc;		/* Auxiliary Stop character     */
	uchar   ch_astartc;		/* Auxiliary Start character    */
	uchar   ch_fepastopc;		/* Auxiliary FEP stop char      */
	uchar   ch_fepastartc;		/* Auxiliary FEP start char     */

	uchar   ch_hflow;		/* FEP hardware handshake       */
	uchar   ch_dsr;			/* stores real dsr value        */
	uchar   ch_cd;			/* stores real cd value         */
	uchar   ch_tx_win;		/* channel tx buffer window     */
	uchar   ch_rx_win;		/* channel rx buffer window     */
	u32  ch_info;			/* Additional channel specific information */
};


/*************************************************************************
 *
 * Prototypes for non-static functions used in more than one module
 *
 *************************************************************************/

int			dgap_lock_poller(struct board_t *brd);
void			dgap_unlock_poller(struct board_t *brd);
void			dgap_mbuf(struct board_t *brd, const char *fmt, ...);
int			dgap_sleep(int ticks);
int			dgap_ms_sleep(u32 ms);

int			dgap_tty_brdinit(struct board_t *brd);
void			dgap_tty_brduninit(struct board_t *brd);

void			*dgap_driver_kzmalloc(size_t size, int priority);

char			*get_event_text(u32);
char			*get_msignal_text(u32);

char			*dgap_ioctl_name(int cmd);

#if defined(DGAP_TRACER)
void			dgap_tracef(const char *fmt, ...);
void			dgap_tracer_free(void);
#endif

void			dgap_proc_unregister_all(void);
void			dgap_proc_register_basic(void);
int                     dgap_finalize_board_init(struct board_t *brd);

void			dgap_do_bios_load(struct board_t *brd, uchar *ubios, int len);
void			dgap_do_fep_load(struct board_t *brd, uchar *ufep, int len);
void			dgap_do_conc_load(struct board_t *brd, uchar *uaddr, int len);
void			dgap_do_config_load(uchar *uaddr, int len);

int			dgap_after_config_loaded(void);

extern spinlock_t		dgap_dl_lock;
extern wait_queue_head_t	dgap_dl_wait;
extern int			dgap_dl_action;

extern int			dgap_driver_state;

extern int			dgap_debug;
extern int			dgap_rawreadok;
extern spinlock_t		dgap_global_lock;

extern uchar			dgap_NumBoards;


#endif
