/*
 * Copyright 2003 Digi International (www.digi.com)
 *	Scott H Kilau <Scott_Kilau at digi dot com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *	NOTE TO LINUX KERNEL HACKERS:  DO NOT REFORMAT THIS CODE! 
 *
 *	This is shared code between Digi's CVS archive and the
 *	Linux Kernel sources.
 *	Changing the source just for reformatting needlessly breaks
 *	our CVS diff history.
 *
 *	Send any bug fixes/changes to:  Eng.Linux at digi dot com. 
 *	Thank you. 
 *
 */

/************************************************************************
 *
 * /proc/dgap/{*} functions
 *
 ************************************************************************/
#define __NO_VERSION__
#include "dgap_driver.h"
#include "dgap_parse.h"
#include "dgap_mgmt.h"
#include <linux/ctype.h>
#include <linux/proc_fs.h>

/* external variables */
extern int              dgap_debug;
extern struct board_t	*dgap_Board[MAXBOARDS];
extern uchar		dgap_NumBoards;
extern u32		dgap_poll_counter;


/*
 * OK, here's the deal with /proc files.  We want to use the
 * simple (yet inefficient) interface to /proc where we just get
 * a buffer and fill it in.  Otherwise, we'd have to provide the
 * full set of functions for a /proc driver.
 *
 * When we get called, "buf" points to a single free page which
 * we can write into.  If you are careful, you can write up
 * to PAGE_SIZE (4096) bytes into it.  But to be paranoid,
 * Linux lies and says the "count" is only PROC_BLOK_SIZE (3072).
 *
 * In any case, its easiest to make several /proc/dgap/ entries
 * if you want to print out more info than can fit in a page.
 *
 * Here's some further information.  We are trying to be compatible
 * with 2.0.xx and 2.1.xx.  There have been improvements in 2.1.x
 * in this area, but it still isn't perfect.  Anyway, our hands
 * are tied by the 2.0.x limitations.  For example, you cannot
 * pass private data to the get_info (2.0.x) routines.  Therefore,
 * you cannot make a generic function to print the data for one
 * board.  If you want to do this, you have to make a wrapper
 * function, one function name per board number.  Yech!
 */

	#define	PROC_READ_PROTO \
		char *buf, char **start, off_t fpos, int count, \
		int *eof, void *private

	#define	PROC_READ_ARGS \
		buf, start, fpos, count, eof, private


/*
 *	Make wrappers for per-board entries. Gag me.  Sorry,
 *	2.0.x forced us into this since it has no way to pass
 *	private data to the read_proc (get_info, actually).
 */
/* FIXME - Fix this, we can convert away from this now */
#define	PROCWRAP_FUNCTION(FUNC, NUM) \
			static int \
			FUNC ## NUM (PROC_READ_PROTO) \
			{ \
				return FUNC(NUM, PROC_READ_ARGS); \
			}

#define	PROCWRAP_FUNC(FUNC,NUM) \
		FUNC ## tbl[NUM]

#define	PROCWRAP_FUNCTIONS(FUNC) \
		PROCWRAP_FUNCTION(FUNC, 0) \
		PROCWRAP_FUNCTION(FUNC, 1) \
		PROCWRAP_FUNCTION(FUNC, 2) \
		PROCWRAP_FUNCTION(FUNC, 3) \
		PROCWRAP_FUNCTION(FUNC, 4) \
		PROCWRAP_FUNCTION(FUNC, 5) \
		PROCWRAP_FUNCTION(FUNC, 6) \
		PROCWRAP_FUNCTION(FUNC, 7) \
		PROCWRAP_FUNCTION(FUNC, 8) \
		PROCWRAP_FUNCTION(FUNC, 9) \
		PROCWRAP_FUNCTION(FUNC, 10) \
		PROCWRAP_FUNCTION(FUNC, 11) \
		PROCWRAP_FUNCTION(FUNC, 12) \
		PROCWRAP_FUNCTION(FUNC, 13) \
		PROCWRAP_FUNCTION(FUNC, 14) \
		PROCWRAP_FUNCTION(FUNC, 15) \
		PROCWRAP_FUNCTION(FUNC, 16) \
		PROCWRAP_FUNCTION(FUNC, 17) \
		PROCWRAP_FUNCTION(FUNC, 18) \
		PROCWRAP_FUNCTION(FUNC, 19) \
		PROCWRAP_FUNCTION(FUNC, 20) \
		PROCWRAP_FUNCTION(FUNC, 21) \
		PROCWRAP_FUNCTION(FUNC, 22) \
		PROCWRAP_FUNCTION(FUNC, 23) \
		PROCWRAP_FUNCTION(FUNC, 24) \
		PROCWRAP_FUNCTION(FUNC, 25) \
		PROCWRAP_FUNCTION(FUNC, 26) \
		PROCWRAP_FUNCTION(FUNC, 27) \
		PROCWRAP_FUNCTION(FUNC, 28) \
		PROCWRAP_FUNCTION(FUNC, 29) \
		PROCWRAP_FUNCTION(FUNC, 30) \
		PROCWRAP_FUNCTION(FUNC, 31) \

#define	PROCWRAP_TBL(FUNC) \
			static read_proc_t * FUNC ## tbl[MAXBOARDS] = \
		{ \
			FUNC ## 0, \
			FUNC ## 1, \
			FUNC ## 2, \
			FUNC ## 3, \
			FUNC ## 4, \
			FUNC ## 5, \
			FUNC ## 6, \
			FUNC ## 7, \
			FUNC ## 8, \
			FUNC ## 9, \
			FUNC ## 10, \
			FUNC ## 11, \
			FUNC ## 12, \
			FUNC ## 13, \
			FUNC ## 14, \
			FUNC ## 15, \
			FUNC ## 16, \
			FUNC ## 17, \
			FUNC ## 18, \
			FUNC ## 19, \
			FUNC ## 20, \
			FUNC ## 21, \
			FUNC ## 22, \
			FUNC ## 23, \
			FUNC ## 24, \
			FUNC ## 25, \
			FUNC ## 26, \
			FUNC ## 27, \
			FUNC ## 28, \
			FUNC ## 29, \
			FUNC ## 30, \
			FUNC ## 31, \
		};

#define	PROCWRAPPERS(FUNC) \
		PROCWRAP_FUNCTIONS(FUNC) \
		PROCWRAP_TBL(FUNC)


/*
 * /proc/dgap/<boardno>/tty
 */
static struct proc_dir_entry *ProcDgAPTTY[MAXBOARDS];

static int dgap_proc_tty(int brdno, PROC_READ_PROTO)
{
	struct board_t *brd;
	char	*p = buf;

	DPR_PROC(("dgap_proc_tty count=%d fpos=%ld\n", count, fpos));

        brd = dgap_Board[brdno];

        if(!brd)
                return(-EINVAL);

	/* Prepare the Header Labels */
	p += sprintf(p, "         Receive Statistics        "
				 "Transmit Statistics\n");
	p += sprintf(p, "%2s %19s %7s %19s %7s %s\n",
                     "Ch", "Chars Rx", "RxCPS",
                     "Chars Tx", "TxCPS", " Line Status Flags");

	DPR_PROC(("dgap_proc_tty returns %lx\n", (long int) (p-buf)));

	return(p-buf);
}

PROCWRAPPERS(dgap_proc_tty)


/*
 * /proc/dgap/<boardno>/ttys
 */
static struct proc_dir_entry *ProcDgAPTTYs[MAXBOARDS];

static int dgap_proc_ttys(int brdno, PROC_READ_PROTO)
{
	struct board_t *brd;
	char	*p = buf;

	DPR_PROC(("dgap_proc_ttys count=%d fpos=%ld\n", count, fpos));

        brd = dgap_Board[brdno];

        if(!brd)
                return(-EINVAL);

	p += sprintf(p, "Chan\t&tty\tflags\n");

	DPR_PROC(("dgap_proc_ttys returns %lx\n", (long int) (p-buf)));

	return(p-buf);
}

PROCWRAPPERS(dgap_proc_ttys)


/*
 * /proc/dgap/<boardno>/info
 *
 * Variables compatible with the shell
 */
static struct proc_dir_entry *ProcDgAPBrdInfo[MAXBOARDS];

static int dgap_proc_brd_info(int brdno, PROC_READ_PROTO)
{
	struct board_t	*brd;
	char		*p = buf;
	char		*name;

	DPR_PROC(("dgap_proc_brd_info\n"));

        brd = dgap_Board[brdno];

        if(!brd)
                return(-EINVAL);

	name = brd->name;

	p += sprintf(p, "Board Name = %s\n", name);
	p += sprintf(p, "Board Type = %d\n", brd->type);

	/*
	 * report some things about the PCI bus that are important
	 * to some applications
	 */
        p += sprintf(p, "Vendor ID = 0x%x\n", brd->vendor);
        p += sprintf(p, "Device ID = 0x%x\n", brd->device);
        p += sprintf(p, "Subvendor ID = 0x%x\n", brd->subvendor);
        p += sprintf(p, "Subdevice ID = 0x%x\n", brd->subdevice);

	/*
	 * report the physical addresses assigned to us when we got
	 * registered
	 */	
        p += sprintf(p, "IO Port = 0x%lx\n", brd->port);
        p += sprintf(p, "Memory Base Address = 0x%lx\n", brd->membase);
        p += sprintf(p, "Remapped IO Port Address = 0x%p\n", brd->re_map_port);
        p += sprintf(p, "Remapped Memory Base Address = 0x%p\n", brd->re_map_membase);

        p += sprintf(p, "Current state of board = %s\n", dgap_state_text[brd->state]);
        p += sprintf(p, "Interrupt #: %d. Times interrupted: %d\n",
		brd->irq, brd->intr_count);
        p += sprintf(p, "Majors allocated to board = TTY: %d PR: %d\n",
		brd->SerialDriver.major, brd->PrintDriver.major);

	/*
	 * report available resources on the card
	 */

	return(p-buf);
}

PROCWRAPPERS(dgap_proc_brd_info)


/*
 * /proc/dgap/info
 *
 *	Variables compatible with the shell
 */
static struct proc_dir_entry *ProcDgAPInfo;

static int dgap_proc_info(PROC_READ_PROTO)
{
	char	*p = buf;

	DPR_PROC(("dgap_proc_info\n"));
	p += sprintf(p, "Max Boards = %d\n", MAXBOARDS);
	p += sprintf(p, "Current number of boards = %d\n", dgap_NumBoards);
	p += sprintf(p, "Poll counter = %d\n", dgap_poll_counter);
	p += sprintf(p, "State of driver: %s\n", dgap_driver_state_text[dgap_driver_state]);
	return(p-buf);
}

/*
 * /proc/dgap/<boardno>/mknod
 */
static struct proc_dir_entry *ProcDgAPBrdMknod[MAXBOARDS];

static int mknod_tty_cmd(char *buf, char *nodename,
                     int major, int firstminor, int count, int start)
{
	return sprintf(buf, "/usr/sbin/dgap_mknod\t%s\t\t%d\t%d\t%d\t%d\n",
		       nodename, major, firstminor, count, start);
}


static int dgap_proc_brd_mknod(int brdno, PROC_READ_PROTO)
{
	struct board_t *brd = dgap_Board[brdno];
	int	bn;
	char	*p = buf;
	struct cnode *cptr = NULL;
	char str[100];
	int found = FALSE;
	int ncount = 0;
	int starto = 0;

	DPR_PROC(("dgap_proc_brd_mknod(%d) %d\n", brdno, count));

        if(!brd)
                return(-ENODEV);

        bn = brd->boardnum;


	/*
	 * Output the boilerplate at the top of the shell script
	 */
	p += sprintf(p, "#!/bin/sh\n\n");

	p += sprintf(p, "# /proc/dgap/mknod [-d]\n");
	p += sprintf(p, "#\tMake DGAP device nodes.\n");
	p += sprintf(p, "#\t-d deletes all existing nodes first\n\n");

	p += sprintf(p, "PATH=" SBINDIR ":$PATH\n");

	p += sprintf(p, "[ -d %s ] || mkdir -p -m 755 %s\n",
                     DEVSTR, DEVSTR);

	p += sprintf(p, "[ \"$1\" != -d ] || rm -rf %s/* || exit 1\n",
                     DEVSTR);

	p += sprintf(p, "cd %s || exit 2\n\n", DEVSTR);

	/*
	 * For each board, output the device information in
	 * a handy table format...
	 */
	p += sprintf(p,	"#\t\tNAME\t\tMAJOR\tMINOR\tCOUNT\tSTART\n");


	for (cptr = brd->bd_config; cptr; cptr = cptr->next) {

		if ((cptr->type == BNODE) &&
		    ((cptr->u.board.type == APORT2_920P) || (cptr->u.board.type == APORT4_920P) ||
		     (cptr->u.board.type == APORT8_920P) || (cptr->u.board.type == PAPORT4) ||
		     (cptr->u.board.type == PAPORT8))) {

				found = TRUE;
				if (cptr->u.board.start)
					starto = cptr->u.board.start;
				else
					starto = 1;
		}

		if (cptr->type == TNODE && found == TRUE) {
			char *ptr1;
			if (strstr(cptr->u.ttyname, "tty")) {
				ptr1 = cptr->u.ttyname;
				ptr1 += 3;
			}
			else {
				ptr1 = cptr->u.ttyname;
			}

			/* TTY devices */
			sprintf(str, "tty%s%%p", ptr1);

			p += mknod_tty_cmd(p, str,
				brd->SerialDriver.major,
				0, dgap_config_get_number_of_ports(brd), starto);

			/* PR devices */
			sprintf(str, "pr%s%%p", ptr1);

			p += mknod_tty_cmd(p, str,
				brd->PrintDriver.major,
				0, dgap_config_get_number_of_ports(brd), starto);
		}

		if (cptr->type == CNODE) {

			sprintf(str, "tty%s%%p", cptr->u.conc.id);

			/* TTY devices */
			p += mknod_tty_cmd(p, str,
				brd->SerialDriver.major,
				ncount, cptr->u.conc.nport,
				cptr->u.conc.start ? cptr->u.conc.start : 1);

			sprintf(str, "pr%s%%p", cptr->u.conc.id);

			p += mknod_tty_cmd(p, str,
				brd->PrintDriver.major,
				ncount, cptr->u.conc.nport,
				cptr->u.conc.start ? cptr->u.conc.start : 1);

			ncount += cptr->u.conc.nport;
		}

		if (cptr->type == MNODE) {

			sprintf(str, "tty%s%%p", cptr->u.module.id);

			/* TTY devices */
			p += mknod_tty_cmd(p, str,
				brd->SerialDriver.major,
				ncount, cptr->u.module.nport,
				cptr->u.module.start ? cptr->u.module.start : 1);

			sprintf(str, "pr%s%%p", cptr->u.module.id);

			p += mknod_tty_cmd(p, str,
				brd->PrintDriver.major,
				ncount, cptr->u.module.nport,
				cptr->u.module.start ? cptr->u.module.start : 1);

			ncount += cptr->u.module.nport;

		}
	}

	return(p-buf);
}

PROCWRAPPERS(dgap_proc_brd_mknod)


/*
 *      /proc/dgap/mknod
 */
static struct proc_dir_entry *ProcDgAPMknod;

static int dgap_proc_mknod(PROC_READ_PROTO)
{
        char    *p = buf;

        DPR_PROC(("dgap_proc_mknod\n"));

        p += sprintf(p, "#!/bin/sh\n\n");

        p += sprintf(p, "#\t/proc/dgap/mknod [-d]\n");
        p += sprintf(p, "#\t\tMake DGAP device nodes.\n");
        p += sprintf(p, "#\t\t-d deletes all existing nodes first\n\n");

	p += sprintf(p, "if [ \"$1\" != -d ]\nthen\n");
	p += sprintf(p, "# Create the management device.\n");
	p += mknod_tty_cmd(p, "/dev/dg/dgap/mgmt", DIGI_DGAP_MAJOR, MGMT_MGMT, 1, 0);
	p += mknod_tty_cmd(p, "/dev/dg/dgap/downld", DIGI_DGAP_MAJOR, MGMT_DOWNLD, 1, 0);
	p += sprintf(p, "exit 1\nfi\n\n");

	p += sprintf(p, "if [ -d %s ]\nthen\n", DEVSTR);

        p += sprintf(p, "\tcd %s\n", DEVSTR);

	/* Delete all our ttys */
        p += sprintf(p, "\tfor i in tty*\n");
        p += sprintf(p, "\tdo\n");

        p += sprintf(p, "\t\tif [ \"$i\" != \"tty*\" ]\n");
        p += sprintf(p, "\t\tthen\n");

        p += sprintf(p, "\t\t\trm -rf $i\n");
        p += sprintf(p, "\t\t\trm -rf /dev/$i\n");

        p += sprintf(p, "\t\tfi\n");

        p += sprintf(p, "\tdone\n");

	/* Delete all our prs */
        p += sprintf(p, "\tfor i in pr*\n");
        p += sprintf(p, "\tdo\n");

        p += sprintf(p, "\t\tif [ \"$i\" != \"pr*\" ]\n");
        p += sprintf(p, "\t\tthen\n");

        p += sprintf(p, "\t\t\trm -rf $i\n");
        p += sprintf(p, "\t\t\trm -rf /dev/$i\n");

        p += sprintf(p, "\t\tfi\n");

        p += sprintf(p, "\tdone\n");

        p += sprintf(p, "fi\n");

	/* Delete anything that was left. */
	p += sprintf(p, "rm -rf %s/*\n", DEVSTR);

	/* Now go built the directory again */
	p += sprintf(p, "mkdir -p -m 755 %s\n", DEVSTR);

        p += sprintf(p, "for i in /proc/dgap/*/mknod\n");
        p += sprintf(p, "do\n");

        p += sprintf(p, "\tif [ \"$i\" != \"/proc/dgap/*/mknod\" ]\n");
        p += sprintf(p, "\tthen\n");

        p += sprintf(p, "\t\t$i\n");

        p += sprintf(p, "\tfi\n");

        p += sprintf(p, "done\n");

	p += sprintf(p, "#\tCreate the management device.\n\n");
	p += mknod_tty_cmd(p, "/dev/dg/dgap/mgmt", DIGI_DGAP_MAJOR, MGMT_MGMT, 1, 0);
	p += mknod_tty_cmd(p, "/dev/dg/dgap/downld", DIGI_DGAP_MAJOR, MGMT_DOWNLD, 1, 0);

        return (p-buf);
}


/*
 * The directory entries /proc/dgap/<boardnum>
 */
static struct proc_dir_entry *ProcDgAPBrd[MAXBOARDS];

/*
 * The directory entry /proc/dgap
 */
static struct proc_dir_entry *ProcDgAP;

/*
 * Gee, the proc I/F for drivers is *still* not well done in 2.1.x.
 *
 * In any event, we'll *make* a better interface here and then
 * use it to abstract the differences between 2.0.x and 2.1.x
 *
 * Unfortunately, the *best* interface would also allow you
 * to pass in "private" data.  But that can't be supported
 * on 2.0.x, so that is why this interface is just "better".
 */

static struct proc_dir_entry *better_create_proc_entry(
        const char *name, mode_t mode, unsigned long size,
        read_proc_t *read_proc,	struct proc_dir_entry *parent)
{
        struct proc_dir_entry	*pde;

        pde = create_proc_entry(name, mode, parent);
        if (!pde) return NULL;

        if (size)
                pde->size = size;
        if (read_proc)
                pde->read_proc = read_proc;

        return pde;
}

static void better_remove_proc_entry(struct proc_dir_entry *pde)
{
        if (!pde) return;

        remove_proc_entry(pde->name, pde->parent);
}


/*
 * Register the basic /proc/dgap files that appear whenever
 * the driver is loaded.
 */
void dgap_proc_register_basic_prescan(void)
{
	char	*buf;
	int	size;
	
	/* Register /proc/dgap */
	ProcDgAP = better_create_proc_entry(PROCSTR, S_IFDIR, 0, NULL, 0);

	/* Register /proc/dgap/info */
	ProcDgAPInfo = better_create_proc_entry( "info", 0, 0,
			dgap_proc_info, ProcDgAP);

	/*
	 * Compute size of /proc/dgap/mknod output, fixup
	 * proc_dir_entry for it, then register it.
	 */
	buf = kmalloc(4096, GFP_KERNEL);
	if (buf) {
                size = dgap_proc_mknod(buf, NULL, 0, 4096, 0, 0);
	} else
		size = 0;
	kfree(buf);

	ProcDgAPMknod =
                better_create_proc_entry("mknod", S_IFREG | S_IRUGO | S_IXUGO,
                                         size, dgap_proc_mknod, ProcDgAP);
}



/*
 * Register the basic /proc/dgap files that appear whenever
 * the driver is loaded.
 */
void dgap_proc_register_basic_postscan(void)
{
	char	*buf;
	int	size;
	int	i;
	
	/*
	 * Register /proc/dgap/<boardnum> directories
	 * and the /proc/dgap/<boardnum>/info files
	 */
	for (i = 0; i < dgap_NumBoards; ++i) {
		char	name[2];

		name[0] = i + '0';
		name[1] = 0;
		ProcDgAPBrd[i] = better_create_proc_entry(name, S_IFDIR, 0,
                                                          NULL, ProcDgAP);

		ProcDgAPBrdInfo[i] = better_create_proc_entry("info", 0, 0,
                        PROCWRAP_FUNC(dgap_proc_brd_info, i),
                        ProcDgAPBrd[i]);

		/*
                 * Compute size of /proc/dgap/<boardno>/mknod output, fixup
                 * proc_dir_entry for it, then register it.
                 */
                buf = kmalloc(4096, GFP_ATOMIC);
                if (buf) {
                        size = dgap_proc_brd_mknod(i, buf, NULL, 0, 4096, 0, 0);
                } else
                        size = 0;
                kfree(buf);
 
                ProcDgAPBrdMknod[i] = better_create_proc_entry("mknod",
                                        S_IFREG | S_IRUGO | S_IXUGO, size,
                                        PROCWRAP_FUNC(dgap_proc_brd_mknod, i),
                                        ProcDgAPBrd[i]);
	}
}


/*
 * Register the proc devices that appear only if we are
 * loading up a fep.
 */
void dgap_proc_register_fep(void)
{
	int	i;

	for (i = 0; i < dgap_NumBoards; ++i) {

		ProcDgAPTTYs[i] = better_create_proc_entry( "ttys", 0, 0,
			PROCWRAP_FUNC(dgap_proc_ttys, i),
			ProcDgAPBrd[i]);

		ProcDgAPTTY[i] = better_create_proc_entry( "tty", 0, 0,
			PROCWRAP_FUNC(dgap_proc_tty, i),
			ProcDgAPBrd[i]);
	}
}

/*
 * Unregister all of our /proc/dgap/{*} devices
 */
void dgap_proc_unregister_all(void)
{
	int	i;

#define NUKE_ONE(PDE) \
	if (PDE) { better_remove_proc_entry(PDE); PDE = NULL; } else

	for (i = 0; i < dgap_NumBoards; ++i) {
		NUKE_ONE(ProcDgAPTTY[i]);
		NUKE_ONE(ProcDgAPTTYs[i]);
		NUKE_ONE(ProcDgAPBrdMknod[i]);
		NUKE_ONE(ProcDgAPBrdInfo[i]);
		NUKE_ONE(ProcDgAPBrd[i]);
	}

	NUKE_ONE(ProcDgAPMknod);
	NUKE_ONE(ProcDgAPInfo);

	NUKE_ONE(ProcDgAP);
}
