/*
 * Copyright 2003 Digi International (www.digi.com)
 *	Scott H Kilau <Scott_Kilau at digi dot com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY, EXPRESS OR IMPLIED; without even the 
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 * PURPOSE.  See the GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *
 *	NOTE TO LINUX KERNEL HACKERS:  DO NOT REFORMAT THIS CODE!
 *
 *	This is shared code between Digi's CVS archive and the
 *	Linux Kernel sources.
 *	Changing the source just for reformatting needlessly breaks
 *	our CVS diff history.
 *
 *	Send any bug fixes/changes to:  Eng.Linux at digi dot com.
 *	Thank you.
 *
 */

char *dgap_version = "$Id: dgap_driver.c,v 1.63 2003/12/09 16:52:51 scottk Exp $";

/*
 * Our driver specific include files.
 */
#include "dgap_driver.h"
#include "dgap_proc.h"
#include "dgap_pci.h"
#include "dgap_fep5.h"
#include "dgap_tty.h"
#include "dgap_conf.h"
#include "dgap_parse.h"
#include "dgap_mgmt.h"


/*
 * Because this driver is supported on older versions of Linux
 * as well, lets be safe, and just make sure on this one.
 */
#if defined(MODULE_LICENSE)
	MODULE_LICENSE("GPL");
#endif

MODULE_AUTHOR("Digi International, http://www.digi.com");
MODULE_DESCRIPTION("Driver for the Digi International EPCA PCI based product line");
MODULE_SUPPORTED_DEVICE("dgap");

/*
 * insmod command line overrideable parameters
 *
 * NOTE: we use a set of macros to create the variables, which allows
 * us to specify the variable type, name, initial value, and description.
 */
PARM_INT(debug,		0x00,		"Driver debugging level");
PARM_INT(rawreadok,	1,		"Bypass flip buffers on input");
PARM_INT(trcbuf_size,	0x100000,	"Debugging trace buffer size. 1M default.");

/*
 * A generic list of Product names, PCI Vendor ID, and PCI Device ID.
 */
struct board_id {
	uint config_type;
	uchar *name;
	u16 vendor;
	u16 device;
	uint maxports;
	u16 dpatype;
};

static struct board_id Ids[] =
{
	{	PPCM,			PCI_DEVICE_XEM_NAME,	DIGI_VID,
		PCI_DEVICE_XEM_DID,	64,			(T_PCXM | T_PCLITE | T_PCIBUS)	},

	{	PCX,			PCI_DEVICE_CX_NAME,	DIGI_VID,
		PCI_DEVICE_CX_DID,	128,			(T_CX | T_PCIBUS)		},

	{	PCX,			PCI_DEVICE_CX_IBM_NAME,	DIGI_VID,
		PCI_DEVICE_CX_IBM_DID,128,			(T_CX | T_PCIBUS)		},

	{	PEPC,			PCI_DEVICE_EPCJ_NAME,	DIGI_VID,
		PCI_DEVICE_EPCJ_DID,	224,			(T_EPC  | T_PCIBUS)		},

	{	APORT2_920P,		PCI_DEVICE_920_2_NAME,	DIGI_VID,
		PCI_DEVICE_920_2_DID,	2,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	APORT4_920P,		PCI_DEVICE_920_4_NAME,	DIGI_VID,
		PCI_DEVICE_920_4_DID,	4,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	APORT8_920P,		PCI_DEVICE_920_8_NAME,	DIGI_VID,
		PCI_DEVICE_920_8_DID,	8,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	PAPORT8,		PCI_DEVICE_XR_NAME,	DIGI_VID,
		PCI_DEVICE_XR_DID,	8,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	PAPORT8,		PCI_DEVICE_XRJ_NAME,	DIGI_VID,
		PCI_DEVICE_XRJ_DID,	8,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	PAPORT8,		PCI_DEVICE_XR_422_NAME,	DIGI_VID,
		PCI_DEVICE_XR_422_DID,8,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

	{	PAPORT8,		PCI_DEVICE_XR_IBM_NAME,	DIGI_VID,
		PCI_DEVICE_XR_IBM_DID, 8,			(T_PCXR | T_PCLITE | T_PCIBUS)	},

};
#define	NIDS	(sizeof(Ids)/sizeof(struct board_id))


char *dgap_state_text[] = {
	"Board Failed",
	"Configuration for board not found.\n\t\t\t Use /usr/bin/dgap_config to configure board.",
	"Board Found",
	"Need Reset",
	"Finished Reset",
	"Need Config",
	"Finished Config",
	"Need Device Creation",
	"Requested Device Creation",
	"Finished Device Creation",
	"Need BIOS Load", 
	"Requested BIOS", 
	"Doing BIOS Load",
	"Finished BIOS Load",
	"Need FEP Load", 
	"Requested FEP",
	"Doing FEP Load",
	"Finished FEP Load",
	"Board READY",
};

char *dgap_driver_state_text[] = {
	"Driver Initialized",
	"Driver needs configuration load.",
	"Driver requested configuration from download daemon.",
	"Driver Ready."
};


/**************************************************************************
 *
 * protos for this file
 *
 */

/* Driver load/unload functions */
int			dgap_init_module(void);
void			dgap_cleanup_module(void);
int			dgap_start(void);
int			dgap_finalize_board_init(struct board_t *brd);

static void		dgap_init_globals(void);
static int		dgap_scan(void);
static int		dgap_found_board(struct pci_dev *pdev, int id);
static void		dgap_cleanup_board(struct board_t *brd);
static void		dgap_poll_handler(ulong dummy);


#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)

static irqreturn_t	dgap_intr(int irq, void *voidbrd, struct pt_regs *regs);
module_init(dgap_init_module);
module_exit(dgap_cleanup_module);

#else

static void		dgap_intr(int irq, void *voidbrd, struct pt_regs *regs);
/*
 * Older 2.4 kernels do not support module_init/exit calls.
 * So we do it this way for now.
 */
int			init_module(void);
void			cleanup_module(void);
#define dgap_init_module init_module
#define dgap_cleanup_module cleanup_module

#endif


/*
 * File operations permitted on Control/Management major.
 */
static struct file_operations BoardFops =
{
	owner:		THIS_MODULE,
	read:		NULL,
	write:		NULL,
	ioctl:		dgap_mgmt_ioctl,
	mmap:		NULL,
	open:		dgap_mgmt_open,
	release:	dgap_mgmt_close
};


/*
 * Statics/Globals
 */
struct board_t		*dgap_Board[MAXBOARDS];
uchar			dgap_NumBoards;
spinlock_t		dgap_global_lock = SPIN_LOCK_UNLOCKED;
ulong			dgap_poll_counter = 0;
static int		dgap_Major_Control_Registered = FALSE;
int			dgap_driver_state = DRIVER_INITIALIZED;

spinlock_t		dgap_dl_lock = SPIN_LOCK_UNLOCKED;
wait_queue_head_t	dgap_dl_wait;
int			dgap_dl_action;

/* Poller stuff */
static spinlock_t	dgap_poll_lock = SPIN_LOCK_UNLOCKED;	/* Poll scheduling lock */
static ulong		dgap_poll_time;				/* Time of next poll */
static ulong		dgap_poll_tick = 20;			/* Poll interval - 20 ms */
static struct timer_list dgap_poll_timer = { function: dgap_poll_handler };

static char		*dgap_config_buf;			/* The config file buffer */


/************************************************************************
 *
 * Driver load/unload functions
 *
 ************************************************************************/

/*
 * init_module()
 *
 * Module load.  This is where it all starts.
 */
int dgap_init_module(void)
{
	APR(("%s, Digi International Part Number %s\n", DG_NAME, DG_PART));
	return dgap_start();
}


/*
 * Start of driver.
 */
int dgap_start(void)
{
	int rc = 0;
	unsigned long flags;

        /* make sure that the globals are init'd before we do anything else */
        dgap_init_globals();

	dgap_NumBoards = 0;

	APR(("For the tools package or updated drivers please visit http://www.digi.com\n"));

	/*
	 * Register our base character device into the kernel.
	 * This allows the download daemon to connect to the downld device
	 * before any of the boards are init'ed.
	 */
	if (!dgap_Major_Control_Registered) {
		/*
		 * Register management/dpa devices
		 */
		rc = register_chrdev(DIGI_DGAP_MAJOR, "dgap", &BoardFops);
		if (rc < 0) {
			APR(("Can't register dgap driver device (%d)\n", rc));
			return (rc);
		}
		dgap_Major_Control_Registered = TRUE;
	}

	/*
	 * Register our basic stuff in /proc/dgap
	 */
	dgap_proc_register_basic_prescan();

	/*
	 * Init any global tty stuff.
	 */
	dgap_tty_preinit();

	/*
	 * Find and configure all the cards
	 */
	rc = dgap_scan();

	/*
	 * If something went wrong in the scan, bail out of driver.
	 */
	if (rc) {
		dgap_cleanup_module();
	}
	else {
		/* Start the poller */
		if (dgap_NumBoards > 0) {
			DGAP_LOCK(dgap_poll_lock, flags);
			dgap_poll_time = jiffies + dgap_jiffies_from_ms(dgap_poll_tick);
			dgap_poll_timer.expires = dgap_poll_time;
			DGAP_UNLOCK(dgap_poll_lock, flags);

			add_timer(&dgap_poll_timer);

		}

		dgap_driver_state = DRIVER_NEED_CONFIG_LOAD;
	}

	return (rc);
}



int dgap_after_config_loaded(void)
{
	int i = 0;
	int rc = 0;

	/*
	 * Register our ttys, now that we have the config loaded.
	 */
	for (i = 0; i < dgap_NumBoards; ++i) {

		/*
		 * Initialize KME waitqueues...
		 */
		init_waitqueue_head(&(dgap_Board[i]->kme_wait));

		/*
		 * allocate flip buffer for board.
		 */
		dgap_Board[i]->flipbuf = dgap_driver_kzmalloc(MYFLIPLEN, GFP_ATOMIC);
	}

	dgap_proc_register_basic_postscan();

	dgap_proc_register_fep();

	return (rc);
}



/*
 * dgap_cleanup_module()
 *
 * Module unload.  This is where it all ends.
 */
void dgap_cleanup_module(void)
{
	int i;
	ulong lock_flags;

	/* Turn off poller right away. */
	DGAP_LOCK(dgap_poll_lock, lock_flags );
	del_timer_sync( &dgap_poll_timer );
	DGAP_UNLOCK(dgap_poll_lock, lock_flags );

	dgap_proc_unregister_all();

	if (dgap_Major_Control_Registered)
		unregister_chrdev(DIGI_DGAP_MAJOR, "dgap");

	if (dgap_config_buf)
		kfree(dgap_config_buf);

	for (i = 0; i < dgap_NumBoards; ++i) {
		dgap_tty_uninit(dgap_Board[i]);
		dgap_cleanup_board(dgap_Board[i]);
	}

	dgap_tty_post_uninit();

#if defined(DGAP_TRACER)
	/* last thing, make sure we release the tracebuffer */
	dgap_tracer_free();
#endif
}


/*
 * dgap_cleanup_board()
 *
 * Free all the memory associated with a board
 */
static void dgap_cleanup_board(struct board_t *brd)
{
	int i = 0;

        if(!brd || brd->magic != DGAP_BOARD_MAGIC)
                return;

	if (brd->intr_used && brd->irq)
		free_irq(brd->irq, brd);

	tasklet_kill(&brd->helper_tasklet);

	if (brd->re_map_port) {
		DGAP_IOUNMAP(brd->re_map_port);
		brd->re_map_port = 0;
	}

	if (brd->re_map_membase) {
		DGAP_IOUNMAP(brd->re_map_membase);
		brd->re_map_membase = 0;
	}

        if (brd->msgbuf_head) {
                unsigned long flags;

                DGAP_LOCK(dgap_global_lock, flags);
                brd->msgbuf = NULL;
                printk(brd->msgbuf_head);
                DGAP_VFREE(brd->msgbuf_head);
                brd->msgbuf_head = NULL;
                DGAP_UNLOCK(dgap_global_lock, flags);
        }

	/* Free all allocated channels structs */
	for (i = 0; i < MAXPORTS ; i++) {
		if (brd->channels[i]) {
			kfree(brd->channels[i]);
			brd->channels[i] = NULL;
		}
	}

	if (brd->flipbuf)
		kfree(brd->flipbuf);

	dgap_Board[brd->boardnum] = NULL;

        DGAP_VFREE(brd);
}


/*
 * dgap_init_globals()
 *
 * This is where we initialize the globals from the static insmod
 * configuration variables.  These are declared near the head of
 * this file.
 */
static void dgap_init_globals(void)
{
	int i = 0;

	dgap_rawreadok		= rawreadok;
        dgap_trcbuf_size	= trcbuf_size;
	dgap_debug		= debug;

	for (i = 0; i < MAXBOARDS; i++) {
		dgap_Board[i] = NULL;
	}

	init_timer( &dgap_poll_timer ); 

	init_waitqueue_head(&dgap_dl_wait);
	dgap_dl_action = 0;
}


/*
 *      dgap_ioctl_name() : Returns a text version of each ioctl value.
 */
char *dgap_ioctl_name(int cmd)
{
	switch(cmd) {

	case TCGETA:		return("TCGETA");
	case TCGETS:		return("TCGETS");
	case TCSETA:		return("TCSETA");
	case TCSETS:		return("TCSETS");
	case TCSETAW:		return("TCSETAW");
	case TCSETSW:		return("TCSETSW");
	case TCSETAF:		return("TCSETAF");
	case TCSETSF:		return("TCSETSF");
	case TCSBRK:		return("TCSBRK");
	case TCXONC:		return("TCXONC");
	case TCFLSH:		return("TCFLSH");
	case TIOCGSID:		return("TIOCGSID");

	case TIOCGETD:		return("TIOCGETD");
	case TIOCSETD:		return("TIOCSETD");
	case TIOCGWINSZ:	return("TIOCGWINSZ");
	case TIOCSWINSZ:	return("TIOCSWINSZ");

	case TIOCMGET:		return("TIOCMGET");
	case TIOCMSET:		return("TIOCMSET");
	case TIOCMBIS:		return("TIOCMBIS");
	case TIOCMBIC:		return("TIOCMBIC");

	/* from digi.h */
	case DIGI_SETA:		return("DIGI_SETA");
	case DIGI_SETAW:	return("DIGI_SETAW");
	case DIGI_SETAF:	return("DIGI_SETAF");
	case DIGI_SETFLOW:	return("DIGI_SETFLOW");
	case DIGI_SETAFLOW:	return("DIGI_SETAFLOW");
	case DIGI_GETFLOW:	return("DIGI_GETFLOW");
	case DIGI_GETAFLOW:	return("DIGI_GETAFLOW");
	case DIGI_GETA:		return("DIGI_GETA");
	case DIGI_GEDELAY:	return("DIGI_GEDELAY");
	case DIGI_SEDELAY:	return("DIGI_SEDELAY");
#if 0
	case DIGI_GETCUSTOMBAUD: return("DIGI_GETCUSTOMBAUD");
	case DIGI_SETCUSTOMBAUD: return("DIGI_SETCUSTOMBAUD");
#endif
	case TIOCMODG:		return("TIOCMODG");
	case TIOCMODS:		return("TIOCMODS");
	case TIOCSDTR:		return("TIOCSDTR");
	case TIOCCDTR:		return("TIOCCDTR");

	default:		return("unknown");
	}
}



void dgap_do_config_load(uchar *uaddr, int len)
{
	int orig_len = len;
	char *to_addr;
	char *from_addr = uaddr;
	static char buf[U2BSIZE];
	int n;

	to_addr = dgap_config_buf = dgap_driver_kzmalloc(len + 1, GFP_ATOMIC);
	if (!dgap_config_buf) {
		DPR_INIT(("dgap_do_config_load - unable to allocate memory for file\n"));
		dgap_driver_state = DRIVER_NEED_CONFIG_LOAD;
		return;
	}

	n = U2BSIZE;
	while (len) {

		if (n > len)
			n = len;

		if (copy_from_user((char *) &buf, from_addr, n) == -1 ) {
			/* TODO, SOMETHING SANE */
			return;
		}

		/* Copy data from buffer to kernel memory */
		memcpy(to_addr, buf, n);

		/* increment counts */
		len -= n;
		to_addr += n;
		from_addr += n;
		n = U2BSIZE;
        }

	dgap_config_buf[orig_len] = '\0';

	to_addr = dgap_config_buf;
	dgap_parsefile(&to_addr, TRUE);

	DPR_INIT(("dgap_config_load() finish\n"));

	return;
}


/*
 * Remap PCI memory.
 */
static void dgap_do_remap(struct board_t *brd)
{

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	brd->re_map_port = DGAP_IOREMAP((brd->membase + PCI_IO_OFFSET), 0x200000);
	brd->re_map_membase = DGAP_IOREMAP(brd->membase, 0x200000);

	DPR_INIT(("remapped io: 0x%p  remapped mem: 0x%p\n",
		brd->re_map_port, brd->re_map_membase));
}


/*=======================================================================
 *
 *      usertoboard - copy from user space to board space.
 *
 *=======================================================================*/

int dgap_usertoboard(struct board_t *brd, char *to_addr, char *from_addr, int len)
{
	static char buf[U2BSIZE];
	int n = U2BSIZE;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return(-EFAULT);

	while (len) {
		if (n > len)
			n = len;

		if (copy_from_user((char *) &buf, from_addr, n) == -1 ) {
			return(-EFAULT);
		}

		/* Copy data from buffer to card memory */
		memcpy_toio(to_addr, buf, n);

		/* increment counts */
		len -= n;
		to_addr += n;
		from_addr += n;   
		n = U2BSIZE;
        }
	return(0);
}


void dgap_do_bios_load(struct board_t *brd, uchar *ubios, int len)
{
	u32 bios;
	uchar *addr = brd->re_map_membase;
	int i;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	DPR_INIT(("dgap_do_bios_load() start\n"));

	/*
	 * clear POST area
	 */
	for (i = 0; i < 16; i++)
		writeb(0, addr + POSTAREA + i);
                                
	/*
	 * Download bios
	 */
	bios = 0x1000;
	if (dgap_usertoboard(brd, addr + bios, ubios, len) == -1 ) {
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
		return;
	}

	writel(0x0bf00401, addr);
	writel(0, (addr + 4));

	/* Clear the reset, and change states. */
	writeb(FEPCLR, brd->re_map_port);
	brd->state = WAIT_BIOS_LOAD;
}


static void dgap_do_wait_for_bios(struct board_t *brd)
{
	uchar *addr;
	u16 word;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	addr = brd->re_map_membase;
	word = readw(addr + POSTAREA);

	/* Check to see if BIOS thinks board is good. (GD). */
	if (word == *(u16 *) "GD") {
		DPR_INIT(("GOT GD in memory, moving states.\n"));
		brd->state = FINISHED_BIOS_LOAD;
		return;
	}

	/* Give up on board after too long of time taken */
	if (brd->wait_for_bios++ > 5000) {
		u16 err1 = readw(addr + SEQUENCE);
		u16 err2 = readw(addr + ERROR);
		APR(("***WARNING*** %s failed diagnostics.  Error #(%x,%x).\n",
			brd->name, err1, err2));
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
	}
}

void dgap_do_fep_load(struct board_t *brd, uchar *ufep, int len)
{
	u32 fepos;
	uchar *addr;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	addr = brd->re_map_membase;

	DPR_INIT(("dgap_do_fep_load() for board %s : start\n", brd->name));

	/*
	 * Download FEP
	 */

	/* TODO: don't hard code this stuff */
	fepos = 0x1000;

	if (dgap_usertoboard(brd, addr + fepos, ufep, len) == -1 ) {
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
		return;
	}

	/*
	 * If board is a concentrator product, we need to give
	 * it its config string describing how the concentrators look.
	 */
	if ((brd->type == PCX) || (brd->type == PEPC)) {
		uchar string[100];
		uchar *config, *xconfig;
		int i = 0;

		xconfig = dgap_create_config_string(brd, string);

		/* Write string to board memory */
		config = addr + CONFIG;
		for (; i < CONFIGSIZE; i++, config++, xconfig++) {
			writeb(*xconfig, config);
			if ((*xconfig & 0xff) == 0xff)
				break;
		}
	}

	writel(0xbfc01004, (addr + 0xc34));
	writel(0x3, (addr + 0xc30));

	/* change states. */
	brd->state = WAIT_FEP_LOAD;

	DPR_INIT(("dgap_do_fep_load() for board %s : finish\n", brd->name));

}


static void dgap_do_wait_for_fep(struct board_t *brd)
{
	uchar *addr;
	u16 word;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC) {
		return;
	}

	addr = brd->re_map_membase;

	if (!addr) {
		APR(("dgap_do_wait_for_fep() addr is NULL\n"));
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
		return;
	}

	DPR_INIT(("dgap_do_wait_for_fep() for board %s : start. addr: %p\n", brd->name, addr));

	word = readw(addr + FEPSTAT);

	/* Check to see if FEP is up and running now. */
	if (word == *(u16 *) "OS") {
		DPR_INIT(("GOT OS in memory for board %s, moving states.\n", brd->name));
		brd->state = FINISHED_FEP_LOAD;
		return;
	}

	/* Give up on board after too long of time taken */
	if (brd->wait_for_fep++ > 5000) {
		u16 err1 = readw(addr + SEQUENCE);
		u16 err2 = readw(addr + ERROR);
		APR(("***WARNING*** FEPOS for %s not functioning.  Error #(%x,%x).\n",
			brd->name, err1, err2));
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
	}

	DPR_INIT(("dgap_do_wait_for_fep() for board %s : finish\n", brd->name));
}


static void dgap_do_reset_board(struct board_t *brd)
{
	uchar check;
	u32 check1;
	u32 check2;
	int i = 0;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	DPR_INIT(("dgap_do_reset_board() start\n"));

	/* FEPRST does not vary among supported boards */
	writeb(FEPRST, brd->re_map_port);

	for (i = 0; i <= 1000; i++) {
		check = readb(brd->re_map_port) & 0xe;
		if (check == FEPRST)
			break;
		udelay(10);

	}
	if (i > 1000) {
		APR(("*** WARNING *** Board not resetting...  Failing board.\n"));
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
		goto failed;
	}

	/*
	 * Make sure there really is memory out there.
	 */
	writel(0xa55a3cc3, (brd->re_map_membase + LOWMEM));
	writel(0x5aa5c33c, (brd->re_map_membase + HIGHMEM));
	check1 = readl(brd->re_map_membase + LOWMEM);
	check2 = readl(brd->re_map_membase + HIGHMEM);

	if ((check1 != 0xa55a3cc3) || (check2 != 0x5aa5c33c)) {
		APR(("*** Warning *** No memory at %p for board.\n", brd->re_map_membase));
		brd->state = BOARD_FAILED;
		brd->dpastatus = BD_NOFEP;
		goto failed;
	}

	if (brd->state != BOARD_FAILED)
		brd->state = FINISHED_RESET;

failed:
	DPR_INIT(("dgap_do_reset_board() finish\n"));
}


void dgap_do_conc_load(struct board_t *brd, uchar *uaddr, int len)
{
	char *vaddr;
	u16 offset = 0;
	struct downld_t *to_dp;

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
		return;

	vaddr = brd->re_map_membase;

	if (!vaddr)
		return;

	offset = readw((u16 *) (vaddr + DOWNREQ));
	to_dp = (struct downld_t *) (vaddr + (int) offset);

	memcpy_toio((char *) to_dp, uaddr, sizeof(struct downld_t));

	/* Tell card we have data for it */
	writew(0, vaddr + (DOWNREQ));

	brd->conc_dl_status = NO_PENDING_CONCENTRATOR_REQUESTS;

}


/*
 * Our board poller function.
 */
static void poll_tasklet(unsigned long data)
{
        struct board_t *bd = (struct board_t *) data;
	ulong  lock_flags;
	ulong  lock_flags2;
	u32 head, tail;
	char *vaddr;
	u16 *chk_addr;
	u16 check = 0;

	if (!bd || bd->magic != DGAP_BOARD_MAGIC) {
		APR(("poll_tasklet() - NULL or bad bd.\n"));
		return;
	}

	if (bd->inhibit_poller)
		return;

	DGAP_LOCK(bd->bd_lock, lock_flags);

	vaddr = bd->re_map_membase;

	/*
	 * If board is ready, parse deeper to see if there is anything to do.
	 */
	if (bd->state == BOARD_READY) {

		struct ev_t *eaddr = NULL;

		if (!bd->re_map_membase) {
			DGAP_UNLOCK(bd->bd_lock, lock_flags);
			return;
		}
		if (!bd->re_map_port) {
			DGAP_UNLOCK(bd->bd_lock, lock_flags);
			return;
		}

		if (!bd->nasync) {
			goto out;
		}

		/*
		 * If this is a CX or EPCX, we need to see if the firmware
		 * is requesting a concentrator image from us.
		 */
		if ((bd->type == PCX) || (bd->type == PEPC)) {
			chk_addr = (u16 *) (vaddr + DOWNREQ);
			check = readw(chk_addr);
			/* Nonzero if FEP is requesting concentrator image. */
			if (check) {
				if (bd->conc_dl_status == NO_PENDING_CONCENTRATOR_REQUESTS)
					bd->conc_dl_status = NEED_CONCENTRATOR;
				/*
				 * Signal downloader, its got some work to do.
				 */
				DGAP_LOCK(dgap_dl_lock, lock_flags2);
				if (dgap_dl_action != 1) {
					dgap_dl_action = 1;
					wake_up_interruptible(&dgap_dl_wait);
				}
				DGAP_UNLOCK(dgap_dl_lock, lock_flags2);

			}
		}

		eaddr = (struct ev_t *) (vaddr + EVBUF);

		/* Get our head and tail */
		head = readw(&(eaddr->ev_head));
		tail = readw(&(eaddr->ev_tail));

		/*
		 * If there is an event pending. Go service it.
		 */
		if (head != tail) {
			DGAP_UNLOCK(bd->bd_lock, lock_flags);
			dgap_event(bd);
			DGAP_LOCK(bd->bd_lock, lock_flags);
		}

out:
		/*
		 * If board is doing interrupts, ACK the interrupt.
		 */
		if (bd && bd->intr_running) {
			readb(bd->re_map_port + 2);
		}

		DGAP_UNLOCK(bd->bd_lock, lock_flags);
		return;
	}

	/* Our state machine to get the board up and running */

	/* Reset board */
	if (bd->state == NEED_RESET) {
		dgap_do_reset_board(bd);
	}

	/* Move to next state */
	if (bd->state == FINISHED_RESET) {
		bd->state = NEED_CONFIG;
	}

	if (bd->state == NEED_CONFIG) {
		/*
		 * Match this board to a config the user created for us.
		 */
		bd->bd_config = dgap_find_config(bd->type);

		/*
		 * Register the ttys (if any) into the kernel.
		 */
		if (bd->bd_config) {
			bd->state = FINISHED_CONFIG;
		}
		else {
			bd->state = CONFIG_NOT_FOUND;
		}
	}

	/* Move to next state */
	if (bd->state == FINISHED_CONFIG) {
		bd->state = NEED_DEVICE_CREATION;
	}

	/* Move to next state */
	if (bd->state == NEED_DEVICE_CREATION) {
		/*
		 * Signal downloader, its got some work to do.
		 */
		DGAP_LOCK(dgap_dl_lock, lock_flags2);
		if (dgap_dl_action != 1) {
			dgap_dl_action = 1;
			wake_up_interruptible(&dgap_dl_wait);
		}
		DGAP_UNLOCK(dgap_dl_lock, lock_flags2);
	}

	/* Move to next state */
	if (bd->state == FINISHED_DEVICE_CREATION) {
		bd->state = NEED_BIOS_LOAD;
	}

	/* Move to next state */
	if (bd->state == NEED_BIOS_LOAD) {
		/*
		 * Signal downloader, its got some work to do.
		 */
		DGAP_LOCK(dgap_dl_lock, lock_flags2);
		if (dgap_dl_action != 1) {
			dgap_dl_action = 1;
			wake_up_interruptible(&dgap_dl_wait);
		}
		DGAP_UNLOCK(dgap_dl_lock, lock_flags2);
	}

	/* Wait for BIOS to test board... */
	if (bd->state == WAIT_BIOS_LOAD) {
		dgap_do_wait_for_bios(bd);
	}

	/* Move to next state */
	if (bd->state == FINISHED_BIOS_LOAD) {
		bd->state = NEED_FEP_LOAD;

		/*
		 * Signal downloader, its got some work to do.
		 */
		DGAP_LOCK(dgap_dl_lock, lock_flags2);
		if (dgap_dl_action != 1) {
			dgap_dl_action = 1;
			wake_up_interruptible(&dgap_dl_wait);
		}
		DGAP_UNLOCK(dgap_dl_lock, lock_flags2);
	}

	/* Wait for FEP to load on board... */
	if (bd->state == WAIT_FEP_LOAD) {
		dgap_do_wait_for_fep(bd);
	}

	/* Move to next state */
	if (bd->state == FINISHED_FEP_LOAD) {

		/*
		 * Do tty device initialization.
		 */
		int rc = dgap_tty_init(bd);
		if (rc < 0) {
			dgap_tty_uninit(bd);
			APR(("Can't init tty devices (%d)\n", rc));
			bd->state = BOARD_FAILED;
			bd->dpastatus = BD_NOFEP;
		}
		else {
			bd->state = BOARD_READY;
			bd->dpastatus = BD_RUNNING;

			/*
			 * If user requested the board to run in interrupt mode,
			 * go and set it up on the board.
			 */
			if (bd->intr_used) {
				writew(1, (bd->re_map_membase + ENABLE_INTR));
				/*
				 * Tell the board to poll the UARTS as fast as possible.
				 */
				writew(FEPPOLL_MIN, (bd->re_map_membase + FEPPOLL));
				bd->intr_running = 1;
			}

			/* Wake up anyone waiting for board state to change to ready */
			wake_up_interruptible(&bd->state_wait);
		}
	}

	DGAP_UNLOCK(bd->bd_lock, lock_flags);
}


/*****************************************************************************
*
* Function:
*                                       
*    dgap_poll_handler
*
* Author:
*
*    Scott H Kilau
*       
* Parameters:
*
*    dummy -- ignored                    
*
* Return Values:
*
*    none
*
* Description:   
*                                       
*    As each timer expires, it determines (a) whether the "transmit"
*    waiter needs to be woken up, and (b) whether the poller needs to
*    be rescheduled.
*
******************************************************************************/

static void dgap_poll_handler(ulong dummy)
{
	int i;
        struct board_t *brd;
        unsigned long lock_flags;
        unsigned long lock_flags2;

	dgap_poll_counter++;


	/*
	 * If driver needs the config file still,
	 * keep trying to wake up the downloader to
	 * send us the file.
	 */
        if (dgap_driver_state == DRIVER_NEED_CONFIG_LOAD) {
		/*
		 * Signal downloader, its got some work to do.
		 */
		DGAP_LOCK(dgap_dl_lock, lock_flags2);
		if (dgap_dl_action != 1) {
			dgap_dl_action = 1;
			wake_up_interruptible(&dgap_dl_wait);
		}
		DGAP_UNLOCK(dgap_dl_lock, lock_flags2);
		goto schedule_poller;
        }
	/*
	 * Do not start the board state machine until
	 * driver tells us its up and running, and has
	 * everything it needs.
	 */
	else if (dgap_driver_state != DRIVER_READY) {
		goto schedule_poller;
	}

	/*
	 * If we have just 1 board, or the system is not SMP,
	 * then use the typical old style poller.
	 * Otherwise, use our new tasklet based poller, which should
	 * speed things up for multiple boards.
	 */
	if ( (dgap_NumBoards == 1) || (DGAP_NUM_CPUS <= 1) ) {
		for (i = 0; i < dgap_NumBoards; i++) {

			brd = dgap_Board[i];

			if (brd->state == BOARD_FAILED) {
				continue;
			}
			if (!brd->intr_running) {
				/* Call the real board poller directly */
				poll_tasklet((unsigned long) brd);
			}
		}
	}
	else {
		/* Go thru each board, kicking off a tasklet for each if needed */
		for (i = 0; i < dgap_NumBoards; i++) {
			brd = dgap_Board[i];

			/*
			 * Attempt to grab the board lock.
			 *
			 * If we can't get it, no big deal, the next poll will get it.
			 * Basically, I just really don't want to spin in here, because I want
			 * to kick off my tasklets as fast as I can, and then get out the poller.
			 */
			if (!spin_trylock(&brd->bd_lock)) {
				continue;
			}

			/* If board is in a failed state, don't bother scheduling a tasklet */
			if (brd->state == BOARD_FAILED) {
				spin_unlock(&brd->bd_lock);
				continue;
			}

			/* Schedule a poll helper task */
			if (!brd->intr_running) {
				tasklet_schedule(&brd->helper_tasklet);
			}

			/*
			 * Can't do DGAP_UNLOCK here, as we don't have
			 * lock_flags because we did a trylock above.
			 */
			spin_unlock(&brd->bd_lock);
		}
	}

schedule_poller:

	/*
	 * Schedule ourself back at the nominal wakeup interval.
	 */
	if (dgap_NumBoards > 0) {
		ulong time;

		DGAP_LOCK(dgap_poll_lock, lock_flags );
		dgap_poll_time +=  dgap_jiffies_from_ms(dgap_poll_tick);

		time = dgap_poll_time - jiffies;

		if ((ulong) time >= 2 * dgap_poll_tick) {
			dgap_poll_time = jiffies +  dgap_jiffies_from_ms(dgap_poll_tick);
		}

		dgap_poll_timer.expires = dgap_poll_time;
		DGAP_UNLOCK(dgap_poll_lock, lock_flags );

		add_timer(&dgap_poll_timer);
	}

}


/*
 * dgap_scan()
 *
 * Scan for all boards
 *
 * Because of the way the PCI device list is scanned, we will always detect
 * boards in the order that they are listed in the Ids table defined near the
 * top of this file.
 *
 */
static int dgap_scan(void)
{
	int id;
	int rc = 0;
        int i;

	for (id = 0; id < NIDS; id++) {
		struct pci_dev *pdev;

		pdev = NULL;
		while ((pdev = pci_find_device(Ids[id].vendor,
					       Ids[id].device, pdev))) {

			rc = dgap_found_board(pdev, id);
			if (rc == 0) {
                                dgap_NumBoards++;
                        }
		}
	}

        for (i = 0; i < dgap_NumBoards ; i++) {
                unsigned long flags;

                if(!dgap_Board[i])
                        continue;

                DPR_INIT(("dgap_scan(%d) - printing out the msgbuf\n", i));
                DGAP_LOCK(dgap_global_lock, flags);
                dgap_Board[i]->msgbuf = NULL;
                printk(dgap_Board[i]->msgbuf_head);
                DGAP_VFREE(dgap_Board[i]->msgbuf_head);
                dgap_Board[i]->msgbuf_head = NULL;
                DGAP_UNLOCK(dgap_global_lock, flags);
                if (rc) {
                        dgap_cleanup_board(dgap_Board[i]);
                        APR(("dgap_finalize_board_init failed - %d\n", rc));
                }
        }

        DPR_INIT(("dgap_scan() - done\n"));
	return(0);
}


/*
 * dgap_found_board()
 *
 * A board has been found, init it.
 */
static int dgap_found_board(struct pci_dev *pdev, int id)
{
        struct board_t *brd;
	DGAP_PCI_IRQ_TYPE pci_irq;
	int i = 0;

        /* get the board structure and prep it */
        brd = dgap_Board[dgap_NumBoards] =
        (struct board_t *) DGAP_VMALLOC(sizeof(struct board_t));
        if (!brd) {
                return(-ENOMEM);
        }
        memset(brd, 0, sizeof(struct board_t));

        /* make a temporary message buffer for the boot messages */
        brd->msgbuf = brd->msgbuf_head =
                (char *) DGAP_VMALLOC(sizeof(char) * 8192);
        if(!brd->msgbuf) {
                DGAP_VFREE(brd);
                return(-ENOMEM);
        }
        memset(brd->msgbuf_head, 0, sizeof(sizeof(char) * 8192));
        
	/* store the info for the board we've found */
	brd->magic = DGAP_BOARD_MAGIC;
	brd->boardnum = dgap_NumBoards;
	brd->firstminor = 0; 
	brd->vendor = Ids[id].vendor;
	brd->device = Ids[id].device;
	brd->name = Ids[id].name;
	brd->maxports = Ids[id].maxports;
	brd->type = Ids[id].config_type;
	brd->dpatype = Ids[id].dpatype;
	brd->dpastatus = BD_NOFEP;
	init_waitqueue_head(&brd->state_wait);

	DGAP_SPINLOCK_INIT(brd->bd_lock);

	brd->state		= BOARD_FOUND;
	brd->runwait		= 0;
	brd->inhibit_poller	= FALSE;
	brd->wait_for_bios	= 0;
	brd->wait_for_fep	= 0;

	for (i = 0; i < MAXPORTS; i++) {
		brd->channels[i] = NULL;
	}

	/* store which card & revision we have */
	pci_read_config_word(pdev, PCI_SUBSYSTEM_VENDOR_ID, &brd->subvendor);
	pci_read_config_word(pdev, PCI_SUBSYSTEM_ID, &brd->subdevice);
	pci_read_config_byte(pdev, PCI_REVISION_ID, &brd->rev);

	pci_irq = pdev->irq;
	brd->irq = pci_irq;

	/* get the PCI Base Address Registers */

	/* Xr Jupiter and EPC use BAR 2 */
	if (brd->device == PCI_DEVICE_XRJ_DID || brd->device == PCI_DEVICE_EPCJ_DID) {
		brd->membase     = pci_resource_start(pdev, 2);
		brd->membase_end = pci_resource_end(pdev, 2);
	}
	/* Everyone else uses BAR 0 */
	else {
		brd->membase     = pci_resource_start(pdev, 0);
		brd->membase_end = pci_resource_end(pdev, 0);
	}

	if (brd->membase & 1)
		brd->membase &= ~3;
	else
		brd->membase &= ~15;

	/*
	 * On the PCI boards, there is no IO space allocated
	 * The I/O registers will be in the first 3 bytes of the
	 * upper 2MB of the 4MB memory space.  The board memory
	 * will be mapped into the low 2MB of the 4MB memory space
	 */
	brd->port = brd->membase + PCI_IO_OFFSET;
	brd->port_end = brd->port + PCI_IO_SIZE;


	/*
	 * Special initialization for non-PLX boards
	 */
	if (brd->device != PCI_DEVICE_XRJ_DID && brd->device != PCI_DEVICE_EPCJ_DID) {

		pci_write_config_byte(pdev, 0x40, 0);
		pci_write_config_byte(pdev, 0x46, 0);

		/* Limit burst length to 2 doubleword transactions */ 
		pci_write_config_byte(pdev, 0x42, 1);
	}

	/* init our poll helper tasklet */
	tasklet_init(&brd->helper_tasklet, poll_tasklet, (unsigned long) brd);

	 /* Log the information about the board */
	dgap_mbuf(brd, DRVSTR": board %d: %s (rev %d), irq %d\n",
		dgap_NumBoards, brd->name, brd->rev, brd->irq);

	dgap_do_remap(brd);

	brd->state = NEED_RESET;

        return(0);
}


int dgap_finalize_board_init(struct board_t *brd) {

        int rc;

        DPR_INIT(("dgap_finalize_board_init() - start\n"));

	if (!brd || brd->magic != DGAP_BOARD_MAGIC)
                return(-ENODEV);

        DPR_INIT(("dgap_finalize_board_init() - start #2\n"));

	brd->use_interrupts = dgap_config_get_useintr(brd);

	/*
	 * Set up our interrupt handler if we are set to do interrupts.
	 */
	if (brd->use_interrupts && brd->irq) {

		rc = request_irq(brd->irq, dgap_intr, SA_SHIRQ, "DGAP", brd);

		if (rc) {
			dgap_mbuf(brd, DRVSTR": Failed to hook IRQ %d. Board will work in poll mode.\n",
                                  brd->irq);
			brd->intr_used = 0;
		}
		else
			brd->intr_used = 1;
	} else {
		brd->intr_used = 0;
	}

	return(0);
}


/*
 * dgap_intr()
 *
 * Driver interrupt handler.
 */
#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,0)
static irqreturn_t dgap_intr(int irq, void *voidbrd, struct pt_regs *regs)
#else
static void dgap_intr(int irq, void *voidbrd, struct pt_regs *regs)
#endif
{
	struct board_t *brd = (struct board_t *) voidbrd;

	if (!brd) {
		APR(("Received interrupt (%d) with null board associated\n", irq));
		DGAP_IRQ_RETURN(IRQ_NONE);
	}

	/*
	 * Check to make sure its for us.
	 */
	if (brd->magic != DGAP_BOARD_MAGIC) {
		APR(("Received interrupt (%d) with a board pointer that wasn't ours!\n", irq));
		DGAP_IRQ_RETURN(IRQ_NONE);
	}

	brd->intr_count++;

	/*
	 * Schedule tasklet to run at a better time.
	 */
	tasklet_schedule(&brd->helper_tasklet);

	DGAP_IRQ_RETURN(IRQ_HANDLED);
}


/************************************************************************
 *
 * Utility functions
 *
 ************************************************************************/


/*
 * dgap_driver_kzmalloc()
 *
 * Malloc and clear memory,
 */
void *dgap_driver_kzmalloc(size_t size, int priority)
{
 	void *p = kmalloc(size, priority);
	if(p)
		memset(p, 0, size);
	return(p);
}


/*
 * dgap_mbuf()
 *
 * Used to print to the message buffer during board init.
 */
void dgap_mbuf(struct board_t *brd, const char *fmt, ...) {
	va_list		ap;
	char		buf[1024];
	int		i;
	unsigned long	flags;

	DGAP_LOCK(dgap_global_lock, flags);

	/* Format buf using fmt and arguments contained in ap. */
	va_start(ap, fmt);
	i = vsprintf(buf, fmt,  ap);
	va_end(ap);

	DPR((buf));

	if (!brd || !brd->msgbuf) {
		printk(buf);
		DGAP_UNLOCK(dgap_global_lock, flags);
		return;
	}

	memcpy(brd->msgbuf, buf, strlen(buf));
	brd->msgbuf += strlen(buf);
	*brd->msgbuf = (char) NULL;

	DGAP_UNLOCK(dgap_global_lock, flags);
}



/*
 * dgap_sleep()
 *
 * Put the driver to sleep for "ticks" clock ticks
 *
 * Returns 0 if timed out, !0 (showing signal) if interrupted by a signal.
 */
int dgap_sleep(int ticks)
{
	current->state = TASK_INTERRUPTIBLE;
	if (HZ == 1000)
		ticks *= 10;
	schedule_timeout(ticks);
	return (signal_pending(current));
}


/*
 * dgap_ms_sleep()
 *
 * Put the driver to sleep for x ms's
 *
 * Returns 0 if timed out, !0 (showing signal) if interrupted by a signal.
 */
int dgap_ms_sleep(u32 ms)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout((ms * HZ) / 1000);
	return (signal_pending(current));
}
