/***************************************************************************
 *
 * Copyright (c) 2002 Promise Technology, Inc.  All rights reserved.
 * No part of this document may be reproduced or transmitted in any form or
 * by any means, electronic or mechanical, for any purpose, without the
 * express written permission of Promise Technology, Inc.
 *
 *
 * cam.c - main functions
 *
 ***************************************************************************/
#include "cam_con.h"
#include "cam_def.h"
#include "cam_imp.h"
#include "cam_macr.h"
#include "cam_gb.h"
#include "cam_exp.h"

#ifdef	_MMIO_

void
WriteIndexUchar(U32 BaseAddress, U8 Index, U8 Data )
{
	camWriteRegByte(BaseAddress+Index, Data);
}

U8 ReadIndexUchar(U32 BaseAddress, U8 Index)
{
	return(camReadRegByte(BaseAddress+Index));
}

#else

void
WriteIndexUchar(U32 BaseAddress, U8 Index, U8 Data )
{
	camOutPortByte(BaseAddress+ INDEX_REGISTER_OFFSET, Index);
	camOutPortByte(BaseAddress+ DATA_REGISTER_OFFSET+(Index&0x03), Data);
}

U8
ReadIndexUchar(U32 BaseAddress, U8 Index )
{
	camOutPortByte(BaseAddress+ INDEX_REGISTER_OFFSET, Index);
	return( camInPortByte(BaseAddress+ DATA_REGISTER_OFFSET+(Index&0x03)));
}
#endif

void
CAMInitSEQIDPool(U8 card_no)
{
    U8 i;

    if (card_no > (MAX_ADAPTER_NUM -1))
        return;

    for (i = START_SEQ_ID; i < MAX_SEQ_ID; i++)
        SEQIDPool[card_no][i] = i;
        
    SEQIDCurrent[card_no] = START_SEQ_ID;

    return;
}


U8 CAMAllocSEQID(U8 card_no)
{
	U8 SEQID;

    if (card_no > (MAX_ADAPTER_NUM -1))
       	return U8NULL;

    if (SEQIDCurrent[card_no] > MAX_SEQ_ID)
       	return U8NULL;

    SEQID = SEQIDPool[card_no][SEQIDCurrent[card_no]];
    SEQIDCurrent[card_no]++;

    return SEQID;
}

void CAMFreeSEQID(U8 card_no, U8 SEQID)
{

    if (card_no > (MAX_ADAPTER_NUM -1))
       	return;

    SEQIDCurrent[card_no]--;
    SEQIDPool[card_no][SEQIDCurrent[card_no]] = SEQID;

    return;
}

void
CAMInitPacketQ(U8 card_no, PAC_Q_TYPE q_type)
{
    U16 	i;
	PU8		base_addr;
	U32		basephyaddr;
	PADAPTER_CONFIG pCurAdapter = &(gAdapterConfig[card_no]);
	
    if (card_no > (MAX_ADAPTER_NUM -1))
        return;
        
   	base_addr = (PU8)pCurAdapter->MemVirAddr;
	basephyaddr = pCurAdapter->MemPhyAddress;
	
    switch (q_type) {
    case ATA_PAC:
    	base_addr += ATA_BASE_ADDR;
        basephyaddr = basephyaddr+ATA_BASE_ADDR+PACKET_HEAD_SIZE;
        for (i = 0; i < MAX_ATA_PAC; i++) {
            ATAPacketQueue[card_no][i] = (PCAM_ATA_PACKET)base_addr;
            (ATAPacketQueue[card_no][i])->PacketHead.Address = basephyaddr;
            ((PCAM_ATA_PACKET)base_addr)++;
            basephyaddr += ATA_PAC_SIZE;
        }
        ATAPacketCurrent[card_no] = 0;
        break;
    case XOR_PAC:
    	base_addr += XOR_BASE_ADDR;
		basephyaddr = basephyaddr+XOR_BASE_ADDR+PACKET_HEAD_SIZE;
		for (i = 0; i < MAX_XOR_PAC; i++) {
            XORPacketQueue[card_no][i] = (PCAM_XOR_PACKET)base_addr;
            (XORPacketQueue[card_no][i])->PacketHead.Address = basephyaddr;
            ((PCAM_XOR_PACKET)base_addr)++;
            basephyaddr += XOR_PAC_SIZE;
		}
        XORPacketCurrent[card_no] = 0;
        break;
    default:
        return;
    }
    return;
}


PVOID
CAMAllocPacket(U8 card_no, PAC_Q_TYPE q_type)
{
    	PVOID	PacketAddress;

    	if (card_no > (MAX_ADAPTER_NUM -1))
        	return ((PVOID)U32NULL);

    	switch (q_type) {
    		case ATA_PAC:
        		if (ATAPacketCurrent[card_no] >= MAX_ATA_PAC)
            			return ((PVOID)U32NULL);

        		PacketAddress = (PVOID)ATAPacketQueue[card_no][ATAPacketCurrent[card_no]];
        		ATAPacketCurrent[card_no]++;
        		break;

    		case XOR_PAC:
        		if (XORPacketCurrent[card_no] >= MAX_XOR_PAC)
            			return ((PVOID)U32NULL);

        		PacketAddress = (PVOID)XORPacketQueue[card_no][XORPacketCurrent[card_no]];
        		XORPacketCurrent[card_no]++;
        		break;
    		default:
        		return ((PVOID)U32NULL);
    }

    return PacketAddress;
}

void
CAMFreePacket(U8 card_no, PAC_Q_TYPE q_type, PVOID PacketAddress)
{
    	if (card_no > (MAX_ADAPTER_NUM -1))
        	return;

    	switch (q_type) {
    		case ATA_PAC:
        		ATAPacketCurrent[card_no]--;
        		ATAPacketQueue[card_no][ATAPacketCurrent[card_no]] = (PCAM_ATA_PACKET)PacketAddress;
        		break;
    		case XOR_PAC:
        		XORPacketCurrent[card_no]--;
        		XORPacketQueue[card_no][XORPacketCurrent[card_no]] = (PCAM_XOR_PACKET)PacketAddress;
        		break;
    		default:
    			break;
    }
    return;
}


void CAMInitSGQ(U8 card_no, PAC_Q_TYPE q_type)
{
    U16 i;
	PU8		base_addr;
	U32		basephyaddr;
	PADAPTER_CONFIG pCurAdapter = &(gAdapterConfig[card_no]);
	
    	if (card_no > (MAX_ADAPTER_NUM -1))
       	return;
        
   	base_addr = (PU8)pCurAdapter->MemVirAddr;
	basephyaddr = pCurAdapter->MemPhyAddress;


    	switch (q_type) {
    		case ATA_PAC:
    			base_addr += ATA_SG_BASE_ADDR;
        		basephyaddr = basephyaddr+ATA_SG_BASE_ADDR;
        		for (i = 0; i < MAX_ATA_PAC; i++) {
            			PCAM_ATA_SG_Queue[card_no][i] = (PCAM_ATA_SG)base_addr;
            			(PCAM_ATA_SG_Queue[card_no][i])->Address = basephyaddr;
            			((PCAM_ATA_SG)base_addr)++;
            			basephyaddr += ATA_SG_SIZE;
        		}
        		CAM_ATA_SG_Current[card_no] = 0;
        		break;
    		case XOR_PAC:
    			base_addr += XOR_SG_BASE_ADDR;
        		basephyaddr = basephyaddr+XOR_SG_BASE_ADDR;
        		for (i = 0; i < MAX_XOR_PAC; i++) {
            			PCAM_XOR_SG_Queue[card_no][i] = (PCAM_XOR_SG)base_addr;
            			(PCAM_XOR_SG_Queue[card_no][i])->Address = basephyaddr;
            			((PCAM_XOR_SG)base_addr)++;
            			basephyaddr += XOR_SG_SIZE;
        		}
        		CAM_XOR_SG_Current[card_no] = 0;
        		break;
    		default:
        		return;
    	}
    	return;
}


PVOID CAMAllocSG(U8 card_no, PAC_Q_TYPE q_type)
{
    	PVOID	PacketAddress;

    	if (card_no > (MAX_ADAPTER_NUM -1))
        	return ((PVOID)U32NULL);

    	switch (q_type) {
    		case ATA_PAC:
        		if (CAM_ATA_SG_Current[card_no] >= MAX_ATA_PAC)
            			return ((PVOID)U32NULL);

        		PacketAddress = (PVOID) PCAM_ATA_SG_Queue[card_no][CAM_ATA_SG_Current[card_no]];
        		CAM_ATA_SG_Current[card_no]++;
        		break;

    		case XOR_PAC:
        		if (CAM_XOR_SG_Current[card_no] >= MAX_XOR_PAC)
            			return ((PVOID)U32NULL);

        		PacketAddress = (PVOID) PCAM_XOR_SG_Queue[card_no][CAM_XOR_SG_Current[card_no]];
        		CAM_XOR_SG_Current[card_no]++;
        		break;
    		default:
        		return ((PVOID)U32NULL);
    	}
	
    	return PacketAddress;
}


void CAMFreeSG(U8 card_no, PAC_Q_TYPE q_type, PVOID PacketAddress)
{
	
    	if (card_no > (MAX_ADAPTER_NUM -1))
        	return;

    	switch (q_type) {
    		case ATA_PAC:
        		CAM_ATA_SG_Current[card_no]--;
        		PCAM_ATA_SG_Queue[card_no][CAM_ATA_SG_Current[card_no]] = (PCAM_ATA_SG)PacketAddress;
        		break;
    		case XOR_PAC:
        		CAM_XOR_SG_Current[card_no]--;
        		PCAM_XOR_SG_Queue[card_no][CAM_XOR_SG_Current[card_no]] = (PCAM_XOR_SG)PacketAddress;
        		break;
    		default:
    			break;
    	}
    	return;
}


/* if p_S_SG->Reserved !=0, set to zero, and if p_S_SG->CtrlFlag & 80 != 0, clear the bits. */
/* copy source p_S_SG to target p_T_SG (contiguous physical memory). */
void checkConvertSG(PCAMSG  p_S_SG,PCAMSG p_T_SG)
{
	while(1){
		p_T_SG->Reserved=0;
		
		p_T_SG->wCount=p_S_SG->wCount;
		p_T_SG->StartAddr = p_S_SG->StartAddr;

		if(p_S_SG->CtrlFlag & EOT){
			p_T_SG->CtrlFlag = EOT;
			break;		
		}
		else{
			p_T_SG->CtrlFlag = 0;		
		}
				
		p_S_SG++;
		p_T_SG++;
	}
			
	return;
}


void Put2SeqIDQueue(U8 bAdapterNumber,PVOID PacketAddress,U8 SeqID)
{

	PSEQ_INFO	pCurSEQ_INFO;
	PCAM_ATA_PACKET ultmpATAPacketAddress;
	PCAM_XOR_PACKET ultmpXORPacketAddress;
	
	pCurSEQ_INFO = &(gSEQ_INFO[bAdapterNumber][SeqID]);
	
	if(pCurSEQ_INFO->SEQID2PacketHead==(PVOID)U32NULL){
		ultmpATAPacketAddress=(PCAM_ATA_PACKET)PacketAddress;
		if (ultmpATAPacketAddress->PacketHead.ModelType == XOR_PAC) 	/* XOR */
		{
			pCurSEQ_INFO->Owner= XOR;
		}
		else /* ATA_PAC */  /* ATA */
		{
			PATA_TASK pATATask;
			pATATask = ultmpATAPacketAddress->PacketHead.PTA;
			pCurSEQ_INFO->Owner=(ModType) (pATATask->bCh%MAX_ATA_CHNL);
		}
		
		pCurSEQ_INFO->PACCNT = 1;
		pCurSEQ_INFO->SEQID2PacketHead=PacketAddress;
		pCurSEQ_INFO->SEQID2PacketTail=PacketAddress;
	}
	else{

		
		/* don't care this is PCAM_XOR_PACKET or PCAM_ATA_PACKET, just a template.*/
		ultmpATAPacketAddress=(PCAM_ATA_PACKET)pCurSEQ_INFO->SEQID2PacketTail;
		ultmpXORPacketAddress=(PCAM_XOR_PACKET)pCurSEQ_INFO->SEQID2PacketTail;
		
		pCurSEQ_INFO->SEQID2PacketTail=PacketAddress;
		ultmpATAPacketAddress->PacketHead.NPA=PacketAddress;

		if (ultmpATAPacketAddress->PacketHead.ModelType == ATA_PAC) 	/* ATA_PAC */
		{
			/* pointer to next ATA Packet */
			ultmpATAPacketAddress->ATAPacket.ulNCA = ((PCAM_ATA_PACKET)PacketAddress)->PacketHead.Address;
		}
		else		/* XOR_PAC */
		{
			/* pointer to next XOR Packet */
			ultmpXORPacketAddress->XORPacket.ulNCA = ((PCAM_XOR_PACKET)PacketAddress)->PacketHead.Address;	
		}
		pCurSEQ_INFO->PACCNT++;
	}			
}

#ifdef	DualMasterMode	/* If you have BIOS, that should be put in BIOS. */
U8 CAMDetect199(U32 ATABaseAddr)
{
	U8	i, DrvSel, bData, btmp;
	
	for ( i = 0; i < 2; i++ )
	{
		DrvSel = 0xE0 |(i << 4);
		bData=DrvSel;			/* select drive */
		WriteIndexUchar(ATABaseAddr, iNDEXIDEDeviceHead, bData);		/* to make sure it isn't in state 6 now */
		ReadIndexUchar(ATABaseAddr, iNDEXIDEStatus);
		bData = 0x80;
		WriteIndexUchar(ATABaseAddr, iNDEXIDESectorCount, bData );		/* make a maker */
	
		/* enter state */	
		bData=DrvSel;
		WriteIndexUchar(ATABaseAddr, iNDEXIDEDeviceHead, bData );
		WriteIndexUchar(ATABaseAddr, iNDEXIDEDeviceHead, bData );
		WriteIndexUchar(ATABaseAddr, iNDEXIDEDeviceHead, bData );
		ReadIndexUchar(ATABaseAddr, iNDEXIDESectorCount);
		ReadIndexUchar(ATABaseAddr, iNDEXIDEDeviceHead);
	
		bData = ReadIndexUchar(ATABaseAddr, iNDEXIDESectorCount);
		if( bData & 0x80 )
			return(FALSE);		/* no box */
	
		btmp = bData | 0x04;	/* PWR bit of SF mode ! */
		bData |= 0x03;	/* amber color  */
		WriteIndexUchar(ATABaseAddr, iNDEXIDESectorCount, bData );
		bData = 0;
		WriteIndexUchar(ATABaseAddr, iNDEXIDESectorNumber, bData );    /* for floating bus */
		bData = ReadIndexUchar(ATABaseAddr, iNDEXIDESectorCount);
		if( (bData & 0x03) != 0x03 )
			return(FALSE);		/* no box */

		bData = btmp & 0xDF; /* clear REP bit*/
		WriteIndexUchar(ATABaseAddr, iNDEXIDESectorCount, bData );	/* restore to orignal status */
			
		/* can the box be SF mode ? */
		WriteIndexUchar(ATABaseAddr, iNDEXIDECylinderHigh, 0x03 );
		WriteIndexUchar(ATABaseAddr, iNDEXIDECylinderHigh, 0x10 );
		bData = ReadIndexUchar(ATABaseAddr, iNDEXIDECylinderHigh);
		if(!(bData & 0x10))
			return(FALSE);
	}
	
	return(TRUE); /* there are two enclosure with 199 in SF mode */
}
#endif 
/*---------------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------------*/
U8 CAM_ReInit(PADAPTER_CONFIG_INFO pAdapter)
{
	
#ifndef SetPLL 
	PADAPTER_CONFIG pCurAdapter = &(gAdapterConfig[pAdapter->Adapter_ID]);
	
	/* restore PLL value */
	WritePortUlong(HostBaseAddr[pAdapter->Adapter_ID], oFFSETHOSTPCICTL, pCurAdapter->PLL_Parms );
#endif

	return(CAM_Init(pAdapter));
}

U8 CAM_Init(PADAPTER_CONFIG_INFO pAdapter)
{
	U8	IsSATA;
	U8	DualOnly;
	U8	i, j;
	U8	bData;
	U8	CurAdapter_ID;
	U8	StartChnl;
	U8	EndChnl;
	U8	CurMaxATAChnl = MAX_ATA_CHNL;
	U32	lindex;
	PADAPTER_CONFIG pCurAdapter;
	PSEQ_INFO	pCurSEQ_INFO;
#ifdef	_MMIO_
	U32	base_addr;
#endif		
	U32 	AddressShift;
	U32 	tempaddress;

	if ( pAdapter->MemSize < MIN_CAM_MEMSIZE )
		return(camFAIL);

	
	/* save adapter data */
	CurAdapter_ID = pAdapter->Adapter_ID;
	StartChnl = CurAdapter_ID*MAX_ATA_CHNL;
	EndChnl = StartChnl + MAX_ATA_CHNL - 1;
	pCurAdapter = &(gAdapterConfig[CurAdapter_ID]);
	for ( lindex = 0; lindex < sizeof(ADAPTER_CONFIG); lindex++ )
	{
		*(((PU8)pCurAdapter)+lindex) = 0;
	}
	pCurAdapter->ATABaseAddr = pAdapter->BaseAddress[0];
	pCurAdapter->XORBaseAddr = pAdapter->BaseAddress[1];
	pCurAdapter->HostBaseAddr = pAdapter->BaseAddress[2];
	pCurAdapter->MemIOBaseAddr = pAdapter->BaseAddress[3];
	pCurAdapter->MemPhyAddress = pAdapter->MemPhyAddress; 
	pCurAdapter->MemVirAddr = pAdapter->MemVirAddr; 
	pCurAdapter->MemSize = pAdapter->MemSize-4; 

	/* set DWord boundary */
	tempaddress = (U32)pCurAdapter->MemVirAddr;
   	if((AddressShift=(tempaddress&0x3)) != 0){
          	pCurAdapter->MemVirAddr = (PVOID)(tempaddress + (4-AddressShift));
          	pCurAdapter->MemPhyAddress = pAdapter->MemPhyAddress+(4-AddressShift); 
   	}      
   	
	for ( lindex = 0; lindex < pCurAdapter->MemSize; lindex++ )
		*((PU8)pCurAdapter->MemVirAddr+lindex) = 0;	/* clear memory*/
	
	/* save module data */
#ifdef	_MMIO_
	base_addr = pCurAdapter->MemIOBaseAddr;
   	base_addr += ATAMEMBASE;
	for (i = StartChnl,j=0; i <= EndChnl; i++, j++ )
		ATABaseAddr[i] = base_addr + ((U32)j << 7);
	XORBaseAddr[CurAdapter_ID] = pCurAdapter->MemIOBaseAddr + XORMEMBASE;
	HostBaseAddr[CurAdapter_ID] = pCurAdapter->MemIOBaseAddr + HOSTMEMBASE;		
#else
	for (i = StartChnl, j = 0; i <= EndChnl; i++, j++ )
		ATABaseAddr[i] = pCurAdapter->ATABaseAddr + ((U32)j << 4);
	XORBaseAddr[CurAdapter_ID] = pCurAdapter->XORBaseAddr;
	HostBaseAddr[CurAdapter_ID] = pCurAdapter->HostBaseAddr;
#endif

	/* check this ASIC is 620/617 series or 320/375 series */
	IsSATA = FALSE;
	ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETMSICTL, bData );
	i = bData; /* keep it */
	WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETMSICTL, 0x02);
	ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETMSICTL, bData );
	if ( bData == 0x02 )	
		IsSATA = TRUE;
	WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETMSICTL, i);	/*restore it*/
	
	if (IsSATA == TRUE ) {
			
		/* eanble BMR_BURST and change FIFO_SHD*/
		ReadPortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL, tempaddress );
		tempaddress |= 0x12000;
		WritePortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL, tempaddress );
	
		/* clear plug/unplug flags */
		ReadPortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTSATAPLUG, tempaddress );	
		tempaddress |= 0xFF;		/* clear plug/unplug flags */
		WritePortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTSATAPLUG, tempaddress );	
	
		/* unmask plug/unplug ints */
		ReadPortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTSATAPLUG, tempaddress );	
		tempaddress &= 0xFF00FFFF;		/* clear plug/unplug flags */
		WritePortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTSATAPLUG, tempaddress );
			
		/* reduce TBG clock to 133 Mhz */	
		ReadIndexUlong( ATABaseAddr[StartChnl], iNDEXIDETBGmode, tempaddress );
		tempaddress &= ~0x30000; /* clear bit 17, 16*/
		tempaddress |= 0x10000;  /* set bit 17:16 = 0:1 */ 
		WriteIndexUlong( ATABaseAddr[StartChnl], iNDEXIDETBGmode, tempaddress );
		camStallExecution(1);		/* wait a while (10 ms) */
		
		/* adjust slew rate control register */
		ReadIndexUlong( ATABaseAddr[StartChnl], iNDEXIDESlewrate, tempaddress );
		tempaddress &= 0xFFFFF03F; /* clear bit 11 ~ 6 */
		tempaddress |= 0x00000900; /* set bit 11-9 = 100b , bit 8-6 = 100 */
		WriteIndexUlong( ATABaseAddr[StartChnl], iNDEXIDESlewrate, tempaddress );

		DualOnly = FALSE;
		ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL+1, bData );
		if (!(bData & 0x80))
		{
			DualOnly = TRUE;
			/* PDC20375 series ASIC */
	  	  #ifdef	DualMasterMode
			WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, 0x00 );	/* disable 3rd and 4th chnl dual master */			
			if ( CAMDetect199(ATABaseAddr[StartChnl+2]) == TRUE )
				WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, 0x02 );	/* enable paraller ATA chnl dual master */
		  #endif	
		
		}
	}
	else {
		/* eanble BMR_BURST */
		/* disable these only for A version ASIC*/
		ReadPortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL, tempaddress );
		tempaddress |= 0x2000;
		WritePortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL, tempaddress );

		DualOnly = FALSE;
		ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTFlashCTL+1, bData );
		if (!(bData & 0x80))
		{
			DualOnly = TRUE;
			#ifdef	DualMasterMode
			/* It is PDC20617, we need to detect if dual master can be turned on. */
			WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, 0x00 );	/* disable Pri/Sec dual master */
			/* Detect if primary channel has two swap boxes with 199 */
			if ( CAMDetect199(ATABaseAddr[StartChnl]) == TRUE ) {
				WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, 0x01 );	/* enable Pri dual master */
				/* Detect if secondary channel has two swap boxes with 199 */
				if ( CAMDetect199(ATABaseAddr[StartChnl+1])== TRUE ) 
					WritePortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, 0x03 );	/* enable Pri/Sec dual master */
			}
			#endif
		}
  	  #ifdef SetPLL 
		/* If you have BIOS, that should be put in BIOS. Don't do that in driver, it will spend 3 seconds at least. */ 	
		/* PLL initialization */
	
		{
			U32		lData;
			U32		PLLCount;
			U32		Fin;
			U32		PLL_Parms;
		
			ReadPortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, lData );
			lData |= 0x0400;	/* bit 10 */
			WritePortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, lData );	/* start the counter */
			camStallExecution((U16)300);		/* wait 3 seconds. That should be exact 3 secs, adjust the count according the resolution of your timer */
			ReadPortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPLLCNT, PLLCount ); /* get the count */
			lData &= ~0x0400;	
			WritePortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, lData );	/* stop the counter */
			Fin = PLLCount/300000;		/* 10 * Internal Clk = Tcount / (Twait * 10^5) */
			PLL_Parms = (13300/Fin); 			/* F = Tgiven / (10*Interal Clk) */
			PLL_Parms <<= 16;
			PLL_Parms |= 0x85000000;				/* OD reg | R reg | F reg */
			ReadPortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, lData );
			lData &= 0x0000FFFF;
			lData |= PLL_Parms; 
			WritePortUlong(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, lData );
		}
  	  #endif	
	}
	
#ifndef SetPLL 
	ReadPortUlong( HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, pCurAdapter->PLL_Parms );
#endif	

	/* initialize ModuleStatus,QHeads,QTails,QCounts */
	
	/* initialize the xor queue */
	XORQueueTail[CurAdapter_ID]=0;
	XORQueueHead[CurAdapter_ID]=0;
	XORModuleBusy[CurAdapter_ID]=0;
	CAMInitSEQIDPool(CurAdapter_ID);
	XORModuleNeedPush[CurAdapter_ID]=0;
	XORModuleSEQID[CurAdapter_ID]=U8NULL;
	
	/* initialize the ata queue */
	for( i = StartChnl; i <= EndChnl; i++ ){
		ATAQueueHead[i] = 0;
		ATAQueueTail[i] = 0;
		ATAModuleBusy[i] = FALSE;
		ATAModuleNeedPush[i] = FALSE;
		ATAModuleSEQID[i] = U8NULL;
	}	
	/* initialization for sequence control */
	pCurSEQ_INFO = &(gSEQ_INFO[CurAdapter_ID][0]);
	for( i = 0; i < MAX_SEQ_ID; i++ )
	{
		pCurSEQ_INFO->SEQID2PacketHead = (PVOID)U32NULL;
		pCurSEQ_INFO->SEQID2PacketTail = (PVOID)U32NULL;
		pCurSEQ_INFO++;
	}
	
	/* initialize packet Queue Count*/
	/* set default PacketQueueCount to 1 (not do packets queue). */
	for (i=0; i< MAX_ATA_CHNL; i++)
		/* PacketQueueCount[CurAdapter_ID].ATAModule[i]=MAX_ATA_PAC; */
		PacketQueueCount[CurAdapter_ID].ATAModule[i]=1;
	for (i=0;i<MAX_XOR_MODULE;i++)
		/* PacketQueueCount[CurAdapter_ID].XORModule[i]=MAX_XOR_PAC; */
		PacketQueueCount[CurAdapter_ID].XORModule[i]=1;
			
	/* initialize packet Queue */
	CAMInitPacketQ(CurAdapter_ID, ATA_PAC);
	CAMInitPacketQ(CurAdapter_ID, XOR_PAC);

	/* initialize packet sg queue */
	CAMInitSGQ(CurAdapter_ID, ATA_PAC);
	CAMInitSGQ(CurAdapter_ID, XOR_PAC);

	/* initialize ATA channel infomation */
	if ( IsSATA == TRUE ) {
		if ( DualOnly == TRUE )
		{
			/* It is PDC20375, we need to detect if dual master is turned on. */
			ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, bData );
			CurMaxATAChnl = ( bData & 0x02 ) ? 4 : 3;		
		}
	
		for (i = StartChnl; i<= EndChnl;i++ )
		{	
			if ( i >= (StartChnl+CurMaxATAChnl) ) {
				gChnlConfig[i].bCh = camCHNL_NOT_FOUND;
				continue;
			}	
		
			gChnlConfig[i].bCh = i;
		
			if ( (i >= StartChnl+2 ) && ( DualOnly == TRUE ) ) {	
				gChnlConfig[i].ChnlType = CHNL_ATA;		/* parallel ATA channel */
				gChnlConfig[i].ChnlSpeed = CHNL_U133;	
				/* check cable type */
				bData = ReadIndexUchar( ATABaseAddr[i], (iNDEXIDECICR+3));
				if ( bData & 0x01 ){
					gChnlConfig[i].CableType = CABLE_40;
					gChnlConfig[i].ChnlSpeed = CHNL_U33;
				}
				else
					gChnlConfig[i].CableType = CABLE_80;
			}
			else {
				gChnlConfig[i].ChnlType = CHNL_SATA;
				#ifdef SATADownToU5
				gChnlConfig[i].ChnlSpeed = CHNL_U100;
				#else
				gChnlConfig[i].ChnlSpeed = CHNL_U150;
				#endif
				gChnlConfig[i].CableType = CABLE_SATA;
				gBOXConfig[i].BOXID = 0xFF;	/* no box */
				gBOXConfig[i].BOXType = camNOBOX;
			}
			
			if ( gChnlConfig[i].ChnlType == CHNL_SATA )
			{
				/* disable the tranalation of IDLE/STANDBY IMME to Partial/Slumber of SATA bridge */
				/* set FIFO threshold value to 1D, for Seagate SATA HDD */
				
				/* if the regidter offset large than 0x80, when use MMIO, must use this method */
				#ifdef	_MMIO_
				ReadIndexUlong( ATABaseAddr[CurAdapter_ID*MAX_ATA_CHNL]  + 0x100 * (i%MAX_ATA_CHNL) , iNDEXIDEPowermodeControl, tempaddress );
				tempaddress &= 0xFFFFE0FF;
				tempaddress |= 0x00081D00;	/* set bit 19 = 1, bit 12-8 = 1Dh*/	
				WriteIndexUlong( ATABaseAddr[CurAdapter_ID*MAX_ATA_CHNL]  + 0x100 * (i%MAX_ATA_CHNL) , iNDEXIDEPowermodeControl, tempaddress );
			
				/* set SATA mode register */
				ReadIndexUlong( ATABaseAddr[CurAdapter_ID*MAX_ATA_CHNL]  + 0x100 * (i%MAX_ATA_CHNL) , iNDEXIDESATAmode, tempaddress );
				tempaddress &= ~0x00001800;	/* set bit [12:11] = 0:0 */
				tempaddress |= 0x00000400;	/* set bit [10:9] = 1:0 */
				WriteIndexUlong( ATABaseAddr[CurAdapter_ID*MAX_ATA_CHNL]  + 0x100 * (i%MAX_ATA_CHNL) , iNDEXIDESATAmode, tempaddress );
				camStallExecution(1);		/* wait a while (10 ms) */
				#else
				ReadIndexUlong( ATABaseAddr[i], iNDEXIDEPowermodeControl, tempaddress );
				tempaddress &= 0xFFFFE0FF;
				tempaddress |= 0x00081D00;	/* set bit 19 = 1, bit 12-8 = 1Dh*/	
				WriteIndexUlong( ATABaseAddr[i], iNDEXIDEPowermodeControl, tempaddress );
			
				/* set SATA mode register */
				ReadIndexUlong( ATABaseAddr[i], iNDEXIDESATAmode, tempaddress );
				tempaddress &= ~0x00001800;	/* set bit [12:11] = 0:0 */
				tempaddress |= 0x00000400;	/* set bit [10:9] = 1:0 */
				WriteIndexUlong( ATABaseAddr[i], iNDEXIDESATAmode, tempaddress );
				camStallExecution(1);		/* wait a while (10 ms) */
				#endif

			}
		}
	}
	else {
		if ( DualOnly == TRUE )
		{
			/* It is PDC20617, we need to detect if dual master is turned on. */
			ReadPortUchar(HostBaseAddr[CurAdapter_ID], oFFSETHOSTPCICTL, bData );
			switch (bData & 0x03) {
	 			case 0x03:
					CurMaxATAChnl = 4;
					break;
				case 0x01:
					CurMaxATAChnl = 3;
					break;
				default:
					CurMaxATAChnl = 2;
					break;
			}
		}
	
		for (i = StartChnl; i<= EndChnl;i++ )
		{	
			if ( i >= (StartChnl+CurMaxATAChnl) ) {
				gChnlConfig[i].bCh = camCHNL_NOT_FOUND;
				continue;
			}
			gChnlConfig[i].bCh = i;
			gChnlConfig[i].ChnlType = CHNL_ATA;	
			gChnlConfig[i].ChnlSpeed = CHNL_U133;				/* PDC20620/617 can support up to U133*/
		
			/* check cable type */
			bData = ReadIndexUchar( ATABaseAddr[i], (iNDEXIDECICR+3));
			if ( bData & 0x01 ){
				gChnlConfig[i].CableType = CABLE_40;
				gChnlConfig[i].ChnlSpeed = CHNL_U33;
			}
			else
				gChnlConfig[i].CableType = CABLE_80;
				
			/* set IDE output driving capability selection */
			bData = ReadIndexUchar( ATABaseAddr[i], (iNDEXIDECICR+1) );
			bData |= 0xC0;		/* set to 10mA */
			WriteIndexUchar( ATABaseAddr[i], (iNDEXIDECICR+1), bData );
		}
	}
		
		
	/* initialize device info */	
	for( i = StartChnl; i<= EndChnl; i++ )
	{	
		for ( j =0; j < 2; j++ ) 
		{ 
			gDrvConfig[i][j].bCh = gChnlConfig[i].bCh;
			gDrvConfig[i][j].bID = j;
			gDrvConfig[i][j].DevFlag = DEV_NOTFOUND;
			gDrvConfig[i][j].pEnclosure = ZERO_NULL;
		}
	}

	/* initialize TimerID */
	for( i = StartChnl; i <= EndChnl; i++ ){
		ATA_Timeout_TimerID[i]=ZERO_NULL;
		BOX_Timeout_TimerID[i]=ZERO_NULL;
	}

	XOR_Timeout_TimerID[CurAdapter_ID]=ZERO_NULL;
	
	/* initialize timeout times*/
	CAMTimeoutTimes[CurAdapter_ID].ATACommand=DEF_ATATIMEOUT;
	CAMTimeoutTimes[CurAdapter_ID].ATAPICommand=DEF_ATAPITIMEOUT;
	CAMTimeoutTimes[CurAdapter_ID].BOXCommand=DEF_CHKREPTIME;		
	CAMTimeoutTimes[CurAdapter_ID].XORCommand=DEF_XORTIMEOUT;
		
	/* enable receive task.*/
	for( i = StartChnl; i <= EndChnl; i++ )
		ATAModuleReceiveTask[i]=TRUE;
	XORModuleReceiveTask[CurAdapter_ID]=TRUE;	
	 
	return(camSUCCESS); /*successful*/
}


U8 CAM_Close(PADAPTER_CONFIG_INFO pAdapter)
{
	U8 i;
	U8 CurAdapter_ID;

	if ( pAdapter->Adapter_ID > MAX_ADAPTER_NUM )
		return(camFAIL);
	
	CurAdapter_ID=pAdapter->Adapter_ID;
	
	
	/* For window hibernate, don't disable receive task */
	/* disable receive task.*/
	/*
	for (i=CurAdapter_ID*MAX_ATA_CHNL;i<(CurAdapter_ID+1)*MAX_ATA_CHNL;i++)
		ATAModuleReceiveTask[i]=FALSE;
	XORModuleReceiveTask[CurAdapter_ID]=FALSE;
	*/
	
	/* remove all task */
	/* stop all ata task */
	for (i=CurAdapter_ID*MAX_ATA_CHNL;i<(CurAdapter_ID+1)*MAX_ATA_CHNL;i++){
		stopATAModule(i);
	}
	/* stop all xor task */
	for (i=CurAdapter_ID*MAX_XOR_MODULE;i<(CurAdapter_ID+1)*MAX_XOR_MODULE;i++){
		stopXORModule(i);
	}

	/* remove all timer */
	for (i=CurAdapter_ID*MAX_ATA_CHNL;i<(CurAdapter_ID+1)*MAX_ATA_CHNL;i++){
		if (ATA_Timeout_TimerID[i] != ZERO_NULL){
			camTimer(0,(void *)ATA_Timeout_TimerID[i]);	
			ATA_Timeout_TimerID[i] = ZERO_NULL;
		}
	}

	for (i=CurAdapter_ID*MAX_ATA_CHNL;i<(CurAdapter_ID+1)*MAX_ATA_CHNL;i++){
		if (BOX_Timeout_TimerID[i] != ZERO_NULL){
			camTimer(0,(void *)BOX_Timeout_TimerID[i]);	
			BOX_Timeout_TimerID[i] = ZERO_NULL;
		}
	}

	for (i=CurAdapter_ID*MAX_XOR_MODULE;i<(CurAdapter_ID+1)*MAX_XOR_MODULE;i++){
		if (XOR_Timeout_TimerID[i] != ZERO_NULL){
			camTimer(0,(void *)XOR_Timeout_TimerID[i]);	
			XOR_Timeout_TimerID[i] = ZERO_NULL;
		}
	}
	
	return(camSUCCESS); /*successful*/ 
}

void CAM_GetInfo(PCAM_INFO pCAMInfo)
{
	pCAMInfo->VerMajor = CAM_VerMajor;
	pCAMInfo->VerMinor = CAM_VerMinor;
	pCAMInfo->VerOEM   = CAM_VerOEM;
	pCAMInfo->VerBuild = CAM_VerBuild;
	pCAMInfo->CAMFlag  = ACCESS_TYPE;
	pCAMInfo->DeviceID = CAM_DEVICEID;
	pCAMInfo->MinMemSize = MIN_CAM_MEMSIZE;
	pCAMInfo->NumofATA = MAX_ATA_CHNL;
	pCAMInfo->NumofXOR = MAX_XOR_MODULE;
	pCAMInfo->NumofHDMA = MAX_HDMA_MODULE;
	pCAMInfo->VirMemBase = bit(3);	/*ask channel driver convert BA#3 for memory-mapped I/O */
	return;	
}

U32 CAM_GetExtraData(U8 Adapter_ID, U8 bID)
{
	U32	lData;
		
	if (!HostBaseAddr[Adapter_ID])
		return(0);
	ReadPortUlong( HostBaseAddr[Adapter_ID],(oFFSETHOSTEXTRA0+(bID << 2)) ,lData);
	
	return(lData);
}

void CAM_SetExtraData(U8 Adapter_ID, U8 bID, U32 value)
{
	if (!HostBaseAddr[Adapter_ID])
		return;
	if ( bID == 0 ) {
		U32	lData;	
		ReadPortUlong( HostBaseAddr[Adapter_ID],(oFFSETHOSTEXTRA0+(bID << 2)) ,lData);
		value &= 0x7FFFFFFF;
		value |= (lData & 0x80000000) ? 0x80000000: 0;
	}
	WritePortUlong( HostBaseAddr[Adapter_ID],(oFFSETHOSTEXTRA0+(bID << 2)) ,value);
	return;
}

void CAM_GetPacketCount(U8 bAdapterNumber,PMODULE_PACKET_COUNT pCAM_Module)
{
	U8 i;
	
	for (i=0;i<MAX_ATA_CHNL;i++)
		pCAM_Module->ATAModule[i]=PacketQueueCount[bAdapterNumber].ATAModule[i];
	for (i=0;i<MAX_XOR_MODULE;i++)
		pCAM_Module->XORModule[i]=PacketQueueCount[bAdapterNumber].XORModule[i];
		
	return;
}

void CAM_SetPacketCount(U8 bAdapterNumber,PMODULE_PACKET_COUNT pCAM_Module)
{
	U8 i;

	for (i=0;i<MAX_ATA_CHNL;i++){

		/* can't be zero or large than MAX_ATA_PAC */
		if (pCAM_Module->ATAModule[i] < 1)
			PacketQueueCount[bAdapterNumber].ATAModule[i]=1;
		else if (pCAM_Module->ATAModule[i] > MAX_ATA_PAC)
			PacketQueueCount[bAdapterNumber].ATAModule[i]= MAX_ATA_PAC;
		else	
			PacketQueueCount[bAdapterNumber].ATAModule[i]=pCAM_Module->ATAModule[i];
	}
	
	for (i=0;i<MAX_XOR_MODULE;i++){

		/* can't be zero or large than MAX_XOR_PAC */
		if (pCAM_Module->XORModule[i] < 1)
			PacketQueueCount[bAdapterNumber].XORModule[i]=1;
		else if (pCAM_Module->XORModule[i] > MAX_XOR_PAC)
			PacketQueueCount[bAdapterNumber].XORModule[i]= MAX_XOR_PAC;
		else	
			PacketQueueCount[bAdapterNumber].XORModule[i]=pCAM_Module->XORModule[i];
	}		

	return;
}

void CAM_GetTimeoutTime(U8 bAdapterNumber,PMODULE_TIMEOUT pCAM_Timeout)
{

	if ( bAdapterNumber > (MAX_ADAPTER_NUM-1) )
		return;
		
	pCAM_Timeout->ATACommand=CAMTimeoutTimes[bAdapterNumber].ATACommand;
	pCAM_Timeout->ATAPICommand=CAMTimeoutTimes[bAdapterNumber].ATAPICommand;
	pCAM_Timeout->BOXCommand=CAMTimeoutTimes[bAdapterNumber].BOXCommand;
	pCAM_Timeout->XORCommand=CAMTimeoutTimes[bAdapterNumber].XORCommand;

	return;
}

void CAM_SetTimeoutTime(U8 bAdapterNumber,PMODULE_TIMEOUT pCAM_Timeout)
{
	
	if ( bAdapterNumber > (MAX_ADAPTER_NUM-1) )
		return;
		
	if (pCAM_Timeout->ATACommand != 0)
		CAMTimeoutTimes[bAdapterNumber].ATACommand=pCAM_Timeout->ATACommand;
	if (pCAM_Timeout->ATAPICommand != 0)
		CAMTimeoutTimes[bAdapterNumber].ATAPICommand=pCAM_Timeout->ATAPICommand;
	if (pCAM_Timeout->BOXCommand != 0)
		CAMTimeoutTimes[bAdapterNumber].BOXCommand=pCAM_Timeout->BOXCommand;
	if (pCAM_Timeout->XORCommand != 0)
		CAMTimeoutTimes[bAdapterNumber].XORCommand=pCAM_Timeout->XORCommand;	

	return;
}
